/***********************************************************************************

	Copyright (C) 2007-2008 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeograph.hpp"
#include "helpers.hpp"

#include <string>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <gcrypt.h>


LIFEOENTRY::LIFEOENTRY ()
: Year(1999), Month(1), Day(1), Text(""),
  Favored(false), FilteredOut(false) {
}

LIFEOENTRY::LIFEOENTRY (unsigned int n_year,
                        unsigned int n_month,
                        unsigned int n_day,
                        const std::string& n_text)
: Year(n_year), Month(n_month), Day(n_day), Text(n_text),
  Favored(false), FilteredOut(false) {
}

const std::string LIFEOENTRY::get_date_str (void) const {
	std::ostringstream l_ss;
	l_ss << Year << ".";
	l_ss << std::setfill('0') << std::setw(2) << Month << ".";
	l_ss << std::setfill('0') << std::setw(2) << Day;
	return (l_ss.str());
}
// LIFEODATA
LIFEODATA::LIFEODATA ()
: OptionPlainDB(false) {

}

void LIFEODATA::set_path (const std::string& l_path) {
	Path = l_path;
}

const std::string& LIFEODATA::get_path(void) const {
	return Path;
}

bool LIFEODATA::is_path_set (void) const {
    return (bool) Path.size();
}

//TODO: this function should return the result of the operation
void LIFEODATA::set_passphrase (const std::string& l_passphrase) {
	// TODO: resetting the password should be carried out by a dedicated function
	if (l_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE || l_passphrase.size() == 0)
		Passphrase = l_passphrase;
}

std::string LIFEODATA::get_passphrase (void) const {
	return Passphrase;
}

bool LIFEODATA::compare_passphrase (const std::string& l_passphrase) const {
	return (Passphrase == l_passphrase);
}

bool LIFEODATA::is_passphrase_set (void) const {
	return (bool) Passphrase.size();
}

LIFEO::RESULT LIFEODATA::read_body () {
    if (OptionPlainDB) {
        return read_plain();
    }
    else
        return read_encrypted();
}

LIFEO::RESULT LIFEODATA::read_plain () {
    std::ifstream l_file(Path.c_str());

    if (!l_file.is_open()) {
		LIFEOHELPERS::ERROR( "Failed to open database file " + Path );
        //TODO(0.3.5): move string to string center
		return LIFEO::COULD_NOT_START;
	}


    l_file.seekg(0, std::ios::end);

    int l_readSize = l_file.tellg() - BodyPosition - 1;

    if (l_readSize <= 0) {
        std::cout << "empty database" << std::endl;
        l_file.close();
        return LIFEO::EMPTY_DATABASE;
    }

    l_file.seekg(BodyPosition);

    char * l_readBuffer = new char [l_readSize+1];
    // TODO: check validity
    
    l_file.read(l_readBuffer, l_readSize);

    l_file.close();

    std::stringstream l_stream(l_readBuffer);
    LIFEO::RESULT l_result = parse_db_body_text(l_stream);
    delete [] l_readBuffer;

    return l_result;
}

LIFEO::RESULT LIFEODATA::read_header () {
    std::ifstream l_file (Path.c_str());

    if (!l_file.is_open()) {
		LIFEOHELPERS::ERROR( "Failed to open database file " + Path );
        //TODO: move string to string center
		return LIFEO::COULD_NOT_START;
	}
    std::string l_line;

    getline (l_file, l_line);

    if (l_line != LIFEO::DB_FILE_HEADER) {
		//LIFEOHELPERS::ERROR( "header: " + l_line );
		LIFEOHELPERS::ERROR( "Corrupt or incompatible file: " + Path );
        //TODO: move string into the central string database
        l_file.close();
        Path="";
        return LIFEO::INCOMPATIBLE_FILE;
    }

    while (getline (l_file, l_line)) {
        switch (l_line[0]) {
            case 'V':
                //TODO: optimize a little bit for memory
                if (LIFEOHELPERS::convert_string(l_line.substr(2)) < 
                    LIFEO::DB_FILE_VERSION_INT_MIN
                    ||
                    LIFEOHELPERS::convert_string(l_line.substr(2)) >
                    LIFEO::DB_FILE_VERSION_INT) {
                    std::cout << "incompatible file version" << std::endl;
                    l_file.close();
                    return LIFEO::INCOMPATIBLE_FILE;
                }
            break;
            case 'E':
                OptionPlainDB = l_line.substr(2) == "no";
            break;
            case '(':
            break;
            case ')':
            case '{':   //TODO: corruption: handle it privately
            // end of header
            BodyPosition = l_file.tellg();
            l_file.close();
            return LIFEO::SUCCESS;
            break;
        }
    }
    
    l_file.close();
    Path="";
    return LIFEO::INCOMPATIBLE_FILE;
}

LIFEO::RESULT LIFEODATA::parse_db_body_text (std::stringstream& l_stream) {
//     std::stringstream l_stream(l_text);
	std::string l_line, l_content("");
	int l_year(0), l_month(0), l_day(0);
    bool l_favored(false);

	while (getline (l_stream, l_line)) {
        switch (l_line[0]) {
            case 'Y':	// year
                l_year = LIFEOHELPERS::convert_string(l_line.substr(2));
            break;
            case 'M':	// month
                l_month = LIFEOHELPERS::convert_string(l_line.substr(2));
            break;
            case 'D':	// day
                l_day = LIFEOHELPERS::convert_string(l_line.substr(2));
            break;
            case 'C':	// content
                // when writing a database file by hand, one can easily forget to put
                // a blank space after C, so we have to check
                if (l_line.size() < 2)
                    continue;

                if (l_content.size())	// if not the first line
                    l_content += "\n";
                l_content += l_line.substr(2);
            break;
            case '}':	// end of Entry
            {	LIFEOENTRY n_entry (l_year, l_month, l_day, l_content);
                if (l_favored)
                    n_entry.make_favored();
                Entry.push_back(n_entry);
                l_content.clear();
            }
            break;
            case '{':
                l_favored=false;
            break;
            case '[':
                l_favored=true;
            break;
            default:
                LIFEOHELPERS::ERROR l_error1("Unrecognized line:");
                LIFEOHELPERS::ERROR l_error2(l_line);
                return LIFEO::FAILURE;
            break;

        }
	}
    std::cout << Entry.size() << " entrie(s) successfully read." << std::endl;
	return LIFEO::SUCCESS;
}

bool LIFEODATA::create_db_header_text (std::stringstream& l_output, bool l_encrypted) {
    l_output << LIFEO::DB_FILE_HEADER << std::endl;
    l_output << "(" << std::endl;
    l_output << "V " << LIFEO::DB_FILE_VERSION_INT << std::endl;
    l_output << "E " << (l_encrypted ? "yes" : "no") << std::endl;
    l_output << ")" << std::endl;

    return true;
}

bool LIFEODATA::create_db_body_text (std::stringstream& l_output) {
    std::string::size_type l_ptStart=0, l_ptEnd;

	for (std::vector<LIFEOENTRY>::iterator l_itr = Entry.begin();
         l_itr != Entry.end();
         l_itr++) {
		l_output << ((*l_itr).is_favored() ? "[" : "{") << std::endl;
		l_output << "Y " << (*l_itr).get_year() << std::endl;
		l_output << "M " << (*l_itr).get_month() << std::endl;
		l_output << "D " << (*l_itr).get_day() << std::endl;

		do {
			l_ptEnd = (*l_itr).get_text().find("\n", l_ptStart);
			l_output << "C " <<
                    (*l_itr).get_text().substr(l_ptStart, l_ptEnd-l_ptStart) << "\n";
			l_ptStart = l_ptEnd+1;
		}
		while (l_ptEnd != std::string::npos);

		l_output << "}" << std::endl;

	}
    return true;
}

LIFEO::RESULT LIFEODATA::read_encrypted () {
	std::ifstream l_file (Path.c_str(), std::ios::in | std::ios::binary);

	if (!l_file.is_open()) {
		LIFEOHELPERS::ERROR( "Failed to open database file" + Path );
		return LIFEO::COULD_NOT_START;
	}
    l_file.seekg(BodyPosition);

	std::string l_line, l_content;
	//int l_year(0), l_month(0), l_day(0);
    unsigned char * l_buffer;
	//std::stringstream l_stream;
	try {
	// Allocate memory for salt
		unsigned char * l_salt = new unsigned char [LIFEOHELPERS::CIPHER::cSALT_SIZE];
		if (!l_salt)
			throw LIFEOHELPERS::ERROR( "Unable to allocate memory for salt" );
		// Read salt value
		l_file.read((char *) l_salt, LIFEOHELPERS::CIPHER::cSALT_SIZE);

		unsigned char * l_iv = new unsigned char [LIFEOHELPERS::CIPHER::cIV_SIZE];
		if (!l_iv)
			throw LIFEOHELPERS::ERROR( "Unable to allocate memory for IV" );
		// Read IV
		l_file.read((char *) l_iv, LIFEOHELPERS::CIPHER::cIV_SIZE);

		unsigned char * l_key;
		LIFEOHELPERS::CIPHER::expand_key(Passphrase.c_str(), l_salt, &l_key);

		// Calulate bytes of data in file
		size_t l_size = LIFEOHELPERS::get_file_size(l_file) - l_file.tellg();
        if (l_size <= 3) {
            l_file.close();
            return LIFEO::EMPTY_DATABASE;
        }
		l_buffer = new unsigned char [l_size+1];
		// TODO(<0.4.0): ex handling here!!!

		l_file.read((char *) l_buffer, l_size);
		LIFEOHELPERS::CIPHER::decrypt_buffer(l_buffer, l_size, l_key, l_iv);
		l_file.close();

		// passphrase check
		if (l_buffer[0] != Passphrase[0] && l_buffer[1] != '\n') {
			return LIFEO::WRONG_PASSWORD;
		}
	}
	catch (LIFEOHELPERS::ERROR l_error) {
		return LIFEO::COULD_NOT_START;
	}

    std::stringstream l_stream;
    l_buffer+=2;    // ignore first two chars which are for passphrase checking
    l_stream << l_buffer;
    l_buffer-=2;    // restore pointer to the start of the buffer before deletion 
    delete [] l_buffer;

    return parse_db_body_text(l_stream);
}

LIFEO::RESULT LIFEODATA::write (bool l_encrypt, const std::string& l_path) {
	return (l_encrypt ? write_encrypted(l_path) : write_plain(l_path));
}
LIFEO::RESULT LIFEODATA::write (const std::string& l_path) {
	return (OptionPlainDB ? write_plain(l_path) : write_encrypted(l_path));
}

LIFEO::RESULT LIFEODATA::write_plain (const std::string& l_path, bool l_headerOnly) {
	std::ofstream l_file (	l_path.size() > 0 ? l_path.c_str() : Path.c_str(),
							std::ios::out | std::ios::trunc);
	if (!l_file.is_open()) {
		std::cout << "i/o error!: " << l_path << std::endl;
		return LIFEO::COULD_NOT_START;
	}

	std::stringstream l_output;
    create_db_header_text(l_output, l_headerOnly);
    // header only mode is for encrypted headers
    if (!l_headerOnly) {
        create_db_body_text(l_output);
        std::cout << "plain database saved successfully" << std::endl;
    }

    l_file << l_output.str();
	l_file.close();

	return LIFEO::SUCCESS;
}

LIFEO::RESULT LIFEODATA::write_encrypted (const std::string& l_path) {
	// writing header:
    write_plain(l_path, true);
	std::ofstream l_file (	l_path.size() > 0 ? l_path.c_str() : Path.c_str(),
							std::ios::out | std::ios::app | std::ios::binary);
	if (!l_file.is_open()) {
		std::cout << "i/o error!: " << l_path << std::endl;
		return LIFEO::COULD_NOT_START;
	}
	std::stringstream l_output;
    // first char of passphrase for checking
	l_output << Passphrase[0] << '\n';
    create_db_body_text(l_output);

	// encryption
	try {
		size_t l_size =  l_output.str().size()+1;

		unsigned char * l_key, * l_salt;
		LIFEOHELPERS::CIPHER::create_new_key(Passphrase.c_str(), &l_salt , &l_key);

		unsigned char * l_iv;
		LIFEOHELPERS::CIPHER::create_iv(&l_iv);

		unsigned char * l_buffer = new unsigned char [l_size];
		memcpy (l_buffer, l_output.str().c_str(), l_size);

		LIFEOHELPERS::CIPHER::encrypt_buffer(l_buffer, l_size, l_key, l_iv);

		l_file.write ((char *) l_salt, LIFEOHELPERS::CIPHER::cSALT_SIZE);
		l_file.write ((char *) l_iv, LIFEOHELPERS::CIPHER::cIV_SIZE);
		l_file.write ((char *) l_buffer, l_size);

	}
	catch (LIFEOHELPERS::ERROR l_error) {
        return LIFEO::FAILURE;
	}

	l_file.close();

    std::cout << "encrypted database saved successfully" << std::endl;
	return LIFEO::SUCCESS;
}

void LIFEODATA::clear () {
    Entry.clear();
    Passphrase = "";
}

int  LIFEODATA::get_todays_first_entry (void) {
    time_t l_timeCurrent;
	tm l_tm;
	time(&l_timeCurrent); // get current time_t value
	l_tm=*(localtime(&l_timeCurrent)); // dereference and assign

    return get_entry(l_tm.tm_year+1900, l_tm.tm_mon+1, l_tm.tm_mday);
}

int LIFEODATA::add_today (void) {
	time_t l_timeCurrent;
	tm l_tm;
	time(&l_timeCurrent); // get current time_t value
	l_tm=*(localtime(&l_timeCurrent)); // dereference and assign
    
    return create_entry(l_tm.tm_year+1900, l_tm.tm_mon+1, l_tm.tm_mday);
}

int LIFEODATA::get_entry (unsigned int n_year, unsigned int n_month, unsigned int n_day) {
	int l_order = 0;
	for (std::vector<LIFEOENTRY>::iterator t1_itr = Entry.begin();
         t1_itr != Entry.end();
         t1_itr++) {
		if ((*t1_itr).get_year() == n_year)
			if ((*t1_itr).get_month() == n_month)
				if ((*t1_itr).get_day() == n_day)
					return (l_order);
		l_order++;
	}
	return (-1);
}

const LIFEOENTRY&	LIFEODATA::get_entry (int l_entry) const {
    return Entry[l_entry];
}

LIFEOENTRY&	LIFEODATA::get_entry (int l_entry) {
    return Entry[l_entry];
}

int LIFEODATA::create_entry (unsigned int n_year,
                             unsigned int n_month,
                             unsigned int n_day,
                             const std::string& n_content) {
	std::vector<LIFEOENTRY>::iterator l_itr;
	LIFEOENTRY n_entry (n_year, n_month, n_day, n_content);
	int l_order = 0;
	//~ bool l_less_exists = false;

	for (l_itr = Entry.begin(); l_itr != Entry.end(); l_itr++) {
		if ((*l_itr).get_year() == n_year) {
			if ((*l_itr).get_month() == n_month) {
				if ((*l_itr).get_day() <= n_day)
					break;
			}
			else if ((*l_itr).get_month() < n_month)
				break;
		}
		else if ((*l_itr).get_year() < n_year)
			break;

		l_order++;
	}

	Entry.insert (l_itr, n_entry);

	return (l_order);
}

bool LIFEODATA::delete_entry (int l_entry) {
	std::vector<LIFEOENTRY>::iterator l_itr=Entry.begin();
	for (int i = 0; i < l_entry; i++)
		l_itr++;
	Entry.erase(l_itr);

	return (true);		// reserved
}

int LIFEODATA::list_entries (void) {
	int t_number_of_entries(0);
	for (std::vector<LIFEOENTRY>::iterator t1_itr =
                Entry.begin(); t1_itr != Entry.end(); t1_itr++) {
		std::cout << (*t1_itr).get_year() << "/" <<
                (*t1_itr).get_month() << "/" << (*t1_itr).get_day() << std::endl;
		t_number_of_entries++;
	}
	std::cout << t_number_of_entries << " entries total." << std::endl;
	return (t_number_of_entries);
}

int LIFEODATA::set_filter (const std::string& l_string) {
    std::vector<LIFEOENTRY>::iterator l_itrEntry;

	int i = 0;
	for (l_itrEntry = Entry.begin(); l_itrEntry != Entry.end(); l_itrEntry++) {
		(*l_itrEntry).set_filtered_out((*l_itrEntry).get_text().find(l_string) ==
                std::string::npos);
	}

	return i;
}

void LIFEODATA::remove_filter () {
    std::vector<LIFEOENTRY>::iterator l_itrEntry;
    for (l_itrEntry = Entry.begin(); l_itrEntry != Entry.end(); l_itrEntry++) {
		(*l_itrEntry).set_filtered_out(false);
	}
}
//TODO...
/*
void LIFEODATA::do_for_each_entry (LIFEOFUNCVOIDVOID l_funcvoidvoid) {

}
*/

