/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_TEXTVIEW_HEADER
#define LIFEOGRAPH_TEXTVIEW_HEADER


#include <gtkmm/textview.h>
#include <gdkmm/cursor.h>

extern "C"
{
#include <gtkspell/gtkspell.h>
}

#include "undo.hpp"
#include "database.hpp"


namespace LIFEO
{

const int		UNDO_MERGE_TIMEOUT = 3;


enum LookingFor
{
	LF_NOTHING,
	LF_FIRSTNEWLINE,
	LF_NEWLINE,
	LF_SPACE,					// subtitle
//	LF_SPECIALCHAR,				// *_=#
	LF_ASTERISK			= '*',	// bold
	LF_UNDERSCORE		= '_',	// italic
	LF_EQUALS			= '=',	// strikethrough
	LF_HASH				= '#',	// highlight
	LF_SLASH1,
	LF_SLASH2,
	LF_SLASH3,
	LF_URI,
	LF_AT				= '@',	// email
	LF_URIFILE,
	LF_DIGITYEAR2		= 9001,
	LF_DIGITYEAR3		= 9002,
	LF_DIGITYEAR4		= 9003,
	LF_DOTYM			= 9004,
	LF_DIGITMONTH1		= 9005,
	LF_DIGITMONTH2		= 9006,
	LF_DOTMD			= 9007,
	LF_SLASHMD			= 9008,
	LF_DIGITDAY1		= 9009,
	LF_DIGITDAY2		= 9010,
	LF_ORDINARYCHAR
};

enum LinkStatus
{
	LS_OK,
	LS_ENTRY_UNAVAILABLE,
	LS_INVALID,		// separator: to check a valid entry link: linkstatus < LS_INVALID
	LS_CYCLIC,

	LS_FILE_OK,
	LS_FILE_INVALID,
	LS_FILE_UNAVAILABLE,
	LS_FILE_UNKNOWN
};


class UndoEdit : public Undoable
{
	public:
							UndoEdit (	int position,
										const Glib::ustring& text,
										UndoableType type )
		: Undoable( "placeholder", type ), m_position( position ), m_text( text ) { }

		unsigned int		get_position (void) const
		{
			return m_position;
		}

		Glib::ustring		get_text (void) const
		{
			return m_text;
		}

	protected:
		virtual void		merge (Undoable*) = 0;

		virtual void		undo (void) = 0;
		virtual void		redo (void) = 0;

		void				erase (void)
		{
			Gtk::TextIter iter_start = m_ptr2buffer->get_iter_at_offset( m_position );
			Gtk::TextIter iter_end = m_ptr2buffer->get_iter_at_offset( m_position +
													m_text.size() );
			m_ptr2buffer->erase( iter_start, iter_end );
		}

		void				insert (void)
		{
			Gtk::TextIter iterator = m_ptr2buffer->get_iter_at_offset( m_position );
			m_ptr2buffer->insert( iterator, m_text );
		}

		unsigned int		m_position;
		Glib::ustring		m_text;

	private:
		static Gtk::TextBuffer		*m_ptr2buffer;

	friend class TextbufferDiary;
};

class UndoInsert :public UndoEdit
{
	public:
							UndoInsert (int position, const Glib::ustring& text)
		: UndoEdit( position, text, UT_INSERT_TEXT ) { }

		bool				can_merge (const Undoable* action) const
		{
			if ( ( action->get_time_sec() - m_time.tv_sec ) < UNDO_MERGE_TIMEOUT )
				if ( action->get_type() == m_type )
					if ( dynamic_cast< const UndoInsert* >( action )->get_position()
							== m_position + m_text.size() )
					return true;
			return false;
		}

		void				merge (Undoable* action)
		{
			m_text += dynamic_cast< UndoInsert* >( action )->get_text();
		}

	protected:
		void				undo (void)
		{
			erase();
		}
		void				redo (void)
		{
			insert();
		}
};

class UndoErase : public UndoEdit
{
	public:
							UndoErase (int position, const Glib::ustring& text)
		: UndoEdit( position, text, UT_ERASE_TEXT ) { }

		bool				can_merge (const Undoable* action) const
		{
			if ( ( action->get_time_sec() - m_time.tv_sec ) < UNDO_MERGE_TIMEOUT )
				if ( action->get_type() == m_type )
				{
					const UndoErase * action_erase =
						dynamic_cast< const UndoErase* >( action );
					if ( action_erase->get_position() +
							action_erase->get_text().size()
								== m_position )
						return true;
				}
			return false;
		}

		void				merge (Undoable* action)
		{
			UndoErase * action_erase = dynamic_cast< UndoErase* >( action );
			m_text.insert( 0, action_erase->get_text() );
			m_position = action_erase->get_position();
		}

	protected:
		void				undo (void)
		{
			insert();
		}
		void				redo (void)
		{
			erase();
		}

};

class Link
{
	public:
									~Link();
		virtual void				go( void ) const = 0;

		// position of link in the buffer:
		Glib::RefPtr< Gtk::TextMark >
									m_mark_start;
		Glib::RefPtr< Gtk::TextMark >
									m_mark_end;

	protected:
									Link(	const Glib::RefPtr< Gtk::TextMark >&,
											const Glib::RefPtr< Gtk::TextMark >& );
};

class LinkEntry : public Link
{
	public:
		typedef sigc::signal< void, Date >
									Signal_void_Date;

									LinkEntry(	const Glib::RefPtr< Gtk::TextMark >&,
												const Glib::RefPtr< Gtk::TextMark >&,
												Date,
												unsigned int = 0 );
		void						go( void ) const;

		static Signal_void_Date		signal_activated( void )
		{ return m_signal_activated; }

		Date						m_date;
		unsigned int				m_order;	// a day can have multiple entries
	protected:
		static Signal_void_Date		m_signal_activated;
};

class LinkUri : public Link
{
	public:
									LinkUri(	const Glib::RefPtr< Gtk::TextMark >&,
												const Glib::RefPtr< Gtk::TextMark >&,
												const std::string& );
		void						go( void ) const;

		std::string					m_url;
		static Gtk::TextBuffer		*m_ptr2buffer;
};


class TextbufferDiary : public Gtk::TextBuffer
{
	public:
		typedef sigc::signal< LinkStatus, Date >
									Signal_LinkStatus_Date;
		typedef sigc::signal< void, const Glib::ustring& >
									Signal_void_ustring;

									TextbufferDiary( UndoManager*, Database* );
		static TextbufferDiary *	create( UndoManager*, Database* );
		void						set_textview( Gtk::TextView *ptr2textview )
		{
			m_ptr2textview = ptr2textview;
			set_spellcheck( m_ptr2database->get_spellcheck() );
		}
		void						reset( void );

		Glib::ustring				get_title( void ) const;
		const Link*					get_link( int ) const;

		void						set_richtext( const Glib::ustring& );
		void						set_richtext( const Entry* );
		void						set_search( bool );
		void						set_searchword( const Glib::ustring& );
		bool						select_searchword_previous( void );
		bool						select_searchword_next( void );

		void						toggle_format(	Glib::RefPtr< Tag >,
													const Glib::ustring& );
		void						toggle_bold( void );
		void						toggle_italic( void );
		void						toggle_highlight( void );
		void						toggle_strikethrough( void );

		void						set_spellcheck( bool );
		void						edit_theme( void );
		void						reset_theme( void );
		void						set_theme( Theme* );
		void						adjust_colors( void );

		Signal_LinkStatus_Date		signal_link_needs_checking( void )
		{
			return m_signal_link_needs_checking;
		}
		Signal_void_ustring			signal_title_changed( void )
		{
			return m_signal_title_changed;
		}

		void						handle_menu( Gtk::Menu* );

	protected:
		void						set_text( const Glib::ustring& );
		void						on_insert(	const Gtk::TextIter&,
												const Glib::ustring&,
												int );
		void						on_erase(	const Gtk::TextIter&,
												const Gtk::TextIter& );

		// PARSING
		void						parse( unsigned int, unsigned int );

		void						add_char_toword(	Glib::ustring&,
														Gtk::TextIter&,
														const Gtk::TextIter&,
														const gunichar );

		void						process_markup(		LookingFor&, LookingFor&,
														const char&, gunichar&,
														const Glib::RefPtr<Tag>&,
														Gtk::TextIter&,
														const Gtk::TextIter& );

		void						process_immediate(	LookingFor& );

		void						add_link(			const std::string&,
														LookingFor&,
														const Gtk::TextIter&,
														const Gtk::TextIter& );

		void						add_link_date(		const Date&,
														LookingFor&,
														const Gtk::TextIter&,
														const Gtk::TextIter& );

		// LINKS
		void						clear_links( int, int );
		void						clear_links( void );

		// TAGS
		static Glib::RefPtr< Tag >	m_tag_heading;
		static Glib::RefPtr< Tag >	m_tag_subheading;
		static Glib::RefPtr< Tag >	m_tag_match;
		static Glib::RefPtr< Tag >	m_tag_markup;
		static Glib::RefPtr< Tag >	m_tag_bold;
		static Glib::RefPtr< Tag >	m_tag_italic;
		static Glib::RefPtr< Tag >	m_tag_strikethrough;
		static Glib::RefPtr< Tag >	m_tag_highlight;
		static Glib::RefPtr< Tag >	m_tag_link;
		static Glib::RefPtr< Tag >	m_tag_link_broken;

		// THEMING
		static Pango::FontDescription
									m_theme_font;
		static Gdk::Color			m_theme_color_base;
		static Gdk::Color			m_theme_color_text;
		static Gdk::Color			m_theme_color_match;
		static Gdk::Color			m_theme_color_markup;
		static Gdk::Color			m_theme_color_heading;
		static Gdk::Color			m_theme_color_subheading;
		static Gdk::Color			m_theme_color_highlight;
		static Gdk::Color			m_theme_color_link;
		static Gdk::Color			m_theme_color_linkbroken;

		// OTHER VARIABLES
		Glib::ustring				m_title;
		Glib::ustring				m_searchword;
		bool						m_option_search;
		bool						m_flag_settextoperation;
		bool						m_flag_ongoingoperation;
		UndoManager					*m_ptr2undomanager;
		Database					*m_ptr2database;
		Gtk::TextView				*m_ptr2textview;
		GtkSpell					*m_ptr2spellobj;

		typedef std::list< Link* >	ListLinks;
		ListLinks					m_list_links;

		Signal_LinkStatus_Date		m_signal_link_needs_checking;
		Signal_void_ustring			m_signal_title_changed;

	friend class DialogTheme;
};


class TextviewDiary : public Gtk::TextView
{
	public:
									TextviewDiary( TextbufferDiary* );

	protected:
		void						update_link( void );
		virtual bool				on_motion_notify_event( GdkEventMotion* );
		virtual bool				on_button_release_event( GdkEventButton* );
		virtual bool				on_key_press_event( GdkEventKey* );
		virtual bool				on_key_release_event( GdkEventKey* );
		//~ virtual void				on_populate_popup( Gtk::Menu* );
//		virtual void				on_style_changed( const Glib::RefPtr< Gtk::Style >& );

	private:
		TextbufferDiary				*m_ptr2buffer;
		const Gdk::Cursor			m_cursor_hand;
		const Gdk::Cursor			m_cursor_xterm;
		const Gdk::Cursor			*m_ptr2cursor_last;
		const Link					*m_link_hovered;
};

} // end of namespace LIFEO

#endif

