/*
 * Line6 Linux USB driver - 0.7.4
 *
 * Copyright (C) 2004-2008 Markus Grabner (grabner@icg.tugraz.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include "driver.h"

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>

#include "audio.h"
#include "pcm.h"
#include "pod.h"


/* hardware definition (playback) */
static struct snd_pcm_hardware snd_pod_playback_hw = {
	.info = (SNDRV_PCM_INFO_MMAP |
					 SNDRV_PCM_INFO_INTERLEAVED |
					 SNDRV_PCM_INFO_BLOCK_TRANSFER |
					 SNDRV_PCM_INFO_MMAP_VALID |
					 SNDRV_PCM_INFO_PAUSE |
					 SNDRV_PCM_INFO_SYNC_START),
	.formats =          SNDRV_PCM_FMTBIT_S24_3LE,
	.rates =            SNDRV_PCM_RATE_KNOT,
	.rate_min =         39062,
	.rate_max =         39063,
	.channels_min =     2,
	.channels_max =     2,
	.buffer_bytes_max = 60000,
	.period_bytes_min = POD_ISO_PACKET_SIZE_MAX * POD_BYTES_PER_FRAME,  /* at least one URB must fit into one period */
	.period_bytes_max = 32768,
	.periods_min =      1,
	.periods_max =      1024,
};


/*
	Software stereo volume control.
*/
static void change_volume(struct urb *urb_out, int volume[])
{
	int chn = 0;
	unsigned char *p, *buf_end;

	if(volume[0] == 256 && volume[1] == 256)
		return;  /* maximum volume - no change */

	p = (unsigned char *)urb_out->transfer_buffer;
	buf_end = p + urb_out->transfer_buffer_length;

	for(; p < buf_end; p += 3) {
		int val = p[0] + (p[1] << 8) + ((signed char)p[2] << 16);
		val = (val * volume[chn & 1]) >> 8;
		p[0] = val;
		p[1] = val >> 8;
		p[2] = val >> 16;
		++chn;
	}
}

/*
	Find a free URB, prepare audio data, and submit URB.
*/
static int submit_audio_out_urb(struct snd_pcm_substream *substream)
{
	int index;
	unsigned long flags;
	int i, urb_size, urb_frames;
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct urb *urb_out;

	spin_lock_irqsave(&chip->lock_audio_out, flags);
	index = find_first_zero_bit(&chip->active_urb_out, POD_ISO_BUFFERS);

	if(index < 0 || index >= POD_ISO_BUFFERS) {
		spin_unlock_irqrestore(&chip->lock_audio_out, flags);
		dev_err(s2m(substream), "no free URB found\n");
		return -EINVAL;
	}

	urb_out = chip->urb_audio_out[index];
	urb_size = 0;

	for(i = 0; i < POD_ISO_PACKETS; ++i) {
		/* compute frame size for given sampling rate */
		int n, fs;
		struct usb_iso_packet_descriptor *fout = &urb_out->iso_frame_desc[i];
		chip->count_out += POD_FRAME_INCREMENT;
		n = chip->count_out / POD_FRAME_FACTOR;
		chip->count_out -= n * POD_FRAME_FACTOR;
		fs = n * POD_BYTES_PER_FRAME;
		fout->offset = urb_size;
		fout->length = fs;
		urb_size += fs;
	}

	urb_frames = urb_size / POD_BYTES_PER_FRAME;

	if(test_bit(BIT_PAUSE_PLAYBACK, &chip->flags)) {
		urb_out->transfer_buffer = chip->wrap_out;
		memset(chip->wrap_out, 0, urb_size);
	}
	else {
		if(chip->pos_out + urb_frames > runtime->buffer_size) {
			/*
				The transferred area goes over buffer boundary,
				copy the data to the temp buffer.
			*/
			int len;
			len = runtime->buffer_size - chip->pos_out;
			urb_out->transfer_buffer = chip->wrap_out;
			
			if(len > 0) {
				memcpy(chip->wrap_out, runtime->dma_area + chip->pos_out * POD_BYTES_PER_FRAME, len * POD_BYTES_PER_FRAME);
				memcpy(chip->wrap_out + len * POD_BYTES_PER_FRAME, runtime->dma_area, (urb_frames - len) * POD_BYTES_PER_FRAME);
			}
			else
				dev_err(s2m(substream), "driver bug: len = %d\n", len);  /* this is somewhat paranoid */
		}
		else {
			/* set the buffer pointer */
			urb_out->transfer_buffer = runtime->dma_area + chip->pos_out * POD_BYTES_PER_FRAME;
		}
	}

	if((chip->pos_out += urb_frames) >= runtime->buffer_size)
		chip->pos_out -= runtime->buffer_size;

	urb_out->transfer_buffer_length = urb_size;
	urb_out->context = substream;
	change_volume(urb_out, chip->volume);

	if(usb_submit_urb(urb_out, GFP_ATOMIC) == 0)
		set_bit(index, &chip->active_urb_out);
	else
		dev_err(s2m(substream), "URB out #%d submission failed\n", index);
	
	spin_unlock_irqrestore(&chip->lock_audio_out, flags);
	return 0;
}

/*
	Submit all currently available playback URBs.
*/
static int submit_audio_out_all_urbs(struct snd_pcm_substream *substream)
{
	int ret, i;

	for(i = 0; i < POD_ISO_BUFFERS; ++i)
		if((ret = submit_audio_out_urb(substream)) < 0)
			return ret;

	return 0;
}

/*
	Unlink all currently active playback URBs.
*/
static void unlink_audio_out_urbs(struct snd_pod_pcm *chip)
{
	unsigned int i;

	for(i = POD_ISO_BUFFERS; i--;) {
		if(test_bit(i, &chip->active_urb_out)) {
			if(!test_and_set_bit(i, &chip->unlink_urb_out)) {
				struct urb *u = chip->urb_audio_out[i];
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
				u->transfer_flags |= URB_ASYNC_UNLINK;
#endif			 
				usb_unlink_urb(u);
			}
		}
	}
}

/*
	Wait until unlinking of all currently active playback URBs has been finished.
*/
static void wait_clear_audio_out_urbs(struct snd_pod_pcm *chip)
{
	int timeout = HZ;
	unsigned int i;
	int alive;

	do {
		alive = 0;
		for (i = POD_ISO_BUFFERS; i--;) {
			if (test_bit(i, &chip->active_urb_out))
				alive++;
		}
		if (! alive)
			break;
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1);
	} while (--timeout > 0);
	if (alive)
		snd_printk(KERN_ERR "timeout: still %d active urbs..\n", alive);

	chip->active_urb_out = 0;
	chip->unlink_urb_out = 0;
}

/*
	Unlink all currently active playback URBs, and wait for finishing.
*/
void unlink_wait_clear_audio_out_urbs(struct snd_pod_pcm *chip)
{
	unlink_audio_out_urbs(chip);
	wait_clear_audio_out_urbs(chip);
}

/*
	Callback for completed playback URB.
*/
static void audio_out_callback(struct urb *urb PT_REGS)
{
	int i, index, length = 0, shutdown = 0;
	unsigned long flags;

	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)urb->context;
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;

	/* find index of URB */
	for(index = POD_ISO_BUFFERS; index--;)
		if(urb == chip->urb_audio_out[index])
			break;

	if(index < 0)
		return;  /* URB has been unlinked asynchronously */

	for(i = POD_ISO_PACKETS; i--;)
		length += urb->iso_frame_desc[i].length;

	spin_lock_irqsave(&chip->lock_audio_out, flags);
	chip->pos_out_done += length / POD_BYTES_PER_FRAME;

	if(chip->pos_out_done >= runtime->buffer_size)
		chip->pos_out_done -= runtime->buffer_size;
	
	clear_bit(index, &chip->active_urb_out);

	for(i = POD_ISO_PACKETS; i--;)
		if(urb->iso_frame_desc[i].status == -ESHUTDOWN) {
			shutdown = 1;
			break;
		}

	if(test_bit(index, &chip->unlink_urb_out))
		shutdown = 1;

	spin_unlock_irqrestore(&chip->lock_audio_out, flags);

	if(!shutdown) {
		submit_audio_out_urb(substream);

		if((chip->bytes_out += length) >= chip->period_out) {
			chip->bytes_out -= chip->period_out;
			snd_pcm_period_elapsed(substream);
		}
	}
}

/* open playback callback */
static int snd_pod_playback_open(struct snd_pcm_substream *substream)
{
	int err;
	struct snd_pcm_runtime *runtime = substream->runtime;

	if((err = snd_pcm_hw_constraint_ratdens(runtime, 0, SNDRV_PCM_HW_PARAM_RATE, &pod_rates)) < 0)
		return err;

	runtime->hw = snd_pod_playback_hw;
	return 0;
}

/* close playback callback */
static int snd_pod_playback_close(struct snd_pcm_substream *substream)
{
	return 0;
}

/* hw_params playback callback */
static int snd_pod_playback_hw_params(struct snd_pcm_substream *substream, struct snd_pcm_hw_params *hw_params)
{
	int ret;
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);

	/* -- Florian Demski [FD] */
	/* don't ask me why, but this fixes the bug on my machine */
	if ( chip == NULL ) {
		if ( substream->pcm == NULL )
			return -ENOMEM;
		if ( substream->pcm->private_data == NULL )
			return -ENOMEM;
		substream->private_data = substream->pcm->private_data;
		chip = snd_pcm_substream_chip( substream );
	}
	/* -- [FD] end */

	if((ret = snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params))) < 0)
		return ret;

	chip->period_out = params_period_bytes(hw_params);
	chip->wrap_out = kmalloc(2 * POD_ISO_PACKET_SIZE_MAX, GFP_KERNEL);

	if(!chip->wrap_out) {
		dev_err(s2m(substream), "cannot malloc wrap_out\n");
		return -ENOMEM;
	}

	return 0;
}

/* hw_free playback callback */
static int snd_pod_playback_hw_free(struct snd_pcm_substream *substream)
{
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
	unlink_wait_clear_audio_out_urbs(chip);

	if(chip->wrap_out) {
		kfree(chip->wrap_out);
		chip->wrap_out = 0;
	}

	return snd_pcm_lib_free_pages(substream);
}

/* trigger playback callback */
int snd_pod_playback_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
	int err;
	chip->count_out = 0;

	switch(cmd) {
	case SNDRV_PCM_TRIGGER_START:
		if(!test_and_set_bit(BIT_RUNNING_PLAYBACK, &chip->flags)) {
			err = submit_audio_out_all_urbs(substream);

			if(err < 0) {
				clear_bit(BIT_RUNNING_PLAYBACK, &chip->flags);
				return err;
			}
		}

		break;

	case SNDRV_PCM_TRIGGER_STOP:
		if(test_and_clear_bit(BIT_RUNNING_PLAYBACK, &chip->flags))
			unlink_audio_out_urbs(chip);

		break;

	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		set_bit(BIT_PAUSE_PLAYBACK, &chip->flags);
		break;

	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		clear_bit(BIT_PAUSE_PLAYBACK, &chip->flags);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* playback pointer callback */
static snd_pcm_uframes_t
snd_pod_playback_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
	return chip->pos_out_done;
}

/* playback operators */
struct snd_pcm_ops snd_pod_playback_ops = {
	.open =        snd_pod_playback_open,
	.close =       snd_pod_playback_close,
	.ioctl =       snd_pcm_lib_ioctl,
	.hw_params =   snd_pod_playback_hw_params,
	.hw_free =     snd_pod_playback_hw_free,
	.prepare =     snd_pod_prepare,
	.trigger =     snd_pod_trigger,
	.pointer =     snd_pod_playback_pointer,
};

int create_audio_out_urbs(struct snd_pod_pcm *chip)
{
	int i;
	struct usb_line6_pod *pod = chip->pod;

	/* create audio URBs and fill in constant values: */
	for(i = 0; i < POD_ISO_BUFFERS; ++i) {
		struct urb *urb;

		/* URB for audio out: */
		urb = chip->urb_audio_out[i] = usb_alloc_urb(POD_ISO_PACKETS, GFP_KERNEL);

		if(urb == NULL) {
			dev_err(pod->line6.ifcdev, "Out of memory\n");
			return -ENOMEM;
		}

		urb->dev = pod->line6.usbdev;
		urb->pipe = usb_sndisocpipe(pod->line6.usbdev, chip->ep_audio_write & USB_ENDPOINT_NUMBER_MASK);
		urb->transfer_flags = URB_ISO_ASAP;
		urb->start_frame = -1;
		urb->number_of_packets = POD_ISO_PACKETS;
		urb->interval = POD_ISO_INTERVAL;
		urb->error_count = 0;
		urb->complete = audio_out_callback;
	}

	return 0;
}
