#include <syslog.h>
#include <string.h>
#include "../lcp3.h"
#include "cmd.h"
#include "cmd_lcp3.h"
#include "client.h"
#include "proc_supp.h"
#include "con.h"
#include "execute.h"
// which of the following header files get taken depends on the
// results of configure
#include "user.h"
#include "user_pam.h"
#include "user_noauth.h"

int process_lcp3_order(struct t_lcp3_cmd *pack, struct sockaddr_in *from, int sock, char type)
{
    int dum;
    struct line_t* line = (struct line_t*)lstlines.first;
    struct t_lcp3_info_auth* p_info_auth = (void*)((char*)pack + sizeof(struct t_lcp3_cmd));
    struct t_lcp3_info_enckey* p_info_enckey = (void*)((char*)pack + sizeof(struct t_lcp3_cmd));
    struct t_lcp3_info_ack* p_info_ack = (void*)((char*)pack + sizeof(struct t_lcp3_cmd));
    struct t_lcp3_info_line* p_info_line = (void*)((char*)pack + sizeof(struct t_lcp3_cmd));
    struct t_lcp3_info_status* p_info_status = (void*)((char*)pack + sizeof(struct t_lcp3_cmd));
    struct client_t *who = get_client(from->sin_addr.s_addr, from->sin_port);
    if ( !who )
    {
		#ifdef DEBUG
		syslog(LOG_ERR, "client %s:%d not in list", inet_ntoa(from->sin_addr), ntohs(from->sin_port));
		#endif
		return(-2);
    }
    if ( type != SOCK_TYPE_TCP )
    {
		syslog(LOG_ERR, "process_lcp3_order(): (type != SOCK_TYPE_TCP)");
		return(-1);
    }
    
#ifdef DEBUG
    if ( who->type != LCS_CLT_LCP3 ) syslog(LOG_DEBUG, "client was not set to LCS_CLT_LCP3... corrected.");
    syslog(LOG_DEBUG, "lcp3: got cmd %d", pack->cmd);
#endif
    who->type = LCS_CLT_LCP3;
    // send an ack
    if ( pack->cmd != CMD3_ACK )
    {
	if ( lcp3_ack_send(pack->cmd, who) )
	{
	    syslog(LOG_WARNING, "%s:%d unable to send ack", inet_ntoa(from->sin_addr), ntohs(from->sin_port));
	    kick_client(who, 7);
	    return(0);
	}
    }
    // update client timeout
    who->timeout = time(NULL) + LCP3_CLT_TIMEOUT;
    switch ( pack->cmd )
    {
	case CMD3_AUTHREQ:
	    p_info_auth->name[LCP3_MAX_NAME_LEN] = 0; // avoid strlen segf
	    p_info_auth->pass[LCP3_MAX_PWD_LEN] = 0;
	    if ( who->status == CLT_ONLINE )
	    {
			lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
			break;
	    }
	    if ( !server->use_user_accounting )
	    {
			strcpy(p_info_auth->name, inet_ntoa(from->sin_addr));
			who->status = CLT_OFFLINE;
	    }
	    else
	    {
			switch ( p_info_auth->enctype )
			{
			    case LCP3_ENCTYPE_NONE:
					if ( !vrfy_user(p_info_auth->name, p_info_auth->pass) )
				    who->status = CLT_OFFLINE;
					break;
			    case LCP3_ENCTYPE_SYM: // access denied, not implemented
					syslog(LOG_INFO, "LCP3_ENCTYPE_SYM not supported yet.");
					break;
			    case LCP3_ENCTYPE_ASYM: // access denied, not implemented
					syslog(LOG_INFO, "LCP3_ENCTYPE_ASYM not supported yet.");
					break;
			}
	    }
	    if ( who->status == CLT_OFFLINE )
	    { // user is ok
			strcpy(who->name, p_info_auth->name);
#ifdef SHUTDOWN_OK
// store passwd in the client record only if we really need it.
			who->passwd = strdup(p_info_auth->pass);
#endif
#ifdef DEBUG
			syslog(LOG_DEBUG, "%s: access allowed", p_info_auth->name);
#endif
			syslog(LOG_DEBUG, "%s:%d registered", inet_ntoa(from->sin_addr), ntohs(from->sin_port));
			lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
	    }
	    else
	    { // access denied
			syslog(LOG_INFO, "%s: access denied", p_info_auth->name);
			lcp3_cmd_direct(CMD3_AUTHBAD, who->tcp_sock, NULL, 0);
			kick_client(who, 6);
			who = NULL;
	    }
	    break;
	case CMD3_ONLINE:
		if ( who->status == CLT_NOAUTH || who->status == CLT_DISCON )
		// CLT_DISCON: just here for security, you never know...
		{
			syslog(LOG_INFO, "%s:%d: can't go online, is not authenticated!",
					inet_ntoa(from->sin_addr), ntohs(from->sin_port));
			break;
		}
	    check_con_status();
	    switch ( who->line->con_stat )
	    {
			case CST_DOWN:
			    who->status = CLT_DIALING;
			    call_con_up(who, who->line);
			    break;
			case CST_DIALING:
			    who->status = CLT_DIALING;
			    break;
			case CST_LOCKED_UP_TIMED:
			    set_line_status(who->line, CST_UP_SERVER);
			case CST_LOCKED_UP_CLT:
			case CST_UP_USER:
			case CST_UP_SERVER:
			    // not sent as broadcast --> don't queue (won't get ACK)!!
			    if ( who->status != CLT_ONLINE )
			    {
					who->line->online++;
					who->status = CLT_ONLINE;
					who->started = time(NULL);
                	if ( who->line->client_online[0] )
					    exec_dont_care_param(who->line->client_online, inet_ntoa(from->sin_addr));
					syslog(LOG_INFO, "%s:%d online", inet_ntoa(from->sin_addr), ntohs(from->sin_port));
			    }
			    break;
			case CST_CLOSING:
			    if ( who->status == CLT_ONLINE )
			    {
					who->line->online--;
                	if ( who->line->client_offline[0] )
					    exec_dont_care_param(who->line->client_offline, inet_ntoa(from->sin_addr));
			    }
			    who->status = CLT_CLOSING;
			    break;
//			case CST_UP_USER:
			case CST_CLOSE_ERROR:
			    if ( who->status == CLT_ONLINE )
			    {
					who->line->online--;
                	if ( who->line->client_offline[0] )
					    exec_dont_care_param(who->line->client_offline, inet_ntoa(from->sin_addr));
			    }
			    who->status = CLT_OFFLINE;
			    break;
			default:
			    ; // uh... don't dial -> status would be incorrect
	    }
	    lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
	    break; // CMD3_ONLINE
	case CMD3_OFFLINE:
	    if ( who->status == CLT_ONLINE )
	    {
		who->line->online--;
		client_log_times(who);
		if ( (who->line->online == 0) && (who->line->con_stat == CST_UP_SERVER) )
		{
		    who->status = CLT_CLOSING;
		    call_con_dn(who, who->line);
		}
		else
		{
		    who->status = CLT_OFFLINE;
		}
        	if ( who->line->client_offline[0] )
		    exec_dont_care_param(who->line->client_offline, inet_ntoa(from->sin_addr));
		who->started = 0;
		syslog(LOG_INFO, "%s:%d offline", inet_ntoa(from->sin_addr), ntohs(from->sin_port));
	    }
	    lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
	    break;
	case CMD3_UNREG:
	    kick_client(who, 1);
	    break;
	case CMD3_LINESREQ:	// LINE
	    line = (struct line_t*)lstlines.first;
	    while ( line )
	    {
			strncpy(p_info_line->linename, line->linename, LCP3_LINENAME_LEN+1);
			// for the case that linename was to long:
			line->linename[LCP3_LINENAME_LEN] = 0;
			p_info_line->line_id = line->id;
			p_info_line->total_lines = lstlines.count;
			p_info_line->capability_baud_up = line->baud_up;
			p_info_line->capability_baud_down = line->baud_down;
			lcp3_cmd_queue(CMD3_LINE, who, p_info_line, sizeof(struct t_lcp3_info_line));
			line = (struct line_t*)line->next;
	    }
	    break;
	case CMD3_STATREQ:
	    dum = 0;
	    line = (struct line_t*)lstlines.first;
	    while ( line )
	    {
			dum += line->online;
			line = (struct line_t*)line->next;
	    }
	    // LINE (send status of pack->line_id)
	    p_info_status->clients = cltlist.count;
	    p_info_status->online = dum;
	    lcp3_cmd_queue(CMD3_STATUS, who, p_info_status, sizeof(struct t_lcp3_info_status));
	    break;
	case CMD3_SHUTDOWN:
#ifdef SHUTDOWN_OK
	    if ( !shutdown_allowed(who) )
	    {
			static char *shutdownmsg = "requested server shutdown";
			switch ( server->use_user_accounting )
			{
			    case 0:
					syslog(LOG_INFO, "%s:%d %s", inet_ntoa(from->sin_addr),
						    ntohs(from->sin_port), shutdownmsg);
					break;
			    case 1:
					syslog(LOG_INFO, "%s %s", who->name, shutdownmsg);
					break;
			}
			shutdown_server_machine();
			kick_client(who, 1);
	    }
	    else
	    { // client may not shutdown the server machine
			p_info_ack->ackcmd = CMD3_SHUTDOWN;
			lcp3_cmd_queue(CMD3_NOACCESS, who, p_info_ack, sizeof(struct t_lcp3_info_ack));
	    }
#else
	    // feature not implemented
	    p_info_ack->ackcmd = CMD3_SHUTDOWN;
	    lcp3_cmd_queue(CMD3_INVALID, who, p_info_ack, sizeof(struct t_lcp3_info_ack));
#endif
	    break;
	case CMD3_PUBKEYREQ:
	    // send public key, hash or null-string for
	    //      asym        sym  or no encryption of passwd
	    memset(p_info_enckey, 0, sizeof(struct t_lcp3_info_enckey));
	    if ( server->use_user_accounting )
			p_info_enckey->enctype = LCP3_ENCTYPE_NONE;
	    else
			p_info_enckey->enctype = LCP3_ENCTYPE_OPEN;
	    lcp3_cmd_queue(CMD3_ENCINFO, who, p_info_enckey, sizeof(struct t_lcp3_info_enckey));
	    break;
	case CMD3_LINESEL:
	    line = (struct line_t*)lstlines.first;
	    while ( line )
	    {
			if ( pack->line_id == line->id ) break;
			line = (struct line_t*)line->next;
	    }
	    if ( line ) who->line = line;
	    lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
	    break;
	case CMD3_ACK:
	    lcp3_proc_ack(p_info_ack->ackcmd, who);
	    break;
	case CMD3_CHANADD:
	    add_isdn_channel(who, who->line);
	    break;
	case CMD3_CHANRM:
	    remove_isdn_channel(who, who->line);
	    break;
	case CMD3_REFRESH: break;
	case CMD3_LOCK:
	    if ( (who->line->line_locking)
		    && (who->line->con_stat == CST_UP_SERVER) )
	    {
			static char *lockmsg = "locked line";
			set_line_status(who->line, CST_LOCKED_UP_CLT);
			switch ( server->use_user_accounting )
			{
			    case 0:
					syslog(LOG_INFO, "%s:%d %s [%s:%d]", inet_ntoa(from->sin_addr),
						    ntohs(from->sin_port), lockmsg,
						    who->line->linename, who->line->id);
					break;
			    case 1:
					syslog(LOG_INFO, "%s %s [%s:%d]", who->name, lockmsg,
						    who->line->linename, who->line->id);
					break;
			}
	    }
	    else if ( (who->line->line_locking)
		    && (who->line->con_stat == CST_DOWN) )
	    {
			static char *lockmsg = "locked line down";
			set_line_status(who->line, CST_LOCKED_DOWN);
			switch ( server->use_user_accounting )
			{
			    case 0:
					syslog(LOG_INFO, "%s:%d %s [%s:%d]", inet_ntoa(from->sin_addr),
						    ntohs(from->sin_port), lockmsg,
						    who->line->linename, who->line->id);
					break;
			    case 1:
					syslog(LOG_INFO, "%s %s [%s:%d]", who->name, lockmsg,
						    who->line->linename, who->line->id);
					break;
			}
	    }
	    else if ( ! who->line->line_locking )
	    {
			p_info_ack->ackcmd = CMD3_LOCK;
			lcp3_cmd_queue(CMD3_NOACCESS, who, p_info_ack, sizeof(struct t_lcp3_info_ack));
	    }
	    break;
	case CMD3_UNLOCK:
	    if ( who->line->con_stat == CST_LOCKED_UP_CLT )
		{
			set_line_status(who->line, CST_UP_SERVER);
		} else if ( who->line->con_stat == CST_LOCKED_DOWN )
        {
			set_line_status(who->line, CST_DOWN);
		} 
	    { // extra block
			static char *lockmsg = "unlocked line";
			switch ( server->use_user_accounting )
			{
			    case 0:
					syslog(LOG_INFO, "%s:%d %s [%s:%d]", inet_ntoa(from->sin_addr),
						    ntohs(from->sin_port), lockmsg,
						    who->line->linename, who->line->id);
					break;
			    case 1:
					syslog(LOG_INFO, "%s %s [%s:%d]", who->name, lockmsg,
						    who->line->linename, who->line->id);
					break;
			}
	    }
	    break;
	case CMD3_GETALL:
		who->getall = 1;
		break;
	case CMD3_UNGETALL:
		who->getall = 0;
		break;
	default:
	    if ( pack->cmd >= MIN_CBR3 ) break; // unneccessary? !
	    p_info_ack->ackcmd = pack->cmd;
	    lcp3_cmd_queue(CMD3_INVALID, who, p_info_ack, sizeof(struct t_lcp3_info_ack));
    }
    return(0);
}
