/* $Id: gtkcellrendererspin.c,v 1.2 2005/03/24 10:27:25 marcusva Exp $
 * GtkCellRendererSpin
 * Copyright (C) 2004-2005 Marcus von Appen
 * Copyright (C) 2004 Lorenzo Gil Sanchez
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkspinbutton.h>
#include "gtkcellrendererspin.h"
#include "../../li18n.h"

static void
gtk_cell_renderer_spin_class_init (GtkCellRendererSpinClass *klass);
static void gtk_cell_renderer_spin_init (GtkCellRendererSpin *self);
static GtkCellEditable*
gtk_cell_renderer_spin_start_editing (GtkCellRenderer *cell,
                                      GdkEvent *event,GtkWidget *widget,
                                      const gchar *path,
                                      GdkRectangle *background_area,
                                      GdkRectangle *cell_area,
                                      GtkCellRendererState flags);
static void gtk_cell_renderer_spin_get_property (GObject *object,
                                                 guint param_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);
static void gtk_cell_renderer_spin_set_property (GObject *object,
                                                 guint param_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);

static GObjectClass *parent_class = NULL;

#define GTK_CELL_RENDERER_SPIN_PATH "gtk-cell-renderer-spin-path"

enum {
     PROP_MIN = 1,
     PROP_MAX,
     PROP_STEP
};

GType
gtk_cell_renderer_spin_get_type (void)
{
     static GType gtk_cell_renderer_spin_type = 0;
     
     if (!gtk_cell_renderer_spin_type)
     {
          static const GTypeInfo gtk_cell_renderer_spin_info = 
               {
                    sizeof (GtkCellRendererSpinClass),
                    (GBaseInitFunc)     NULL,
                    (GBaseFinalizeFunc) NULL,
                    (GClassInitFunc)    gtk_cell_renderer_spin_class_init,
                    NULL,
                    NULL,
                    sizeof (GtkCellRendererSpin),
                    0,
                    (GInstanceInitFunc) gtk_cell_renderer_spin_init
               };
          gtk_cell_renderer_spin_type =
               g_type_register_static (GTK_TYPE_CELL_RENDERER_TEXT,
                                       "GtkCellRendererSpin",
                                       &gtk_cell_renderer_spin_info, 0);
     }
     return gtk_cell_renderer_spin_type;
}

static void
gtk_cell_renderer_spin_class_init (GtkCellRendererSpinClass *klass)
{
     GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (klass);
     GObjectClass *object_class = G_OBJECT_CLASS (klass);
     
     parent_class = g_type_class_peek_parent (klass);
     
     cell_class->start_editing = gtk_cell_renderer_spin_start_editing;
     
     object_class->set_property = gtk_cell_renderer_spin_set_property;
     object_class->get_property = gtk_cell_renderer_spin_get_property;
     
     g_object_class_install_property (object_class, PROP_MIN,
                                      g_param_spec_double ("min-value",
                                                           N_("Minimum value"),
                                                           N_("Minimum value"),
                                                           -G_MAXDOUBLE,
                                                           G_MAXDOUBLE,
                                                           1.0,
                                                           G_PARAM_READABLE |
                                                           G_PARAM_WRITABLE));
     g_object_class_install_property (object_class, PROP_MAX,
                                      g_param_spec_double ("max-value",
                                                           N_("Maximum value"),
                                                           N_("Maximum value"),
                                                           -G_MAXDOUBLE,
                                                           G_MAXDOUBLE,
                                                           1.0,
                                                           G_PARAM_READABLE |
                                                           G_PARAM_WRITABLE));
     g_object_class_install_property (object_class, PROP_STEP,
                                      g_param_spec_double ("step-value",
                                                           N_("Step value"),
                                                           N_("Step value"),
                                                           0.0,
                                                           G_MAXDOUBLE,
                                                           1.0,
                                                           G_PARAM_READABLE |
                                                           G_PARAM_WRITABLE));
}

static void
gtk_cell_renderer_spin_get_property (GObject *object, guint param_id,
                                     GValue *value, GParamSpec *pspec)
{
     GtkCellRendererSpin* spin = GTK_CELL_RENDERER_SPIN (object);
     switch (param_id)
     {
     case PROP_MIN:
          g_value_set_double (value, spin->min);
          break;
     case PROP_MAX:
          g_value_set_double (value, spin->max);
          break;
     case PROP_STEP:
          g_value_set_double (value, spin->step);
          break;
     default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
          break;
     }
     return;
}

static void
gtk_cell_renderer_spin_set_property (GObject *object, guint param_id,
                                     const GValue *value, GParamSpec *pspec)
{
     GtkCellRendererSpin* spin = GTK_CELL_RENDERER_SPIN (object);
     switch (param_id)
     {
     case PROP_MIN:
          spin->min = g_value_get_double (value);
          g_object_notify (G_OBJECT (object), "min-value");
          break;
     case PROP_MAX:
          spin->max = g_value_get_double (value);
          g_object_notify (G_OBJECT (object), "max-value");
          break;
     case PROP_STEP:
          spin->step = g_value_get_double (value);
          g_object_notify (G_OBJECT (object), "step-value");
          break;
     default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
          break;
     }
}


static void
gtk_cell_renderer_spin_init (GtkCellRendererSpin *self)
{
     self->min = 1.0;
     self->max = 10.0;
     self->step = 1.0;
}

GtkCellRenderer *
gtk_cell_renderer_spin_new (void)
{
     return GTK_CELL_RENDERER (g_object_new (GTK_TYPE_CELL_RENDERER_SPIN,
                                             NULL));
}

GtkCellRenderer *
gtk_cell_renderer_spin_new_with_range (gdouble min, gdouble max, gdouble step)
{
     GtkCellRendererSpin *renderer =
          GTK_CELL_RENDERER_SPIN (gtk_cell_renderer_spin_new ());

     g_return_val_if_fail (min < max, NULL);
     g_return_val_if_fail (step != 0.0, NULL);

     renderer->min = 1.0;
     renderer->max = 10.0;
     renderer->step = 1.0;

     return GTK_CELL_RENDERER (renderer);
}

void
gtk_cell_renderer_spin_set_range (GtkCellRendererSpin *renderer, gdouble min,
                                  gdouble max)
{
     g_return_if_fail (GTK_IS_CELL_RENDERER_SPIN (renderer));
     g_return_if_fail (min < max);

     renderer->min = min;
     renderer->max = max;
     return;
}

void
gtk_cell_renderer_spin_get_range (GtkCellRendererSpin *renderer, gdouble *min,
                                  gdouble *max)
{
     g_return_if_fail (GTK_IS_CELL_RENDERER_SPIN (renderer));
     
     if (min)
          *min = renderer->min;
     if (max)
          *max = renderer->max;
     return;
}

static gboolean
gtk_cell_renderer_spin_focus_out_event (GtkWidget *widget, GdkEvent *event,
                                        gpointer data)
{
     const gchar *path;
     const gchar *new_text;
     path = g_object_get_data (G_OBJECT (widget), GTK_CELL_RENDERER_SPIN_PATH);
     
     new_text = gtk_entry_get_text (GTK_ENTRY (widget));
     g_signal_emit_by_name (data, "edited", path, new_text);
     g_signal_handlers_disconnect_by_func
          (widget, gtk_cell_renderer_spin_focus_out_event, data);
     return FALSE;
}

static GtkCellEditable *
gtk_cell_renderer_spin_start_editing (GtkCellRenderer *cell, GdkEvent *event,
				      GtkWidget *widget,const gchar *path,
				      GdkRectangle *background_area,
				      GdkRectangle *cell_area,
				      GtkCellRendererState flags)
{
     GtkCellRendererText *cell_text;
     GtkCellRendererSpin *sp = GTK_CELL_RENDERER_SPIN (cell);
     GtkWidget *spin;
     
     cell_text = GTK_CELL_RENDERER_TEXT (cell);
     if (cell_text->editable == FALSE)
          return NULL;

     spin = gtk_spin_button_new_with_range (sp->min, sp->max, sp->step);
     gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
     gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), sp->min);

     if (cell_text->text)
          gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin),
                                     g_ascii_strtod (cell_text->text, NULL));

     g_object_set_data_full (G_OBJECT (spin), GTK_CELL_RENDERER_SPIN_PATH,
                             g_strdup (path), g_free);

     gtk_widget_show (spin);

     g_signal_connect (spin, "focus_out_event",
                       G_CALLBACK (gtk_cell_renderer_spin_focus_out_event),
                       cell);

     return GTK_CELL_EDITABLE (spin);
}
