/* $Id: gtklearn-wiz.c,v 1.26 2005/03/28 16:24:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <string.h>
#include <math.h>
#include <libxml/xmlstring.h>

#include "../li18n.h"
#include "../errors.h"
#include "../lesson.h"
#include "hig-widgets.h"
#include "gtkcellrendererspin.h"
#include "wizard.h"
#include "gtkdefs.h"

#include "gtkconfig.h"
#include "gtklesson.h"
#include "util-lang.h"
#include "util.h"
#include "gtklearn-wiz.h"

#define WIZ_STEP "\n%s %s"

#define LESSON_DESC "<b>%s %s</b>\n\n%s"

#define SESSION "session"
#define COLUMNNO "column_number"

/* used on page 1 */
enum
{
     LESSON_USED,
     LESSON_NAME,
     LESSON_COLUMNS
};

/* used on page 2 */
enum
{
     L_NAME,
     L_ACCESS,
     L_WORDS,
     L_MIN,
     L_MAX,
     L_COLUMNS
};

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

static const gchar *methods[] = { N_("Random"), N_("Sequence"), N_("Review"),
                                  N_("Learn") };

static GtkWidget* create_lesson_param_tree (learnSession *session);
static GtkWidget* create_page_1 (learnSession *session);
static GtkWidget* create_page_2 (learnSession *session);
static GtkWidget* create_page_3 (learnSession *session);
static GtkWidget* create_page_4 (learnSession *session);
static gboolean check_session_1 (learnSession *session);

static void switch_lesson (GtkTreeView *tree, GtkWidget *label);
static void change_amount (GtkCellRendererSpin *cell, const gchar *path,
                           const gchar *value, GtkListStore *list);
static void switch_access (GtkCellRendererText *cell, const gchar *path,
                           const gchar *value, GtkListStore *list);
static void toggle_use_lesson (GtkCellRendererToggle *cell, gchar *path,
                               GtkTreeView *tree);
static void refresh_list (GtkTreeView *tree, learnSession *session);

static GtkWidget*
create_lesson_param_tree (learnSession *session)
{
     GtkWidget *tree;
     GtkTreeIter iter;
     GtkListStore *list;
     GtkListStore *list_acc;
     GtkCellRenderer *renderer;
     learnLesson *tmp = session->lessons;
     guint i = 0;
     gdouble max_words = 2.0;

     list = gtk_list_store_new (L_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
                                G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT);
     /* set temporary data for callbacks */
     g_object_set_data (G_OBJECT (list), SESSION, session);
     while (tmp) /* create entries */
     {
          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              L_NAME, tmp->lesson->type,
                              L_WORDS, tmp->amount,
                              L_ACCESS, methods[tmp->access],
                              L_MIN, tmp->start,
                              L_MAX, tmp->end,
                              -1);
          max_words = (tmp->end > max_words) ? tmp->end : max_words;
          tmp = tmp->next;
     }
     
     tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (list);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tree),
                                                  L_NAME, _("Lesson"),
                                                  renderer,
                                                  "text", L_NAME, NULL);

     renderer = gtk_cell_renderer_combo_new ();
     list_acc = gtk_list_store_new (1, G_TYPE_STRING);
     while (i < G_N_ELEMENTS (methods))
     {
          gtk_list_store_append (list_acc, &iter);
          gtk_list_store_set (list_acc, &iter, 0, methods[i], -1);
          i++;
     }
     g_object_set (G_OBJECT (renderer), "model", list_acc, "editable", TRUE,
                   "text-column", 0, NULL);
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (switch_access), list);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tree),
                                                  L_ACCESS, _("Access Method"),
                                                  renderer,
                                                  "text", L_ACCESS, NULL);

     renderer = gtk_cell_renderer_spin_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE, "xalign", 1.0, NULL);
     g_object_set_data (G_OBJECT (renderer), COLUMNNO,
                        GUINT_TO_POINTER (L_WORDS));
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (change_amount), list);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tree),
                                                  L_WORDS, _("Amount"),
                                                  renderer,
                                                  "text", L_WORDS,
                                                  "max-value", L_MAX, NULL);
 
     renderer = gtk_cell_renderer_spin_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE, "xalign", 1.0, NULL);
     g_object_set_data (G_OBJECT (renderer), COLUMNNO,
                        GUINT_TO_POINTER (L_MIN));
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (change_amount), list);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tree),
                                                  L_MIN, _("Minimum ID"),
                                                  renderer,
                                                  "text", L_MIN,
                                                  "min-value", L_MIN,
                                                  "max-value", L_MAX, NULL);

     renderer = gtk_cell_renderer_spin_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE, "xalign", 1.0, NULL);
     g_object_set_data (G_OBJECT (renderer), COLUMNNO,
                        GUINT_TO_POINTER (L_MAX));
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (change_amount), list);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tree),
                                                  L_MAX, _("Maximum ID"),
                                                  renderer, "text", L_MAX,
                                                  "min-value", L_MIN,
                                                  "max-value", L_MAX, NULL);
     return tree;
}

static GtkWidget*
create_page_1 (learnSession *session)
{
     GtkWidget *box_step;
     GtkWidget *lbl_step;
     GtkWidget *box_main;
     GtkWidget *box_lessons;
     GtkWidget *btn_add;
     GtkWidget *scr_win;
     GtkWidget *view_lessons;
     GtkWidget *lbl_lesson;
     GtkCellRenderer *renderer;
     GtkTreeSelection *select;

     gchar *text = NULL;

     box_step = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_step), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_step), 0);
     text = g_strdup_printf (WIZ_STEP,
                             _("Select the lessons you want to learn from.\n\n"
                               "The <b>Use</b> toggle indicates, if the "
                               "lesson should be used in this session.\n\n"),
                             WIZARD_TEXT_NEXT);
     lbl_step = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_step), TRUE);
     gtk_label_set_markup (GTK_LABEL (lbl_step), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_step), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_step), lbl_step, FALSE, FALSE, 0);
     g_free (text);

     box_main = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_box_pack_start (GTK_BOX (box_step), box_main, TRUE, TRUE, 0);

     box_lessons = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_lessons), 0);
     gtk_box_pack_start (GTK_BOX (box_main), box_lessons, FALSE, TRUE, 0);

     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win),
                                     GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
     gtk_box_pack_start (GTK_BOX (box_lessons), scr_win, TRUE, TRUE, 0);

     view_lessons = gtk_tree_view_new ();
     refresh_list (GTK_TREE_VIEW (view_lessons), session);

     /* register object */
     main_app->objects = event_append_listener (main_app->objects,
                                                view_lessons,
                                                "lessons-changed",
                                                EV_CALLBACK (refresh_list),
                                                session);
     g_signal_connect (G_OBJECT (view_lessons), "destroy",
                       G_CALLBACK (util_event_unregister), NULL);

     /* set temporary data for the toggle callback */
     g_object_set_data (G_OBJECT (view_lessons), SESSION, session);

     gtk_container_add (GTK_CONTAINER (scr_win), view_lessons);

     select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view_lessons));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

     renderer = gtk_cell_renderer_toggle_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_lessons),
                                                  LESSON_USED, _("Use"),
                                                  renderer, "active",
                                                  LESSON_USED, NULL);
     g_signal_connect (G_OBJECT (renderer), "toggled",
                       G_CALLBACK (toggle_use_lesson), view_lessons);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_lessons),
                                                  LESSON_NAME, _("Lesson"),
                                                  renderer, "text",
                                                  LESSON_NAME, NULL);

     /* create add button for lessons */
     btn_add = gtk_button_new_with_mnemonic (_("Add _lesson"));
     g_signal_connect (G_OBJECT (btn_add), "clicked",
                       G_CALLBACK (gtklesson_add_lesson), NULL);
     gtk_box_pack_start (GTK_BOX (box_lessons), btn_add, FALSE, FALSE, 0);
     
     lbl_lesson = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_lesson), TRUE);
     gtk_misc_set_alignment (GTK_MISC (lbl_lesson), 0, 0);
     gtk_box_pack_start (GTK_BOX (box_main), lbl_lesson, FALSE, FALSE, 0);

     /* add signal handler for lesson descriptions */
     g_signal_connect (G_OBJECT (view_lessons), "cursor-changed",
                       G_CALLBACK (switch_lesson), lbl_lesson);

     return box_step;
}

static GtkWidget*
create_page_2 (learnSession *session)
{
     GtkWidget *box_step;
     GtkWidget *lbl_step;
     GtkWidget *scr_win;
     GtkWidget *tree_lessons;
     gchar *text = NULL;

     box_step = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_step), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_step), 0);
    
     text = g_strdup_printf (WIZ_STEP,
                             _("Select the word amount to learn from each "
                               "lesson.\n\n"
                               "<b>Amount</b> is the amount of meanings to "
                               "choose from the lesson.\n"
                               "<b>Minumum ID</b> and <b>Maximum ID</b> "
                               "define the range to choose meanings from."
                               "\n\n"),
                             WIZARD_TEXT_NEXT);
     lbl_step = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_step), TRUE);
     gtk_label_set_markup (GTK_LABEL (lbl_step), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_step), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_step), lbl_step, FALSE, FALSE, 0);
     g_free (text);

     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_box_pack_start (GTK_BOX (box_step), scr_win, TRUE, TRUE, 0);

     tree_lessons = create_lesson_param_tree (session);
     gtk_container_add (GTK_CONTAINER (scr_win), tree_lessons);

     return box_step;
}

static GtkWidget*
create_page_3 (learnSession *session)
{
     GtkWidget *box_step;
     GtkWidget *lbl_step;
     GtkWidget *win_lang;
     gchar *text = NULL;
     learnLanguage *tmp = settings->learn_lang;

     const gchar *titles[3] = { N_("Use"), N_("Learn from"), N_("Language") };

     box_step = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_step), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_step), 0);

     text = g_strdup_printf (WIZ_STEP,
                             _("Select the languages you want to learn.\n\n"),
                             WIZARD_TEXT_NEXT);
     lbl_step = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_step), TRUE);
     gtk_label_set_markup (GTK_LABEL (lbl_step), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_step), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_step), lbl_step, FALSE, FALSE, 0);
     g_free (text);

     /* copy languages */
     if (!session->language)
          while (tmp)
          {
               session->language =
                    learn_language_append_language (session->language,
                                                    tmp->language, tmp->used,
                                                    tmp->main);
               tmp = tmp->next;
          }

     /* create language list view */
     win_lang = util_lang_create_language_tree_view (settings,
                                                     session->language,
                                                     titles);
     gtk_container_set_border_width (GTK_CONTAINER (win_lang), 0);

     gtk_box_pack_start (GTK_BOX (box_step), win_lang, TRUE, TRUE, 0);

     return box_step;
}

static GtkWidget*
create_page_4 (learnSession *session)
{
     GtkWidget *box_step;
     GtkWidget *lbl_step;
     GtkWidget *scr_win;
     GtkWidget *viewport;
     GtkWidget *tbl_lessons;
     GtkWidget *lbl_lesson;
     GtkWidget *lbl_word;
     GtkWidget *btn_help;
     guint cnt_lesson = 0;
     learnLesson *tmp = session->lessons;

     gchar *text = NULL;
     gchar *words = NULL;

     box_step = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_step), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_step), 0);

     text = g_strdup_printf (WIZ_STEP,
                             _("Please revise your settings. Click "
                               "\"Finish\", if your wish to use those "
                               "learning setting or use the \"Back\" button "
                               "to go back to a prior step and adjust your "
                               "settings."), "");
     lbl_step = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_step), TRUE);
     gtk_label_set_markup (GTK_LABEL (lbl_step), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_step), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_step), lbl_step, FALSE, FALSE, 0);
     g_free (text);

     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr_win),
                                          GTK_SHADOW_NONE);
     gtk_box_pack_start (GTK_BOX (box_step), scr_win, TRUE, TRUE, 0);

     /* count the lessons */
     while (tmp)
     {
          cnt_lesson++;
          tmp = tmp->next;
     }

     tbl_lessons = hig_table_new (cnt_lesson, 3);
     gtk_container_set_border_width (GTK_CONTAINER (tbl_lessons), 0);
     
     tmp = session->lessons;
     while (tmp)
     {
          text = g_strdup_printf ("%s \"%s\":", _("Lesson"),
                                  tmp->lesson->type);
          lbl_lesson = gtk_label_new (NULL);
          gtk_label_set_markup (GTK_LABEL (lbl_lesson), text);
          gtk_misc_set_alignment (GTK_MISC (lbl_lesson), 0, 0.5);
          gtk_table_attach (GTK_TABLE (tbl_lessons), lbl_lesson,
                            0, 1, cnt_lesson, cnt_lesson + 1,
                            GTK_FILL, GTK_FILL, 0, 0);
          g_free (text);

          words = g_strdup_printf ("<b>%i</b> %s <b>\"%s\"</b> %s",
                                   tmp->amount, _("words and sentences. Use"),
                                   methods[tmp->access], _("access."));
          text = g_strdup_printf (MARKUP_ITALIC, words);
          lbl_word = gtk_label_new (NULL);
          gtk_label_set_markup (GTK_LABEL (lbl_word), text);
          gtk_misc_set_alignment (GTK_MISC (lbl_word), 0, 0.5);
          gtk_table_attach (GTK_TABLE (tbl_lessons), lbl_word,
                            1, 2, cnt_lesson, cnt_lesson + 1,
                            GTK_FILL, GTK_FILL, 0, 0);
          g_free (text);
          g_free (words);

          cnt_lesson++;
          tmp = tmp->next;
     }

     viewport = gtk_viewport_new (NULL, NULL);
     gtk_viewport_set_shadow_type (GTK_VIEWPORT (viewport), GTK_SHADOW_NONE);
     gtk_container_add (GTK_CONTAINER (viewport), tbl_lessons);
     gtk_container_add (GTK_CONTAINER (scr_win), viewport);

     btn_help = gtk_check_button_new_with_mnemonic
                  (_("_Display additional descriptions and images"));
     session->hints = settings->learn_hints;
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_help),
                                   session->hints);
     g_signal_connect (G_OBJECT (btn_help), "toggled",
                       G_CALLBACK (util_copy_boolean), &session->hints);
     gtk_box_pack_end (GTK_BOX (box_step), btn_help, FALSE, FALSE, 0);
     return box_step;
}

static gboolean
check_session_1 (learnSession *session)
{
     gboolean ok = FALSE;
     if (session)
     {
          if (session->lessons)
               ok = TRUE;
          else
               ok = FALSE;
     }

     if (!ok)
          util_info (_("You must at least add and use one lesson!"));
     return ok;
}

static void 
switch_lesson (GtkTreeView *tree, GtkWidget *label)
{
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     GtkTreeSelection *select = gtk_tree_view_get_selection (tree);
     GtkTreeIter iter;
     lingLesson *tmp = settings->prefs->lessons;
     gchar *name = NULL;
     gchar *text = NULL;
     gchar *meaning = NULL;

     /* TODO A more detailled description should be added to the lessons */
     if (gtk_tree_selection_get_selected (select, &model, &iter))
     {
          gtk_tree_model_get (model, &iter, LESSON_NAME, &name, -1);

          while (tmp && g_utf8_collate (name, tmp->type) != 0)
               tmp = tmp->next;
          
          if (!tmp) /* none found, this should not happen */
          {
               g_free (name);
               return;
          }
          meaning = g_strdup_printf (_("This lesson contains about %i words "
                                       "and sentences."),
                                     ling_lesson_get_meaning_amount (tmp));

          text = g_strdup_printf (LESSON_DESC, _("Lesson"), name, meaning);
          gtk_label_set_markup (GTK_LABEL (label), text);
          
          g_free (meaning);
          g_free (text);
          g_free (name);
     }
     return;
}

static void
change_amount (GtkCellRendererSpin *cell, const gchar *path,
               const gchar *value, GtkListStore *list)
{
     GtkTreeIter iter;
     gchar *name = NULL;
     learnSession *session = g_object_get_data (G_OBJECT (list), SESSION);
     learnLesson *tmp = session->lessons;
     guint col_no = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (cell),
                                                         COLUMNNO));
     guint val = 0;

     if (gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (list), &iter,
                                              path))
     {
          gtk_tree_model_get (GTK_TREE_MODEL (list), &iter, L_NAME, &name, -1);

          while (tmp && xmlStrncasecmp (tmp->lesson->type, name,
                                        xmlStrlen (tmp->lesson->type)) != 0)
               tmp = tmp->next;
          g_free (name);

          if (!tmp) /* this should not happen */
               return;

          val = (guint) (atoi (value));
          gtk_list_store_set (list, &iter, col_no, val, -1);

          switch (col_no)
          {
          case L_WORDS:
               tmp->amount = val;
               break;
          case L_MIN:
               tmp->start = val;
               break;
          case L_MAX:
               tmp->end = val;
               break;
          }
     }
     return;
}

static void
switch_access (GtkCellRendererText *cell, const gchar *path,
               const gchar *value, GtkListStore *list)
{
     GtkTreeIter iter;
     guint i = 0;
     gchar *name = NULL;
     learnSession *session = g_object_get_data (G_OBJECT (list), SESSION);
     learnLesson *tmp = session->lessons;

     if (gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (list), &iter,
                                              path))
     {
          while (i < G_N_ELEMENTS (methods))
          {
               if (g_utf8_collate (value, methods[i]) == 0)
               {
                    gtk_tree_model_get (GTK_TREE_MODEL (list), &iter,
                                        L_NAME, &name, -1);
                    while (tmp)
                    {
                         if (xmlStrncasecmp
                             (tmp->lesson->type, name,
                              xmlStrlen (tmp->lesson->type)) == 0)
                         {
                              tmp->access = i;
                              break;
                         }
                         tmp = tmp->next;
                    }
                    gtk_list_store_set (list, &iter, L_ACCESS, methods[i], -1);
                    g_free (name);
               }
               i++;
          }
     }    
     return;
}

static void 
toggle_use_lesson (GtkCellRendererToggle *cell, gchar *path, GtkTreeView *tree)
{
     gboolean use = FALSE;
     gchar *name = NULL;
     guint amount = 0;
     guint total = 0;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     lingLesson *tmp = settings->prefs->lessons;
     learnSession *session = g_object_get_data (G_OBJECT (tree), SESSION);
     learnSession *tmp_session = session;

     debug ("Modifying lesson list of session...\n");

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          gtk_tree_model_get (model, &iter, 
                              LESSON_USED, &use,
                              LESSON_NAME, &name,
                              -1);
          while (tmp && g_utf8_collate (name, tmp->type) != 0)
               tmp = tmp->next;
          g_free (name);
          
          if (!tmp) /* TODO: this should never happen */
               return;

          /* toggle */
          if (ling_lesson_get_meaning_amount (tmp) <= 1)
          {
               util_info (_("This lesson contains less than two meanings, "
                            "which does not make much sense to learn from "
                            "and thus will not be selectable. Sorry."));
               return;
          }

          use ^= 1;
          gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                              LESSON_USED, use, -1);

          /* add or remove lesson in session */
          if (use)
          {
               debug ("Lesson \"%s\" will be used now\n", tmp->type);

               /* check amount */
               total = ling_lesson_get_meaning_amount (tmp);
               if (settings->learn_words > total)
                    amount = total;
               else
                    amount = settings->learn_words;

               tmp_session->lessons =
                    learn_lesson_append_lesson
                        (session->lessons, tmp, amount, 1,
                         (guint) ling_lesson_get_meaning_amount (tmp),
                         settings->learn_method);
               if (!tmp_session)
               {
                    error_warning (_("Lesson could not be added!"),
                                   ERR_NOT_AVAILABLE, ERR_NOT_AVAILABLE);
                    return;
               }
               session = tmp_session;
          }
          else
          {
               debug ("Lesson \"%s\" will NOT be used anymore\n", tmp->type);
               session->lessons = learn_lesson_remove_lesson (session->lessons,
                                                              tmp);
          }
     }
     return;
}

static void
refresh_list (GtkTreeView *tree, learnSession *session)
{
     GtkListStore *list;
     GtkTreeIter iter;
     lingLesson *lessons = settings->prefs->lessons;
     learnLesson *tmp = NULL;
     gboolean used = FALSE;

     list = gtk_list_store_new (LESSON_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING);
     while (lessons)
     {
          /* check the learnLessons for an entry */
          used = FALSE;
          tmp = session->lessons;
          while (tmp)
          {
               if (tmp->lesson == lessons)
                    used = TRUE;
               tmp = tmp->next;
          }
          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              LESSON_USED, used,
                              LESSON_NAME, lessons->type, -1);
          lessons = lessons->next;
     }
     gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
     g_object_unref (G_OBJECT (list));
     return;
}

learnSession*
gtklearn_wiz_run (void)
{
     learnSession *session = NULL;
     learnLesson *tmp = NULL;
     learnEntry *entry = NULL;
     learnEntry *tmp_entry = NULL;
     GtkWidget *wizard;

     session = learn_session_new ();
     if (!session)
     {
          error_warning (ERR_MEM_INFO, _("The wizard could not be created."),
			 ERR_NOT_AVAILABLE);
          return NULL;
     }

     wizard = wizard_create_new (_("Create a new learning session"),
                                 _("The following wizard will guide you "
                                   "through the process of creating a new "
                                   "learning session.\n"
                                   "Please read the following instructions "
                                   "carefully. If you think, you have made "
                                   "wrong input, you can go back at any time "
                                   "to a previous step and change your "
                                   "input."));
     if (!wizard)
     {
          error_warning (ERR_MEM_INFO, _("The wizard could not be created."),
                         ERR_NOT_AVAILABLE);
          learn_session_free (session);
          return NULL;
     }

     wizard_append_page_with_callbacks (wizard, _("Select your lessons"),
                                        (WizPageCallback) create_page_1,
                                        session,
                                        (WizardCallback) check_session_1,
                                        session);
     wizard_append_page (wizard, _("Set the amount to learn"),
                         (WizPageCallback) create_page_2, session);
     wizard_append_page (wizard, _("Choose the languages"),
                         (WizPageCallback) create_page_3, session);
     wizard_append_page (wizard, _("Evaluate your settings"),
                         (WizPageCallback) create_page_4, session);

     switch (wizard_run (wizard))
     {
     case GTK_RESPONSE_OK:
          debug ("Wizard finished! Creating learning session...\n");
          tmp = session->lessons;
          while (tmp)
          {
               debug ("Creating entries for %s\n", tmp->lesson->type);
               
               if (entry) /* link the lesson entries manually */
               {
                    tmp_entry = entry;
                    while (tmp_entry->next)
                         tmp_entry = tmp_entry->next;
                    tmp_entry->next = learn_entry_create_from_lesson (tmp);
                    tmp_entry->next->prev = tmp_entry;
               }
               else
                    entry = learn_entry_create_from_lesson (tmp);
               tmp = tmp->next;
          }
          session->entry = entry;
          session->cur_entry = entry;
          break;
     default:
          debug ("Wizard aborted!\n");
          learn_session_free (session);
          session = NULL;
          break;
     }
     return session;
}
