/* $Id: lingoteach.c,v 1.14 2005/05/10 17:18:39 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2001-2005 The LingoTeach Team. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include <stdio.h>
#include <getopt.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef ENABLE_NLS
#include <locale.h>
#endif

#include "li18n.h"
#include "defs.h"
#include "errors.h"
#include "lesson.h"
#include "cfg.h"
#ifdef WITHGTK
#include "gtk/gui.h"
#endif

/* globally used variables */
lingoteachSettings *preferences = NULL;

/* prototypes */
void quit (int status);
static void usage (void);
static void parse_args (int argc, char *argv[]);
static void init_config (void);

static void
usage (void)
{
     printf (_("usage: lingoteach [option] ... [-f file] ...\n"));
     printf (_("Options and arguments:\n"));
     printf (_("-h, --help     : Show this help and exit.\n"));
     printf (_("-v, --version  : Show the LingoTeach version and exit.\n"));
     printf (_("-n, --noconf   : Do not load the user config at start.\n"));
     printf (_("-f, --file arg : Load lesson file.\n"));
     printf (_("-l, --lang arg : Load specific language definitions.\n"));
     
     quit (EXIT_SUCCESS);
}

static void
parse_args (int argc, char *argv[])
{
     char *lessons[MAXFILES] = { NULL };
     int file_count = 0;
     int i = 0;
     int c;

     static struct option opts[] =
         {
               { "help", no_argument, NULL, 'h' },
               { "version", no_argument, NULL, 'v' },
               { "noconf", no_argument, NULL, 'n' },
               { "lang", required_argument, NULL, 'l' },
               { "file", required_argument, NULL, 'f' },
               { NULL, 0, NULL, 0},
          };
    
     while ((c = getopt_long (argc, argv, "hvnl:f:", opts, NULL)) != -1)
     {
          switch (c)
          {
          case 'v':
               printf ("%s %s - %s\n", NAME, VERSION, INTVERSION);
               printf ("%s\n", COPYRIGHT);
               quit (EXIT_SUCCESS);
               break;
         
          case 'n':
               debug ("Configuration file will be skipped\n");
               preferences->skip = TRUE;
               break;

          case 'l':
               if (!preferences->config && !conf_load_languages (preferences,
                                                                 optarg))
                    quit (EXIT_FAILURE);
               break;
              
          case 'f':
               if (file_count > MAXFILES)
               {
                    printf (_("Warning: Only %d files can be loaded at "
                              "startup.\n"), MAXFILES);
                    printf (_("File %s will be ignored.\n"), optarg);
                    continue;
               }
               lessons[file_count] = optarg;
               file_count++;
               break;
             
          case 'h':
          case '?':
          default:
               usage ();
          }
     }
    
     /* load the lesson files, if any */
     if (!preferences->config && !CONF_LOAD_DEFAULT_LANGUAGES (preferences))
          quit (EXIT_FAILURE);

     for (i = 0; i < file_count; i++)
     {
          if (lesson_lesson_in_list (preferences->lessons, lessons[i]))
               continue; /* TODO: should an error be reported? */
          if (!lesson_load_lesson_to_prefs (preferences, lessons[i]))
               printf (_("Error: Lesson %s could not be loaded.\n"),
                       lessons[i]);
     }
     return;
}

static void
init_config (void)
{
     size_t size = 0;
     char *home = getenv (HOME);

     if (!home)
     {
          error_warning (_("Missing environment setting. Is $HOME set?"),
                         _("The HOME environment variable could not be "
			   "found. Settings will not be saved in this "
                           "session."),
                         _("Please make sure that you set up your "
			   "environment properly and verify that you set "
                           "your HOME variable to point to your user "
                           "directory."));
          return;
     }
    
     /* build path */
     size = strlen (home) + strlen (CONFDIR);
     preferences->conf_dir = malloc (size + 1);
     if (!preferences->conf_dir)
     {
          error_critical (ERR_MEM_INFO, ERR_MEM_DESC, ERR_NOT_AVAILABLE);
          return;
     }
     snprintf (preferences->conf_dir, size + 1, "%s%s", home, CONFDIR);
     
     /* create configuration dir */
     if (access (preferences->conf_dir, F_OK) != 0)
     {
          if (mkdir (preferences->conf_dir, S_IRUSR | S_IWUSR | S_IXUSR) != 0)
          {
               error_critical (_("Configuration directory can not be "
                                 "created!"),
                               _("The configuration directory can not be "
                                 "created. Settings will not be saved in this "
                                 "session."),
                               _("Make sure that you have proper access "
                                 "rights to your home directory and that you "
                                 "are able to create new directories and "
                                 "files as well. Also check, if there is "
                                 "enough quota left.")
                    );
               return;
          }
          preferences->predef = TRUE;
     }
         
     /* set configuration directory and file */
     size = strlen (preferences->conf_dir) + strlen (CONFFILE);
     preferences->conf_file = malloc (size + 1);
     if (!preferences->conf_file)
     {
          error_critical (ERR_MEM_INFO, ERR_MEM_DESC, ERR_NOT_AVAILABLE);
          return;
     }
     snprintf (preferences->conf_file, size + 1, "%s%s", preferences->conf_dir,
               CONFFILE);

     if (access (preferences->conf_file, F_OK) != 0)
          preferences->old_conf = TRUE;

     /* load the configuration */
     if (!preferences->skip)
     {
          if (preferences->predef)
               conf_load_defaults (preferences);
          else if (preferences->old_conf)
               conf_load_old_conf (preferences);
          else
               conf_load_config (preferences);
     }

     return;
}

int
main (int argc, char *argv[])
{
     /* catch critical signals for analysis */
     signal (SIGSEGV, (void *) error_sigsegv);
     signal (SIGQUIT, (void *) error_sigquit);

#ifdef ENABLE_NLS
     setlocale (LC_ALL, "");
     bindtextdomain (PACKAGE, LOCALEDIR);
     textdomain (PACKAGE);
     bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif /* ENABLE_NLS */

     /* init anything */
     preferences = conf_init_new ();
     if (!preferences)
     {
          error_critical (ERR_MEM_INFO, ERR_MEM_DESC, ERR_NOT_AVAILABLE);
          exit (EXIT_FAILURE);
     }

     if (argc > 1)
          parse_args (argc, argv);

     init_config ();

#ifdef WITHGTK
     gui_create_gui (argc, argv, preferences);
#endif /* WITHGTK */
     
     /* what should be done, if there is no gui? maybe we should integrate
      * a standards compliant(?) text interface...
      */
     
     return 0;
}

void
quit (int status)
{
     /* clean anything up, etc.pp. */
     conf_free_config (preferences);
     exit (status);
}
