/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details. 
 * 
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <lingoteach.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach-i18n.h"
#include "errors.h"
#include "gui.h"
#include "module.h"
#include "util.h"
#include "learningpref.h"

/* some global definitions */
#define METHODS 4
#define PREF_MAX_LEN 1000
#define CONFIG "/.lingoteach/config"

enum  /* enumeration for notebook pages */
  {
    P_SYSTEM,
    P_GENERAL,
    P_LESSON,
    P_MODULE
  };

extern int err;              /*error flag for util.c */
extern lingConfig *settings; /* global settings */

struct lpreferences genLprefs; /* general preferences for lingoteach */
lingConfig *settings = NULL;   /* global settings */
lingLesson *lessons = NULL;    /* global lesson list */

/* methods for access */
static gchar* selection[METHODS] =        
  {
    N_("Random"), N_("Sequence"), N_("Review"), N_("Learn")
  };

static GtkWidget *box_module_set;         /* widget, which can contain the 
					     module content */
static GtkListStore *lst_lessons = NULL;  /* global list for the lessons */
static GtkListStore *lst_modules = NULL;  /* global list for the modules */
static struct                             /* preference widget struct */
{
  GtkWidget *txt_soundpath;
  GtkWidget *txt_browser;
  GtkWidget *txt_player;
} prefWidgets; 


/* prototypes */
GtkWidget* create_dlg_lpref (void);

void on_btn_fileselect_clicked (gpointer user_data);
void on_notebook_lpref_switch_page (GtkNotebook *notebook,
				    GtkNotebookPage *page,
				    guint page_num,
                                    gpointer user_data);
void on_opt_method_changed (GtkOptionMenu *optionmenu);
void on_opt_mainlang_changed (GtkOptionMenu *optionmenu);
void on_btn_lpref_apply_clicked (gpointer user_data);
void on_used_cell_toggled (GtkCellRendererToggle *cell, gchar *path_str, 
			   gpointer data);
void on_btn_add_lesson_clicked (void);
void lpref_add_lesson_file (gchar *filename);
void on_btn_rem_lesson_clicked (GtkTreeView *tree);
void on_btn_add_module_clicked (void);
void on_btn_rem_module_clicked (GtkTreeView *tree);
void lpref_add_module (gchar* filename);
void on_tree_list_modules_row_changed (GtkTreeView *tree);
void lpref_clear_module_box (GtkWidget *module_box);
void lpref_module_set_content (GtkTreeView *tree);


GtkWidget* lpref_get_languages (void);
GtkListStore* get_list_lessons (void);
GtkListStore* get_lpref_modules (void);

/* used for preferences */
void lpref_set_soundpath (gchar *name);
void lpref_set_browser (gchar *name);
void lpref_set_mlanguage (gchar *name);
void lpref_set_player (gchar *name);
void write_config (void);
gint get_lang_index (gchar *language);



/*********************
 * private functions *
 *********************/

/* 
 * preference dialog 
 */
GtkWidget*
create_dlg_lpref (void)
{
  GtkWidget *dlg_lpref;
  GtkWidget *box_lpref_dlg;
  GtkWidget *notebook_lpref;
  GtkWidget *tbl_notebook;

  GtkWidget *lbl_soundpath;
  GtkWidget *btn_fileselect1;

  GtkWidget *sep_lpref1;
  GtkWidget *sep_lpref2;

  GtkWidget *lbl_browser;
  GtkWidget *btn_fileselect2;

  GtkWidget *lbl_player;
  GtkWidget *btn_fileselect3;
  
  GtkWidget *lbl_system;
  GtkWidget *lbl_general;

  GtkWidget *tbl_gen_pref;
  
  GtkWidget *lbl_mainlang;
  GtkWidget *opt_mainlang;
  GtkWidget *menu_mainlang;

  GtkWidget *opt_method;
  GtkWidget *menu_method;
  GtkWidget *lbl_method;

  GtkWidget *tbl_lang;

  GtkWidget *scr_win_lessons;
  GtkWidget *tree_list_lessons;
  GtkWidget *lbl_lessons;

  GtkWidget *box_lpref_mod;
  GtkWidget *scr_win_mod;
  GtkWidget *tree_list_modules;
  GtkWidget *scr_win_mod_con;
  GtkWidget *lbl_modules;

  GtkWidget *btn_box_lpref;
  GtkWidget *btn_lpref_cancel;
  GtkWidget *btn_lpref_apply;

  GtkTreeViewColumn *column;
  GtkCellRenderer   *renderer;

  lingchar **lang;
  int        i;

  /* the dialog itself */
  dlg_lpref = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg_lpref), _("Settings"));
  gtk_window_set_position (GTK_WINDOW (dlg_lpref), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW (dlg_lpref), TRUE);
  gtk_window_set_resizable (GTK_WINDOW (dlg_lpref), FALSE);
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dlg_lpref), TRUE);
  /* main box of dialog */
  box_lpref_dlg = GTK_DIALOG (dlg_lpref)->vbox;
  gtk_widget_show (box_lpref_dlg);

  /* the notebook */
  notebook_lpref = gtk_notebook_new ();
  gtk_widget_show (notebook_lpref);
  gtk_box_pack_start (GTK_BOX (box_lpref_dlg), notebook_lpref, TRUE, TRUE, 0);

  /* table for system stuff */
  tbl_notebook = gtk_table_new (6, 3, FALSE);
  gtk_widget_show (tbl_notebook);
  gtk_container_add (GTK_CONTAINER (notebook_lpref), tbl_notebook);
  gtk_container_set_border_width (GTK_CONTAINER (tbl_notebook), 4);
  gtk_table_set_row_spacings (GTK_TABLE (tbl_notebook), 2);
  gtk_table_set_col_spacings (GTK_TABLE (tbl_notebook), 2);

  /* soundpath */
  lbl_soundpath = gtk_label_new (_("Sound Path"));
  gtk_widget_show (lbl_soundpath);
  gtk_table_attach (GTK_TABLE (tbl_notebook), lbl_soundpath, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (lbl_soundpath), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (lbl_soundpath), 0, 0.5);

  prefWidgets.txt_soundpath = gtk_entry_new ();
  gtk_widget_set_name (prefWidgets.txt_soundpath, "txt_soundpath");
  gtk_widget_show (prefWidgets.txt_soundpath);
  gtk_table_attach (GTK_TABLE (tbl_notebook), 
		    prefWidgets.txt_soundpath, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  if (genLprefs.soundpath != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_soundpath), 
			genLprefs.soundpath);

  btn_fileselect1 = gtk_button_new_with_label ("...");
  gtk_widget_show (btn_fileselect1);
  gtk_table_attach (GTK_TABLE (tbl_notebook), btn_fileselect1, 2, 3, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  /* seperator */
  sep_lpref1 = gtk_hseparator_new ();
  gtk_widget_show (sep_lpref1);
  gtk_table_attach (GTK_TABLE (tbl_notebook), sep_lpref1, 0, 3, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 1, 4);

  /* web browser */
  lbl_browser = gtk_label_new (_("Web Browser"));
  gtk_widget_show (lbl_browser);
  gtk_table_attach (GTK_TABLE (tbl_notebook), lbl_browser, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (lbl_browser), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (lbl_browser), 0, 0.5);

  prefWidgets.txt_browser = gtk_entry_new ();
  gtk_widget_set_name (prefWidgets.txt_browser, "txt_browser");
  gtk_widget_show (prefWidgets.txt_browser);
  gtk_table_attach (GTK_TABLE (tbl_notebook), 
		    prefWidgets.txt_browser, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  if (genLprefs.browser != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_browser), 
			genLprefs.browser);

  btn_fileselect2 = gtk_button_new_with_label ("...");
  gtk_widget_show (btn_fileselect2);
  gtk_table_attach (GTK_TABLE (tbl_notebook), btn_fileselect2, 2, 3, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  /* sound player */
  lbl_player = gtk_label_new (_("Sound player"));
  gtk_widget_show (lbl_player);
  gtk_table_attach (GTK_TABLE (tbl_notebook), lbl_player, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (lbl_browser), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (lbl_browser), 0, 0.5);

  prefWidgets.txt_player = gtk_entry_new ();
  gtk_widget_set_name (prefWidgets.txt_player, "txt_player");
  gtk_widget_show (prefWidgets.txt_player);
  gtk_table_attach (GTK_TABLE (tbl_notebook), 
		    prefWidgets.txt_player, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  if (genLprefs.player != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_player), genLprefs.player);

  btn_fileselect3 = gtk_button_new_with_label ("...");
  gtk_widget_show (btn_fileselect3);
  gtk_table_attach (GTK_TABLE (tbl_notebook), btn_fileselect3, 2, 3, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  /* system settings label */
  lbl_system = gtk_label_new (_("System settings"));
  gtk_widget_show (lbl_system);
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_lpref), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_lpref), 0), 
     lbl_system);
  gtk_label_set_justify (GTK_LABEL (lbl_system), GTK_JUSTIFY_LEFT);

  /* general preferences */
  tbl_gen_pref = gtk_table_new (4, 2, FALSE);
  gtk_widget_show (tbl_gen_pref);
  gtk_container_add (GTK_CONTAINER (notebook_lpref), tbl_gen_pref);
  gtk_container_set_border_width (GTK_CONTAINER (tbl_gen_pref), 4);
  gtk_table_set_row_spacings (GTK_TABLE (tbl_gen_pref), 2);
  gtk_table_set_col_spacings (GTK_TABLE (tbl_gen_pref), 2);

  /* method selection */
  opt_method = gtk_option_menu_new ();
  gtk_widget_show (opt_method);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), opt_method, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  menu_method = util_get_menu ((lingchar **) selection, METHODS);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_method),menu_method);
  
  lbl_method = gtk_label_new (_("Learning Method"));
  gtk_widget_show (lbl_method);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), lbl_method, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (lbl_method), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (lbl_method), 0, 0.5);

  /* separator */
  sep_lpref2 = gtk_hseparator_new ();
  gtk_widget_show (sep_lpref2);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), sep_lpref2, 0, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 1, 4);

  /* main language */
  lbl_mainlang = gtk_label_new (_("Main Language"));
  gtk_widget_show (lbl_mainlang);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), lbl_mainlang, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (lbl_mainlang), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (lbl_mainlang), 0, 0.5);

  opt_mainlang = gtk_option_menu_new ();
  gtk_widget_show (opt_mainlang);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), opt_mainlang, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  /* create language selection */
  lang = ling_lang_get_languages (settings);
  i =  ling_lang_get_max_languages (settings);
  if (i < 0)
    {
      err_error_cb (_("Language file seems to be corrupted!\nExiting.."));
      lingoteach_quit ();
    }
  menu_mainlang = util_get_menu (lang, i);
  if (genLprefs.mlanguage != NULL)
    gtk_menu_set_active (GTK_MENU (menu_mainlang), 
			 get_lang_index (genLprefs.mlanguage));
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_mainlang), menu_mainlang); 
  free (lang);
  
  /* languages */
  tbl_lang = lpref_get_languages ();
  gtk_widget_show (tbl_lang);
  gtk_table_attach (GTK_TABLE (tbl_gen_pref), tbl_lang, 0, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  /* tab label */
  lbl_general = gtk_label_new (_("General"));
  gtk_widget_show (lbl_general);
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_lpref), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_lpref), 1), 
     lbl_general);
  gtk_label_set_justify (GTK_LABEL (lbl_general), GTK_JUSTIFY_LEFT);

  /* scrolled window for lesson selection treeview */
  scr_win_lessons = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scr_win_lessons);
  gtk_container_add (GTK_CONTAINER (notebook_lpref), scr_win_lessons);
  gtk_container_set_border_width (GTK_CONTAINER (scr_win_lessons), 2);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win_lessons), 
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);

  /* the tree itself */
  if (lst_lessons == NULL)
    lst_lessons = gtk_list_store_new (NUMBER, 
				      G_TYPE_STRING, 
				      G_TYPE_STRING, 
				      G_TYPE_BOOLEAN);
  tree_list_lessons = 
    gtk_tree_view_new_with_model (GTK_TREE_MODEL (lst_lessons));
  gtk_widget_show (tree_list_lessons);
  gtk_container_add (GTK_CONTAINER (scr_win_lessons), tree_list_lessons);
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tree_list_lessons), FALSE);

  /* columns */
  renderer = gtk_cell_renderer_toggle_new ();
  g_signal_connect (G_OBJECT (renderer), "toggled",
		    G_CALLBACK (on_used_cell_toggled), 
		    (gpointer) GTK_TREE_MODEL (lst_lessons));
  
  column = gtk_tree_view_column_new_with_attributes (_("Used"),
						     renderer,
						     "active",
						     CURR_USED,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_list_lessons), column);
  
  /* the lesson name */
  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (_("Lesson Name"),
						       renderer,
						       "text",
						       LESSON_NAME,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_list_lessons), column);
  
  /* the lesson file */
  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (_("Lesson File"),
						       renderer,
						       "text",
						       LESSON_PATH,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_list_lessons), column);

  /* tab label */
  lbl_lessons = gtk_label_new (_("Lesson Selection"));
  gtk_widget_show (lbl_lessons);
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_lpref), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_lpref), 2), 
     lbl_lessons);
  gtk_label_set_justify (GTK_LABEL (lbl_lessons), GTK_JUSTIFY_LEFT);

  /* modules */
  box_lpref_mod = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (box_lpref_mod);
  gtk_container_add (GTK_CONTAINER (notebook_lpref), box_lpref_mod);

  /* scrolled window for module selection */
  scr_win_mod = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scr_win_mod);
  gtk_box_pack_start (GTK_BOX (box_lpref_mod), scr_win_mod, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win_mod), 
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);

  /* the tree for the modules */
  lst_modules = get_lpref_modules ();
  tree_list_modules = 
    gtk_tree_view_new_with_model (GTK_TREE_MODEL (lst_modules));
  gtk_widget_show (tree_list_modules);
  gtk_container_add (GTK_CONTAINER (scr_win_mod), tree_list_modules);

  /* the module row */
  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (_("Module"),
						       renderer,
						       "text",
						       0,
						       NULL);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				   GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 100);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_list_modules),
			       column);

  /* box for module content */
  box_module_set = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box_module_set);
  
  /* view for module content */
  scr_win_mod_con = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scr_win_mod_con);
  gtk_box_pack_start (GTK_BOX (box_lpref_mod), scr_win_mod_con, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win_mod_con), 
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scr_win_mod_con),
					 box_module_set);

  /* tab label */
  lbl_modules = gtk_label_new (_("Module Selection"));
  gtk_widget_show (lbl_modules);
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_lpref), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_lpref), 3), 
     lbl_modules);
  gtk_label_set_justify (GTK_LABEL (lbl_modules), GTK_JUSTIFY_LEFT);

  /* buttons for the dialog */
  btn_box_lpref = GTK_DIALOG (dlg_lpref)->action_area;
  gtk_widget_show (btn_box_lpref);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (btn_box_lpref), 
			     GTK_BUTTONBOX_END);

  btn_lpref_cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
  gtk_widget_show (btn_lpref_cancel);
  gtk_dialog_add_action_widget (GTK_DIALOG (dlg_lpref), btn_lpref_cancel, 
				GTK_RESPONSE_CANCEL);
  GTK_WIDGET_SET_FLAGS (btn_lpref_cancel, GTK_CAN_DEFAULT);

  btn_lpref_apply = gtk_button_new_from_stock (GTK_STOCK_APPLY);
  gtk_widget_show (btn_lpref_apply);
  gtk_dialog_add_action_widget (GTK_DIALOG (dlg_lpref), btn_lpref_apply, 
				GTK_RESPONSE_APPLY);
  GTK_WIDGET_SET_FLAGS (btn_lpref_apply, GTK_CAN_DEFAULT);

  /* signal connectors */
  g_signal_connect_swapped (G_OBJECT (btn_fileselect1), "clicked",
			    G_CALLBACK (on_btn_fileselect_clicked),
			    (gpointer) prefWidgets.txt_soundpath);

  g_signal_connect_swapped (G_OBJECT (btn_fileselect2), "clicked",
			    G_CALLBACK (on_btn_fileselect_clicked),
			    (gpointer) prefWidgets.txt_browser);

  g_signal_connect_swapped (G_OBJECT (btn_fileselect3), "clicked",
			    G_CALLBACK (on_btn_fileselect_clicked),
			    (gpointer) prefWidgets.txt_player);

  g_signal_connect (G_OBJECT (opt_method), "changed",
		    G_CALLBACK (on_opt_method_changed),
		    NULL);

  g_signal_connect (G_OBJECT (opt_mainlang), "changed",
		    G_CALLBACK (on_opt_mainlang_changed),
		    NULL);

  g_signal_connect (G_OBJECT (btn_lpref_cancel), "clicked",
		    G_CALLBACK (gtk_widget_destroy),
		    NULL);

  g_signal_connect_swapped (G_OBJECT (btn_lpref_apply), "clicked",
			    G_CALLBACK (on_btn_lpref_apply_clicked),
			    (gpointer) notebook_lpref);

  g_signal_connect (G_OBJECT (tree_list_modules), "row-activated", 
		    G_CALLBACK (on_tree_list_modules_row_changed), 
		    (gpointer) box_module_set);

  g_signal_connect_after (G_OBJECT (notebook_lpref), "switch-page",
			  G_CALLBACK (on_notebook_lpref_switch_page),
			  (gpointer) btn_box_lpref);
  
  /* Store pointers to all necessary widgets, for use by lookup_widget(). */
  GLADE_HOOKUP_OBJECT (dlg_lpref, tree_list_lessons, "tree_list_lessons"); 
  GLADE_HOOKUP_OBJECT (dlg_lpref, tree_list_modules, "tree_list_modules");
  GLADE_HOOKUP_OBJECT (dlg_lpref, tbl_lang, "tbl_lang");   

  return dlg_lpref;
}

/*
 * file selection 
 */
void
on_btn_fileselect_clicked (gpointer user_data)
{
  GtkWidget *txt_entry = GTK_WIDGET (user_data);
  GtkWidget *file_select = NULL;
  const gchar *name = gtk_widget_get_name (txt_entry);

  if (g_ascii_strcasecmp (name, "txt_soundpath") == 0)
    file_select = util_create_dlg_fileselect (_("Soundpath"), 
					      lpref_set_soundpath);

  if (g_ascii_strcasecmp (name, "txt_browser") == 0)
    file_select = util_create_dlg_fileselect (_("Web Browser"), 
					      lpref_set_browser);

  if (g_ascii_strcasecmp (name, "txt_player") == 0)
    file_select = util_create_dlg_fileselect (_("Sound player"), 
					      lpref_set_player);
  
  if (file_select)
    gtk_widget_show (file_select);

  return;
}

/* 
 * additional buttons and callbacks on page switching 
 */
void on_notebook_lpref_switch_page (GtkNotebook *notebook,
				    GtkNotebookPage *page,
				    guint page_num,
				    gpointer user_data)
{
  GtkWidget *btn_add = NULL;
  GtkWidget *btn_rem = NULL;
  GtkWidget *button_box = (GtkWidget *) user_data;  
  GList     *buttons;
  gint n;
  gint i = 0;

  /* apply, cancel, add, remove button list */
  buttons = gtk_container_get_children (GTK_CONTAINER (button_box));
  while (buttons->next != NULL)
    {
      i++;
      buttons = buttons->next;
    }

  buttons = g_list_first (buttons);
  for (n = 1; n < i; n++) 
    {
      /* FIXME: add freeing of list elements? */
      gtk_object_destroy (buttons->data);   
      buttons = buttons->next;
    }

  /* we can create the buttons so far, if they are needed */
  if (gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)) > P_GENERAL)
    {
      btn_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
      gtk_widget_show (btn_add);
      gtk_box_pack_start (GTK_BOX (button_box), btn_add, FALSE, FALSE, 0);
      gtk_box_reorder_child (GTK_BOX (button_box), btn_add, 0);
      
      btn_rem = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
      gtk_widget_show (btn_rem);
      gtk_box_pack_start (GTK_BOX (button_box), btn_rem, FALSE, FALSE, 0);
      gtk_box_reorder_child (GTK_BOX (button_box), btn_rem, 1);
      /* gtk_widget_show (button_box); */
    }

  switch (page_num)
    {
    case P_SYSTEM:
      break;
    case P_GENERAL:
      break;
    case P_LESSON:
       g_signal_connect (G_OBJECT (btn_add), "clicked",
                         G_CALLBACK (on_btn_add_lesson_clicked), NULL);

       g_signal_connect_swapped (G_OBJECT (btn_rem), "clicked",
				 G_CALLBACK (on_btn_rem_lesson_clicked),
				 util_lookup_widget (GTK_WIDGET (notebook),
						     "tree_list_lessons"));
      break;
    case P_MODULE:
      g_signal_connect (G_OBJECT (btn_add), "clicked",
			G_CALLBACK (on_btn_add_module_clicked), NULL);

      g_signal_connect_swapped (G_OBJECT (btn_rem), "clicked",
				G_CALLBACK (on_btn_rem_module_clicked),
				util_lookup_widget (GTK_WIDGET (notebook),
						    "tree_list_modules"));
      break;
    default:
      err_error_cb (_("Page could not be found?!"));
      break;
    }
  return;
}


/* 
 * access selection 
 */
void
on_opt_method_changed (GtkOptionMenu *optionmenu)
{
  genLprefs.method = gtk_option_menu_get_history (optionmenu);
  return;
}

/* 
 * set the actual main language
 */
void 
on_opt_mainlang_changed (GtkOptionMenu *optionmenu)
{
  gchar *name = util_get_menu_label (optionmenu);
  
  if (strlen (name) == 0)
    name = selection[RANDOM];
  
  lpref_set_mlanguage (name);
  return;
}

/* 
 * apply  changes 
 */
void
on_btn_lpref_apply_clicked (gpointer user_data)
{
  GList         *tbl_children;
  GtkTableChild *child;
  GtkWidget     *notebook = (GtkWidget *) user_data;
  GtkWidget     *tbl_langs = util_lookup_widget (notebook, "tbl_lang");
  const gchar   *tmp;
  gint i = 0;
  int  count = ling_lang_get_max_languages (settings);

  /* save language selection */
  if (genLprefs.languages != NULL)
    {
      while (genLprefs.languages[i] != NULL)
	{
	  free (genLprefs.languages[i]);
	  i++;
	}
      free (genLprefs.languages);
    }

  genLprefs.languages = malloc (sizeof (char *) * (count+1));
  for (i = 0; i <= count; i++)
    genLprefs.languages[i] = NULL;

  i = 0;

  /* get the languages and store them */
  tbl_children = GTK_TABLE(tbl_langs)->children;

  for (tbl_children = g_list_first (tbl_children);
       tbl_children != NULL;       
       tbl_children = tbl_children->next)
    {
      child  = (GtkTableChild *) tbl_children->data;
      /* is it a toggle button and active? */
      if (GTK_IS_BUTTON(child->widget) 
	  && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child->widget)))
	{
	  tmp = gtk_button_get_label (GTK_BUTTON (child->widget));
	  genLprefs.languages[i] = (char *) malloc (strlen (tmp));
	  strcpy (genLprefs.languages[i], tmp);
	  i++;

#ifdef DEBUG
	  printf ("Debug: Language: %s\n", genLprefs.languages[i-1]);
#endif

  	}
    }
  write_config ();      /* write all to the config file */
  gui_new_status (_("Configuration saved."));
  return;
}

/* 
 * toggles the 'used' flag in the lesson list 
 */
void 
on_used_cell_toggled (GtkCellRendererToggle *cell, gchar *path_str, 
		      gpointer data)
{
  /* taken from the gtk-demo, thanks a lot */
  GtkTreeModel *model = (GtkTreeModel *) data;
  GtkTreePath  *path = gtk_tree_path_new_from_string (path_str);
  GtkTreeIter   iter;
  gboolean      fixed;
  gboolean     *yes_no;
  lingLesson   *tmp = lessons;
  gchar        *name;
  
  /* get toggled iter */
  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get      (model, &iter, CURR_USED, &fixed, -1);

  fixed ^= 1;    /* toggle */

  /* set new value */
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, CURR_USED, fixed, -1);

  gtk_tree_path_free (path);   /* clean up */
  
  /* now we get the lesson name and apply the changes to our internal system */
  gtk_tree_model_get (model, &iter, 
                      LESSON_NAME, &name,
                      CURR_USED, &yes_no,
                      -1);
  
  while (g_ascii_strcasecmp (tmp->type, name) != 0)
    tmp = tmp->next;

  ling_lesson_use_lesson (tmp, (lingbool) yes_no);
  
#ifdef DEBUG
  printf ("Debug: Lesson %s is toggled to %s\n", name, 
	  (yes_no) ? "used" : "unused");
#endif

  g_free (name);
  return;
}

/* 
 * add a lesson 
 */
void 
on_btn_add_lesson_clicked (void)
{
  GtkWidget *file_select = util_create_dlg_fileselect (_("Lesson Selection"), 
						       lpref_add_lesson_file);
  gtk_widget_show (file_select);
  return;
}

/* 
 * internal 'callback' for on_btn_add_lesson_clicked (void) 
 */
void
lpref_add_lesson_file (gchar *filename)
{
  GtkTreeIter  iter;
  gchar       *type;
  lingLesson  *lesson;

  if (lst_lessons == NULL)
    lst_lessons = gtk_list_store_new (NUMBER, 
				      G_TYPE_STRING, 
				      G_TYPE_STRING, 
				      G_TYPE_BOOLEAN);

  if ((lessons = ling_lesson_add_lesson (lessons, filename, settings)) == NULL)
    {
      err_error_cb (_("This is not a valid lesson file!"));
      return;
    }
  lesson = lessons; /* go to the last one (the one we newly added) */
  while (lesson->next != NULL)
    lesson = lesson->next;

  ling_lesson_use_lesson (lesson, TRUE); /* assume, the user _wants_ it :) */

  if ((type = lesson->type) == NULL)
    type = _("Undefined");

  gtk_list_store_insert_before (lst_lessons, &iter, NULL);
  gtk_list_store_set (lst_lessons, &iter,
		      CURR_USED, ling_lesson_return_used (lesson),
		      LESSON_NAME, type,
		      LESSON_PATH, filename,
		      -1);

  err = 0; /* reset error */
  gui_update (); /* update the gui elements */
  return;
}

/*
 * unload and remove an unwanted lesson
 */
void 
on_btn_rem_lesson_clicked (GtkTreeView *tree)
{
  gchar            *name;
  GtkTreeIter       iter;
  GtkTreeModel     *model     = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);
  lingLesson       *tmp = lessons;

  /* what is the selected and should be deleted? */
  if (gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      gtk_tree_model_get (model, &iter, 
			  LESSON_NAME, &name, 
			  -1);
      while (g_ascii_strcasecmp (tmp->type, name) != 0)
	tmp = tmp->next;

      lessons = ling_lesson_remove_lesson (lessons, tmp);

      /* delete it from the list and free its allocated mem */
      gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
      g_free (name);
    }
  gui_update (); /* update the gui elements */
  return;
}

/* 
 * add a module 
 */
void 
on_btn_add_module_clicked (void)
{
  GtkWidget *file_select = util_create_dlg_fileselect (_("Module Selection"), 
						       lpref_add_module);
  gtk_widget_show (file_select);
  return;
}

/* 
 * internal callback for on_btn_add_module_clicked (void) 
 */
void 
lpref_add_module (gchar *filename)
{
 module_load (lst_modules, filename);
 return;
}

/* 
 * removes a module from the module list 
 */
void 
on_btn_rem_module_clicked (GtkTreeView *tree)
{
  gchar            *name;
  GtkTreeIter       iter;
  GtkTreeModel     *model     = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);
  
  /* what is the selected, which should be deleted? */
  if (gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      /* get the module name */
      gtk_tree_model_get (model, &iter, 0, &name, -1);

      if (!module_unload (name))
	{
	  err_critical_cb (_("Error while removing the module from the list!\n"
			     "Please quit and restart LingoTeach"));
	  g_free (name);
	  return;
	}
      /* delete it from the list and free its allocated memory */
      lpref_clear_module_box (box_module_set);
      gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
    }
  g_free (name);
  return;
}

/* 
 * changes the module box content, if another module is selected 
*/
void 
on_tree_list_modules_row_changed (GtkTreeView *tree)
{
  lpref_clear_module_box (box_module_set);
  lpref_module_set_content (tree);
  return;
}

/* 
 * clear the module box 
*/
void 
lpref_clear_module_box (GtkWidget *module_box)
{
  GList *modules =  gtk_container_get_children (GTK_CONTAINER (module_box));  
  if (modules != NULL)
    {
      gtk_widget_destroy (modules->data);
      g_list_free (modules);
      /* gtk_widget_show_all (module_box); */
    }
   return;
}

/* 
 * set the module content 
 */
void
lpref_module_set_content (GtkTreeView *tree)
{  
  gchar            *name      = NULL;
  GtkTreeModel     *model     = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);
  GtkTreeIter       iter;
  GtkWidget        *module;

  /* what is selected */
  if (gtk_tree_selection_get_selected (selection, NULL, &iter))
    gtk_tree_model_get (model, &iter, 0, &name, -1); /* get the selected one */
    
  lpref_clear_module_box (box_module_set);  
  
  module = GTK_WIDGET (module_get_info (name));
  gtk_box_pack_start (GTK_BOX (box_module_set), module, FALSE, FALSE, 0);

  /* gtk_widget_show_all (box_module_set); */
  
  g_free (name);
  return;
}

/* return the available languages */
GtkWidget* 
lpref_get_languages (void)
{
  GtkWidget  *tbl_languages;
  GtkWidget  *btn_lang;
  GtkWidget  *lbl_languages;
  lingchar  **languages = ling_lang_get_languages (settings);

  gint count = ling_lang_get_max_languages (settings);
  gint i;
  gint j = 0;
  gint old = 1;
  
  tbl_languages = gtk_table_new (2, ABS (count / 2), FALSE);
  lbl_languages = gtk_label_new (_("Languages"));
  gtk_widget_show (lbl_languages);
  gtk_table_attach_defaults (GTK_TABLE (tbl_languages), lbl_languages,
			     0, 2, 0, 1);
  
  /* create language buttons */
  for (i = 0; i < count; i += 2, old++)
    {
      btn_lang = gtk_check_button_new_with_label (languages[i]);
      gtk_widget_show (btn_lang);
      gtk_table_attach_defaults (GTK_TABLE (tbl_languages), btn_lang,
				 0, 1, old, old+1);
      /* set language */
      j = 0;
      if (genLprefs.languages != NULL)
	while (genLprefs.languages[j] != NULL)
	  {
	    if (strcmp (genLprefs.languages[j], languages[i]) == 0)
	      {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_lang), 
					      TRUE);
		break;
	      }
	    j++;
	  }
      
      btn_lang = gtk_check_button_new_with_label (languages[i+1]);
      gtk_widget_show (btn_lang);
      gtk_table_attach_defaults (GTK_TABLE (tbl_languages), btn_lang,
				 1, 2, old, old+1);
      /* set language */
      j = 0;
      if (genLprefs.languages != NULL)
	while (genLprefs.languages[j] != NULL)
	  {
	    if (strcmp (genLprefs.languages[j], languages[i+1]) == 0)
	      {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_lang), 
					      TRUE);
		break;
	      }
	    j++;
	  }
    }
  
  gtk_widget_show (tbl_languages);
  free (languages);
  return tbl_languages;
}

/* 
 * returns the list of lessons 
 */
GtkListStore*
get_list_lessons (void)
{
  GtkTreeIter iter;
  lingLesson *tmp = lessons;
  gchar      *filename;
  gchar      *type;
  
  /* get the used files */
  while ((filename = ling_lesson_return_path (tmp)) != NULL) 
    {
      if ((type = tmp->type) == NULL)
	    type = _("Undefined");
      
      /* add the lessons to the store */
      gtk_list_store_insert_before (lst_lessons, &iter, NULL);
      gtk_list_store_set (lst_lessons, &iter, 
			  LESSON_NAME, type,
			  LESSON_PATH, filename, 
			  CURR_USED, ling_lesson_return_used (tmp),
			  -1);
      tmp = tmp->next;
    }
  return lst_lessons;
}

/* 
 * returns the list of loaded modules  (if any)
 */
GtkListStore* 
get_lpref_modules ()
{
  if (!lst_modules)
    {
      lst_modules = gtk_list_store_new (1, G_TYPE_STRING);
      module_get_list (lst_modules);
    }
  return lst_modules;
}

/* 
 * preference functions 
 */
void 
lpref_set_soundpath (gchar *name)
{
  if (genLprefs.soundpath != NULL)
    free (genLprefs.soundpath);
  genLprefs.soundpath = (gchar *) malloc (strlen (name));
  memset (genLprefs.soundpath, 0, strlen (name));
  strcpy (genLprefs.soundpath, name);

  if (prefWidgets.txt_soundpath != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_soundpath),
			genLprefs.soundpath);
  return;
}

void 
lpref_set_mlanguage (gchar *name)
{
  if (genLprefs.mlanguage != NULL)
    free (genLprefs.mlanguage);
  genLprefs.mlanguage = (gchar *) malloc (strlen (name));
  memset (genLprefs.mlanguage, 0, strlen (name));
  strcpy (genLprefs.mlanguage, name);
  err = 0;
  return;
}

void 
lpref_set_browser (gchar *name)
{
  if (genLprefs.browser != NULL)
    free (genLprefs.browser);
  genLprefs.browser = (gchar *) malloc (strlen (name));
  memset (genLprefs.browser, 0, strlen (name));
  strcpy (genLprefs.browser, name);

  if (prefWidgets.txt_browser != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_browser), 
			genLprefs.browser);
  return;
}

void 
lpref_set_player (gchar* name)
{
  if (genLprefs.player != NULL)
    free (genLprefs.player);
  genLprefs.player = (gchar *) malloc (strlen (name));
  memset (genLprefs.player, 0, strlen (name));
  strcpy (genLprefs.player, name);

  if (prefWidgets.txt_player != NULL)
    gtk_entry_set_text (GTK_ENTRY (prefWidgets.txt_player), genLprefs.player);
  return;
}

/* this takes care of writing the whole stuff into a config file */
void 
write_config (void)
{
  FILE  *fp;
  gint   i = 0;
  gint   use;
  gchar *name;

  GtkTreeIter iter;
  lingLesson  *tmp;

  gchar *file = g_build_filename (getenv ("HOME"), CONFIG, NULL);
  fp = fopen (file, "w+");

  if (fp == NULL)
    {
      err_error_cb (_("Cannot write configuration, check your\n"
		      "HOME/.lingoteach/config"));
      return;
    }

  /* general settings */
  name = (gchar *) gtk_entry_get_text (GTK_ENTRY (prefWidgets.txt_soundpath)); 
  if (strlen (name) > 0)
    {
      lpref_set_soundpath (name);
      fprintf (fp, "soundpath=%s\n", genLprefs.soundpath);
    }

  name = (gchar *) gtk_entry_get_text (GTK_ENTRY (prefWidgets.txt_browser));
  if (strlen (name) > 0)
    {
      lpref_set_browser (name);
      fprintf (fp, "browser=%s\n", genLprefs.browser);
    }

  name = (gchar *) gtk_entry_get_text (GTK_ENTRY (prefWidgets.txt_player));
  if (strlen (name) > 0)
    {
      lpref_set_player (name);
      fprintf (fp, "player=%s\n", genLprefs.player);
    }
  
  fprintf (fp, "method=%i\n", genLprefs.method);
  
  if (genLprefs.mlanguage != NULL && strcmp (genLprefs.mlanguage, "") != 0)
    fprintf (fp, "mlanguage=%s\n", genLprefs.mlanguage);

  while (genLprefs.languages[i] != NULL 
	 && strcmp (genLprefs.languages[i], "") != 0)
    {
      fprintf (fp, "language=%s\n", genLprefs.languages[i]);
      i++;
    }
  
  g_free (file);

  /* lessons */
  tmp = lessons;
  while (tmp != NULL) 
    {
      (ling_lesson_return_used (tmp) == TRUE) ? (use = 1) : (use = 0);
      fprintf (fp, "lesson=%s\n", ling_lesson_return_path (tmp));
      tmp = tmp->next;
    }
  
  /* modules */
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (lst_modules), &iter))
    do 
      {
	gtk_tree_model_get (GTK_TREE_MODEL (lst_modules), &iter,
			    0, &name, -1);
	fprintf (fp, "module=%s\n", module_get_path (name));
      }
    while (gtk_tree_model_iter_next (GTK_TREE_MODEL (lst_modules), &iter));
  
  //g_free (name);
  fclose (fp);
  return;
}

/*
 * gets the currently active language
 */
gint
get_lang_index (gchar* language)
{
  gint i = 0;
  lingchar **lang = ling_lang_get_languages (settings);
  
  while (lang[i] != NULL && g_ascii_strcasecmp (language, lang[i]) != 0)
    i++;

  free (lang);
  return i;
}



/********************
 * public functions *
 ********************/

/* 
 * preference click in main menu
 */
void
on_preferences1_activate (void)
{
  GtkWidget *prefs;
  gint      result;

  prefs = create_dlg_lpref ();
  result = gtk_dialog_run (GTK_DIALOG (prefs));
  switch (result)
    {
    case GTK_RESPONSE_APPLY:
      gtk_widget_destroy (prefs);
      break;
    case GTK_RESPONSE_CANCEL:
      gtk_widget_destroy (prefs);
      break;
    default:
      gtk_widget_destroy (prefs);
      break;
    }
  return;
}


/* 
 * reads the configuration file 
 */
lingbool 
read_config (void)
{
  /* FIXME: make this stuff much better */
  FILE *fp;
  char  line[PREF_MAX_LEN];
  char *tmp;
  int   i = 0;
  int   j = 0;

  gchar *file = g_build_filename (getenv ("HOME"), CONFIG, NULL);
  fp = fopen (file, "r+");

  if (fp == NULL)
    {
      err_error_cb (_("Cannot read configuration, check your\n"
		      "$HOME/.lingoteach/config"));
      return FALSE;
    }
  /* not a good solution, but works for now :-/ */
  while (fgets (line, sizeof (line), fp) != NULL)
    {
      line[strlen(line)-1] = '\0';
      if (strstr (line, "soundpath") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  lpref_set_soundpath (tmp);
	}
      else if (strstr (line, "browser") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  lpref_set_browser (tmp);
	}
      else if (strstr (line, "player") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  lpref_set_player (tmp);
	}
      else if (strstr (line, "method") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  genLprefs.method = atoi (tmp);
	}
      else if (strstr (line, "mlanguage") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  lpref_set_mlanguage (tmp);
	}
      else if (strstr (line, "language") != NULL)
	{
	  if (genLprefs.languages == NULL)
	    {
	      j = ling_lang_get_max_languages (settings);
	      genLprefs.languages = 
		malloc (sizeof (char *) * j);

	      for (i = 0; i <= j; i++)
		genLprefs.languages[i] = NULL;
	      i = 0;
	    }
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  genLprefs.languages[i] = malloc (strlen (tmp));
	  strcpy (genLprefs.languages[i], tmp);
	  i++;
	}
      else if (strstr (line, "lesson") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  lpref_add_lesson_file (tmp);
	}
      else if (strstr (line, "module") != NULL)
	{
	  tmp = strtok (line, "=");
	  tmp = strtok (NULL, "=");
	  if (lst_modules == NULL)
	    get_lpref_modules ();
	  lpref_add_module (tmp);	  
	}
    }
  fclose (fp);
  g_free (file);
  
  return TRUE;
}

/*
 * creates a new configuration file
 */
void
create_config (void)
{
  gchar *dir = g_build_filename (getenv ("HOME"), ".lingoteach", NULL);
  if (access (dir, X_OK) == 0)
    {
      g_free (dir);
      return;
    }
    
  if (mkdir (dir, 0700) == -1)
    err_error_cb (_("Could not create configuration directory..."));

  g_free (dir);
  return;
}
