/*
 * Copyright (C) 2004-2006 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdint.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#if WITH_IPV6
#include <netinet/icmp6.h>
#endif /* WITH_IPV6 */
#include <glib.h>
#include <glib/gi18n.h>
#include "lm-icmp.h"
#include "lm-util.h"

/*** cpp *********************************************************************/

/* receive buffer lengths */
#define MAX_IP_PACKET_LEN	IP_MAXPACKET
#define MAX_ICMP6_PACKET_LEN	2352	/* from FreeBSD's ping6.c (packlen when F_VERBOSE is set) */

/* send buffer (and packet) lengths */
#define ICMP_PACKET_LEN		(ICMP_MINLEN + sizeof(LMICMPData))
#define ICMP6_ECHO_LEN		8	/* from FreeBSD's ping6.c */
#define ICMP6_PACKET_LEN	(ICMP6_ECHO_LEN + sizeof(LMICMPData))

/* these ICMPv6 types are not defined by all platforms */
#ifndef ICMP6_DST_UNREACH_BEYONDSCOPE
#define ICMP6_DST_UNREACH_BEYONDSCOPE	2	/* RFC 3542 */
#endif
#ifndef ICMP6_MEMBERSHIP_QUERY
#define ICMP6_MEMBERSHIP_QUERY		130
#endif
#ifndef ICMP6_MEMBERSHIP_REPORT
#define ICMP6_MEMBERSHIP_REPORT		131
#endif
#ifndef ICMP6_MEMBERSHIP_REDUCTION
#define ICMP6_MEMBERSHIP_REDUCTION	132
#endif
#ifndef ICMP6_NI_QUERY
#define ICMP6_NI_QUERY			139
#endif
#ifndef ICMP6_NI_REPLY
#define ICMP6_NI_REPLY			140
#endif

/*** types *******************************************************************/

typedef struct
{
  uint16_t	host_id;
  struct
  {
    uint32_t	tv_sec;
    uint32_t	tv_usec;
  } sent;
} LMICMPData;

/*** functions ***************************************************************/

static void lm_icmp_fill_data (LMICMPData *data, uint16_t host_id);
static u_short lm_icmp_in_cksum (u_short *addr, int len);

static const char *lm_icmp_reply_get_description4 (const LMICMPReply *reply);
#if WITH_IPV6
static const char *lm_icmp_reply_get_description6 (const LMICMPReply *reply);
#endif

/*** implementation **********************************************************/

static void
lm_icmp_fill_data (LMICMPData *data, uint16_t host_id)
{
  GTimeVal now;

  g_return_if_fail(data != NULL);

  data->host_id = host_id;
  g_get_current_time(&now);
  data->sent.tv_sec = now.tv_sec;
  data->sent.tv_usec = now.tv_usec;
}

/*
 * Compute an IP checksum (based on in_cksum() from FreeBSD's ping.c).
 *
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Mike Muuss.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
static u_short
lm_icmp_in_cksum (u_short *addr, int len)
{
  int nleft, sum;
  u_short *w;
  union {
    u_short	us;
    u_char	uc[2];
  } last;
  u_short answer;

  g_return_val_if_fail(addr != NULL, 0);

  nleft = len;
  sum = 0;
  w = addr;

  /*
   * Our algorithm is simple, using a 32 bit accumulator (sum), we add
   * sequential 16 bit words to it, and at the end, fold back all the
   * carry bits from the top 16 bits into the lower 16 bits.
   */
  while (nleft > 1)
    {
      sum += *w++;
      nleft -= 2;
    }

  /* mop up an odd byte, if necessary */
  if (nleft == 1)
    {
      last.uc[0] = *(u_char *) w;
      last.uc[1] = 0;
      sum += last.us;
    }

  /* add back carry outs from top 16 bits to low 16 bits */
  sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
  sum += (sum >> 16);			/* add carry */
  answer = ~sum;			/* truncate to 16 bits */

  return answer;
}

gboolean
lm_icmp_send_echo_request (const LMSocket *sock,
			   const struct addrinfo *addrinfo,
			   uint16_t host_id,
			   uint16_t seq,
			   GError **err)
{
  char buf[MAX(ICMP_PACKET_LEN, ICMP6_PACKET_LEN)];
  size_t packet_len;
  ssize_t sent_len;

  g_return_val_if_fail(sock != NULL, FALSE);
  g_return_val_if_fail(sock->init_error == FALSE, FALSE);
  g_return_val_if_fail(addrinfo != NULL, FALSE);

  if (addrinfo->ai_family == AF_INET)
    {
      struct icmp *icmp;

      packet_len = ICMP_PACKET_LEN;
      icmp = (struct icmp *) buf;
      icmp->icmp_type = ICMP_ECHO;
      icmp->icmp_code = 0;
      icmp->icmp_id = g_htons(lm_shell->icmp_ident);
      icmp->icmp_seq = g_htons(seq);
      lm_icmp_fill_data((LMICMPData *) icmp->icmp_data, host_id);
      icmp->icmp_cksum = 0;
      icmp->icmp_cksum = lm_icmp_in_cksum((u_short *) icmp, packet_len);
    }
#if WITH_IPV6
  else if (addrinfo->ai_family == AF_INET6)
    {
      struct icmp6_hdr *icmp6;

      packet_len = ICMP6_PACKET_LEN;
      icmp6 = (struct icmp6_hdr *) buf;
      icmp6->icmp6_type = ICMP6_ECHO_REQUEST;
      icmp6->icmp6_code = 0;
      icmp6->icmp6_id = g_htons(lm_shell->icmp_ident);
      icmp6->icmp6_seq = g_htons(seq);
      lm_icmp_fill_data((LMICMPData *) (icmp6 + 1), host_id);
      /* icmp_cksum is filled in by the kernel */
    }
#endif /* WITH_IPV6 */
  else
    g_assert_not_reached();

  do
    sent_len = sendto(sock->desc, buf, packet_len, 0, addrinfo->ai_addr, addrinfo->ai_addrlen);
  while (sent_len < 0 && errno == EINTR);

  if (sent_len != packet_len)
    {
      if (sent_len < 0)
	g_set_error(err, 0, 0, "%s", g_strerror(errno));
      else
	g_set_error(err, 0, 0, _("could not send whole packet"));

      return FALSE;
    }

  return TRUE;
}

gboolean
lm_icmp_reply_get (const LMSocket *sock, LMICMPReply *reply)
{
  gboolean status = FALSE;
  char buf[MAX(MAX_IP_PACKET_LEN, MAX_ICMP6_PACKET_LEN)];
  ssize_t nbytes;

  g_return_val_if_fail(sock != NULL, FALSE);
  g_return_val_if_fail(sock->init_error == FALSE, FALSE);
  g_return_val_if_fail(reply != NULL, FALSE);

  do
    nbytes = recvfrom(sock->desc, buf, sizeof(buf), 0, NULL, NULL);
  while (nbytes < 0 && errno == EINTR);

  if (nbytes > 0)
    {
      LMICMPData *data;

      if (sock->domain == AF_INET)
	{
	  struct ip *ip;
	  struct icmp *icmp;
	  int header_len;
	  int icmp_len;

	  ip = (struct ip *) buf;
	  header_len = ip->ip_hl << 2;
	  icmp_len = nbytes - header_len;
	  icmp = (struct icmp *) (buf + header_len);

	  if (icmp_len >= ICMP_PACKET_LEN && g_ntohs(icmp->icmp_id) == lm_shell->icmp_ident)
	    {
	      data = (LMICMPData *) icmp->icmp_data;
	      reply->icmp_type = icmp->icmp_type;
	      reply->icmp_code = icmp->icmp_code;
	      reply->seq = g_ntohs(icmp->icmp_seq);

	      status = TRUE;
	    }
	}
#if WITH_IPV6
      else if (sock->domain == AF_INET6)
	{
	  struct icmp6_hdr *icmp6;

	  icmp6 = (struct icmp6_hdr *) buf;

	  if (nbytes >= ICMP6_PACKET_LEN && g_ntohs(icmp6->icmp6_id) == lm_shell->icmp_ident)
	    {
	      data = (LMICMPData *) (icmp6 + 1);
	      reply->icmp_type = icmp6->icmp6_type;
	      reply->icmp_code = icmp6->icmp6_code;
	      reply->seq = g_ntohs(icmp6->icmp6_seq);

	      status = TRUE;
	    }
	}
#endif /* WITH_IPV6 */
      else
	g_assert_not_reached();

      if (status)
	{
	  reply->host_id = data->host_id;
	  reply->sent.tv_sec = data->sent.tv_sec;
	  reply->sent.tv_usec = data->sent.tv_usec;
	  g_get_current_time(&reply->received);
	}
    }

  return status;
}

gboolean
lm_icmp_reply_is_echo_reply (const LMICMPReply *reply, int domain)
{
  g_return_val_if_fail(reply != NULL, FALSE);

  switch (domain)
    {
    case AF_INET:	return reply->icmp_type == ICMP_ECHOREPLY;
#if WITH_IPV6
    case AF_INET6:	return reply->icmp_type == ICMP6_ECHO_REPLY;
#endif
    default:		g_assert_not_reached(); return FALSE;
    }
}

double
lm_icmp_reply_get_roundtrip_time (const LMICMPReply *reply)
{
  GTimeVal rtt;

  g_return_val_if_fail(reply != NULL, 0);

  rtt = reply->received;
  lm_tvsub(&rtt, &reply->sent);

  return (double) rtt.tv_sec * 1000 + (double) rtt.tv_usec / 1000;
}

static const char *
lm_icmp_reply_get_description4 (const LMICMPReply *reply)
{
  g_return_val_if_fail(reply != NULL, NULL);

  switch (reply->icmp_type)
    {
    case ICMP_ECHOREPLY:			return _("echo reply");
    case ICMP_UNREACH:
      switch (reply->icmp_code)
	{
	case ICMP_UNREACH_NET:			return _("destination network unreachable");
	case ICMP_UNREACH_HOST:			return _("destination host unreachable");
	case ICMP_UNREACH_PROTOCOL:		return _("destination protocol unreachable");
	case ICMP_UNREACH_PORT:			return _("destination port unreachable");
	case ICMP_UNREACH_NEEDFRAG:		return _("fragmentation needed and DF set");
	case ICMP_UNREACH_SRCFAIL:		return _("source route failed");
	case ICMP_UNREACH_FILTER_PROHIB:	return _("communication prohibited by filter");
	default:				return _("destination unreachable, unknown ICMP code");
	}
    case ICMP_SOURCEQUENCH:			return _("source quench");
    case ICMP_REDIRECT:
      switch (reply->icmp_code)
	{
	case ICMP_REDIRECT_NET:			return _("redirect network");
	case ICMP_REDIRECT_HOST:		return _("redirect host");
	case ICMP_REDIRECT_TOSNET:		return _("redirect type of service and network");
	case ICMP_REDIRECT_TOSHOST:		return _("redirect type of service and host");
	default:				return _("redirect, unknown ICMP code");
	}
    case ICMP_ECHO:				return _("echo request");
    case ICMP_TIMXCEED:
      switch (reply->icmp_code)
	{
	case ICMP_TIMXCEED_INTRANS:		return _("time to live exceeded");
	case ICMP_TIMXCEED_REASS:		return _("fragment reassembly time exceeded");
	default:				return _("time exceeded, unknown ICMP code");
	}
    case ICMP_PARAMPROB:			return _("parameter problem");
    case ICMP_TSTAMP:				return _("timestamp");
    case ICMP_TSTAMPREPLY:			return _("timestamp reply");
    case ICMP_IREQ:				return _("information request");
    case ICMP_IREQREPLY:			return _("information reply");
    case ICMP_MASKREQ:				return _("address mask request");
    case ICMP_MASKREPLY:			return _("address mask reply");
    case ICMP_ROUTERADVERT:			return _("router advertisement");
    case ICMP_ROUTERSOLICIT:			return _("router solicitation");
    default:					return _("unknown ICMP type");
    }
}

#if WITH_IPV6
static const char *
lm_icmp_reply_get_description6 (const LMICMPReply *reply)
{
  g_return_val_if_fail(reply != NULL, NULL);

  switch (reply->icmp_type)
    {
    case ICMP6_DST_UNREACH:
      switch (reply->icmp_code)
	{
	case ICMP6_DST_UNREACH_NOROUTE:		return _("no route to destination");
	case ICMP6_DST_UNREACH_ADMIN:		return _("destination administratively unreachable");
	case ICMP6_DST_UNREACH_BEYONDSCOPE:	return _("destination unreachable beyond scope");
	case ICMP6_DST_UNREACH_ADDR:		return _("destination host unreachable");
	case ICMP6_DST_UNREACH_NOPORT:		return _("destination port unreachable");
	default:				return _("destination unreachable, unknown ICMPv6 code");
	}
    case ICMP6_PACKET_TOO_BIG:			return _("packet too big");
    case ICMP6_TIME_EXCEEDED:
      switch (reply->icmp_code)
	{
	case ICMP6_TIME_EXCEED_TRANSIT:		return _("time to live exceeded");
	case ICMP6_TIME_EXCEED_REASSEMBLY:	return _("fragment reassembly time exceeded");
	default:				return _("time exceeded, unknown ICMPv6 code");
	}
    case ICMP6_PARAM_PROB:
      switch (reply->icmp_code)
	{
	case ICMP6_PARAMPROB_HEADER:		return _("parameter problem: erroneous header");
	case ICMP6_PARAMPROB_NEXTHEADER:	return _("parameter problem: unknown next header");
	case ICMP6_PARAMPROB_OPTION:		return _("parameter problem: unrecognized option");
	default:				return _("parameter problem, unknown ICMPv6 code");
	}
    case ICMP6_ECHO_REQUEST:			return _("echo request");
    case ICMP6_ECHO_REPLY:			return _("echo reply");
    case ICMP6_MEMBERSHIP_QUERY:		return _("listener query");
    case ICMP6_MEMBERSHIP_REPORT:		return _("listener report");
    case ICMP6_MEMBERSHIP_REDUCTION:		return _("listener done");
    case ND_ROUTER_SOLICIT:			return _("router solicitation");
    case ND_ROUTER_ADVERT:			return _("router advertisement");
    case ND_NEIGHBOR_SOLICIT:			return _("neighbor solicitation");
    case ND_NEIGHBOR_ADVERT:			return _("neighbor advertisement");
    case ND_REDIRECT:				return _("redirect");
    case ICMP6_NI_QUERY:			return _("node information query");
    case ICMP6_NI_REPLY:			return _("node information reply");
    default:					return _("unknown ICMPv6 type");
    }
}
#endif /* WITH_IPV6 */

const char *
lm_icmp_reply_get_description (const LMICMPReply *reply, int domain)
{
  g_return_val_if_fail(reply != NULL, NULL);

  switch (domain)
    {
    case AF_INET:	return lm_icmp_reply_get_description4(reply);
#if WITH_IPV6
    case AF_INET6:	return lm_icmp_reply_get_description6(reply);
#endif
    default:		g_assert_not_reached(); return NULL;
    }
}
