///	@file	dialog.cpp
///	@brief	dialog.cpp
#include <string.h>
#include <stdarg.h>

#include "dialog.h"
#include "mlslocale.h"
#include "drawutil.h"
#include "keydef.h"
#include "strutil.h"

using namespace strutil;
using namespace std;

namespace MLS {

///	@brief	메세지 박스를 그린다
///	@param	b_c		배경 색상
///	@param	title	title message
///	@param	msg		출력할 글자
///	@return	성공시 1을 반환
int MsgBox(int b_c, const char *title, const char *msg,...)
{
	va_list args;
    char szBuf[256];

    va_start(args,msg);
    vsprintf(szBuf,msg,args);
    va_end(args);

    return MsgBox(title, szBuf, b_c);
}


///	@brief	메세지 박스를 그린다
///	@param	title	title message
///	@param	msg		출력할 글자
///	@param	back_color	배경 색상
///	@return	성공시 1을 반환
int MsgBox(const char *title, const char *msg, int back_color)
{
	int l_title = krstrlen(title);
	int l_msg = krstrlen(msg);
	int width = l_title > l_msg ? l_title : l_msg;

	width +=4;

	WINDOW *win = newwin(5, width, (g_nLINES-5)/2, (g_nCOLS-width)/2);
	//	wclear(win);
	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', width-2);
	mvwprintw(win, 1, (width - l_title)/2 , "%s", title);
	wattroff(win, A_BOLD);

	// msg 출력
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);
	mvwprintw(win, 3, (width - l_msg)/2, "%s", msg);
	wattroff(win, A_BOLD);

	wrefresh(win);
	getch();
	wclear(win);
	delwin(win);
	return 1;
}

///	@brief	메세지 박스를 그린다
///	@param	title	title message
///	@param	msg		입력받은 글자[반환값]
///	@param	back_color	배경 색상
///	@return	메세지 박스 WINDOW
WINDOW* MsgWaitBox(const char *title, const char *msg, int back_color)
{
	int l_title = krstrlen(title);
	int l_msg = krstrlen(msg);
	int width = l_title > l_msg ? l_title : l_msg;

	width +=4;

	// . width,5[x,y] 크기의 박스를 만든다.
	WINDOW *win = newwin(5, width, (g_nLINES-5)/2, (g_nCOLS-width)/2);

	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', width-2);
	mvwprintw(win, 1, (width - l_title)/2 , "%s", title);
	wattroff(win, A_BOLD);

	// msg
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);
	mvwprintw(win, 3, (width - l_msg)/2, "%s", msg);
	wattroff(win, A_BOLD);

	wrefresh(win);
	return win;
}

///	@brief	해당 Window를 제거한다.
///	@param	win		제거할 window
///	@return	1을 반환
int MsgWaitEnd(WINDOW*	 win)
{
	//	wclear(win);
	wrefresh(win);
	delwin(win);
	refresh();
	return 1;
}

///	@brief	Input Box를 그린다.
///	@param	title	title message
///	@param	str		입력받은 글자[반환값]
///	@param	back_color	배경 색상
///	@return	1 : 성공\n -1 : 실패
int InputBox(const char *title, std::string &str, int back_color)
{ // 글자가 넘어갈 경우 스크롤시킨다.

	int l_title = krstrlen(title);
	//int width = g_nCOLS - 4;
	int width = 44;

	if (width < l_title) 		width = l_title+4;
	if (l_title+6 > g_nCOLS) 	width = g_nCOLS - 4;
	
	int W = width - 4;
	curs_set(1); // 커서를 보이게 한다.
	WINDOW *win = newwin(5, width, (g_nLINES-5)/2, (g_nCOLS-width)/2);

	wclear(win);
	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', width-2);
	mvwprintw(win, 1, (width - l_title)/2 , "%s", title);
	wattroff(win, A_BOLD);

	wstring		wstr = strtowstr(str);
	int cur=wstr.size(), scroll= wstr.size() > W ? wstr.size()-W : 0; // 커서, 스크롤

	int 	nFirst = 0, nEnd = 0;
	int 	key, key2, key3, key4;
	int 	W1;
	int		viewcur = 0;
	string	sKey, sViewStr;
	
	while(1)
	{
		// 찍고
		wattron (win, COLOR(COLOR_WHITE, back_color));
		wattron (win, A_BOLD);
		wmove(win, 3, 2);
		whline(win, ' ', W);

		W1 = W;
		if (wstrlen(wstr.substr(nFirst, W1)) > W)
		{
			while(1) {
				if (wstrlen(wstr.substr(nFirst, W1)) <= W) break;
				W1--;
			}
			
		}
		nEnd = nFirst + W1;
		
		sViewStr = wstrtostr(wstr.substr(nFirst, W1));
		
		wprintw(win, "%s", sViewStr.c_str());
		wattroff(win, A_BOLD);

		// 커서가 있는 곳으로 이동한다.
		viewcur = wstrlen(wstr.substr(nFirst, cur-nFirst));
		wmove(win, 3, 2+viewcur);

		wrefresh(win);

		key = getch();

		if (key >= 0x80 || key == 27)
		{
			nodelay(stdscr, TRUE);
			key2 = getch();
			key3 = getch();
			key4 = getch();
			nodelay(stdscr, FALSE);
		}
		else
		{
			key2 = ERR;	key3 = ERR; key4 = ERR;
		}

		// 리모트 접속 End, Home 키
		// 27, 79, 72 (linux) ::  27, 91, 72 (vt100)
		if (key == 27 && (key2 == 91 || key2 == 79))
		{
			if (key3 == 72)	key = KEY_HOME;
			if (key3 == 70)	key = KEY_END;
			// Konsole. End Home 키
			if (key3 == 49 && key4 == 126) key = KEY_HOME;
			if (key3 == 52 && key4 == 126) key = KEY_END;
		}

		if (key != 27)
		{
			sKey = "";
			sKey.append(1, (char)key);
	
			// utf8 -> 3char, etc -> 2char
			if (key2 != ERR) sKey.append(1, (char)key2);
			if (key3 != ERR) sKey.append(1, (char)key3);
		}

		// home, end, pgup, pgdn, 화살표, 아스키
		if (key == KEY_HOME || key == KEY_PPAGE || key == KEY_UP)
		{
			cur = 0;
			nFirst = 0;
			continue;
		}

		if (key == KEY_END || key == KEY_NPAGE || key == KEY_DOWN)
		{
			cur = wstr.size();
			nFirst = wstr.size()-10;
			if (nFirst < 0) nFirst = 0;
			continue;
		}

		if (key == KEY_LEFT)
		{
			cur--;
			if (cur < 0) cur = 0;
			if (cur-nFirst < 3)
			{
				nFirst = cur - 3;
				if (nFirst < 0) nFirst = 0;
			}
			continue;
		}

		if (key == KEY_RIGHT)
		{
			cur++;
			if (cur > wstr.size()) cur = wstr.size();
			if (cur > nEnd-3)
			{
				if (nFirst < wstr.size()-5)
					nFirst = nFirst + 1;
			}
			continue;
		}

		if (key == 27) { curs_set(0); delwin(win); return -1; }
		if (key == 10 || key == 13) break;

		// backspace 커서 위치에서 한글자 지운다.
		// Konsole bugfix
		if (key == KEY_BS || key == KEY_BS_2 || key == 8) 
		{
			if (!wstr.empty() && cur)
			{
				wstr.erase(cur-1, 1);
				cur--;
				if (cur-nFirst > 4)
				{
					nFirst=nFirst-1;
					if (nFirst < 0) nFirst = 0;
				}
			}
			continue;
		}

		// del..
		if (key == 330)
		{
			if (!wstr.empty())
				wstr.erase(cur, 1);
			continue;
		}

		// 한글일 경우도 입력 가능하게
		if (32 <= key)
		{
			//MsgBox(1, "f", "%d", cur);
			wstring	wstrkey = strtowstr(sKey);
			if (wstrkey.size() > 0)
			{
				wstr.insert((wstring::size_type)cur, wstrkey);
				cur = cur + wstrkey.size();
				if (cur >= nEnd) nFirst=nFirst+wstrkey.size();
			}
			continue;
		}
	}

	str = wstrtostr(wstr);
	wclear(win);
	delwin(win);
	curs_set(0); // // 커서를 보이지 않게 한다.
	return 1;
}

///	@brief	Select Box를 그린다.
///	@param	title	title message
///	@param	p		선택할 수 있는 글자들[화면에 출력할 글자]
///	@param	sel		기본으로 sel번째 글자가 선택됨
///	@param	back_color	배경 색상
///	@return	0 이상 : 선택된 글자 index\n -1 : 실패
int SelectBox(const char *title, vector<string> &p, int sel, int back_color)
{
	typedef vector<string>::iterator Itor;
	vector<int> location;

	// . 화면에 선택 박스는 최소한 2개이상의 선택할수 있는 것이 있어야함
	if (p.size() < 2) return -1;

	// . 화면에 출력할 글자들의 위치를 구한다.
	location.push_back(3);
	for (Itor i=p.begin(); i!=p.end()-1; ++i)
		location.push_back( location.back() + (*i).size() + 3);

	// . 화면에 출력할 박스 크기를 구한다.[좌우3크기의 여백을 가진다.]
	int width = location.back() + p.back().size() + 3;
	if (width < krstrlen(title) + 4) width = krstrlen(title) + 4;

	WINDOW *win = newwin(5, width, (g_nLINES-5)/2, (g_nCOLS-width)/2);
	wbkgd(win, COLOR(COLOR_WHITE, back_color));
	wattron(win, A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	setcol(COLOR_BLACK, COLOR_WHITE, win);
	wmove(win, 1, 1);
	whline(win, ' ', width-2);
	mvwprintw(win, 1, (width - krstrlen(title))/2 , "%s", title);

	int key;

	//--Yes--- ---No---
	while(1)
	{// 출력
		wmove (win, 3, (width-17)/2);

		setcol(COLOR_WHITE, back_color, win);
		for (int t=0; t<p.size() ;++t)
		{
			wmove(win, 3, location[t]-1);
			wprintw(win, " %s ", p[t].c_str());
		}

		setcol(back_color, COLOR_WHITE, win);
		wmove(win, 3, location[sel]-1);
		wprintw(win, " %s ", p[sel].c_str());

		wmove(win, 0, 0);

		wrefresh(win);

		key=getch();

		if (key == 10 || key == 13) break;

		if (key == 27 || key == '`')
		{
			delwin(win);
			return -1;
		}

		switch(key)
		{
			case '\t':
				sel = (sel + 1) % p.size();
				break;
	
			case KEY_UP:
				sel = 0;
				break;
	
			case KEY_DOWN:
				sel = p.size() - 1;
				break;
	
			case KEY_LEFT:
				sel = (sel - 1 + p.size()) % p.size();
				break;
	
			case KEY_RIGHT:
				sel = (sel + 1) % p.size();
				break;
				
			case KEY_RESIZE:
				g_nLINES = LINES;
				g_nCOLS = COLS;
				break;
		}
	}
	delwin(win);

	return sel;

}

FormWindow::FormWindow(const char *title, int backColor)
	: _title(title), _shutdown(false), _confirm(false), _backColor(backColor), _width(10), _height(3)
{}
void FormWindow::SetBackColor(int color) { _backColor = color; }
void FormWindow::SetWidth(int width) { _width = width; }
void FormWindow::SetHeight(int height) { _height = height; }
WINDOW* FormWindow::GetWindow() { return _win; }

void FormWindow::Shutdown(bool confirm)
{
	_shutdown = true;
	_confirm = confirm;
}

void FormWindow::Draw(WINDOW *_win)
{
	wbkgd(_win, COLOR(COLOR_WHITE, _backColor));
	wattron(_win, A_BOLD);
	wborder(_win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(_win, A_BOLD);

	// title 출력
	setcol(COLOR_BLACK, COLOR_WHITE, _win);
	wmove(_win, 1, 1);
	whline(_win, ' ', _width-2);
	mvwprintw(_win, 1, (_width - krstrlen(_title))/2 , "%s", _title.c_str());
}

int FormWindow::Do()
{
	_win = newwin(_height, _width, (g_nLINES-_height)/2, (g_nCOLS-_width)/2);

	_shutdown = _confirm = false;
	while(!_shutdown)
	{// 출력
		Draw(_win);

		int key=getch();
		nodelay(stdscr, TRUE);
		int nKey2 = getch();
		int nKey3 = getch();
		int nKey4 = getch();
		nodelay(stdscr, FALSE);
		
		// 리모트 접속 End, Home 키
		// 27, 79, 72 (linux) ::  27, 91, 72 (vt100)
		if (key == 27 && (nKey2 == 91 || nKey2 == 79))
		{
			// 리모트 접속 End, Home 키
			if (nKey3 == 72) key = KEY_HOME;
			if (nKey3 == 70) key = KEY_END;
			// Konsole. End Home 키
			if (nKey3 == 49 && nKey4 == 126) key = KEY_HOME;
			if (nKey3 == 52 && nKey4 == 126) key = KEY_END;
		}
		
		if (key == KEY_RESIZE)
		{
			g_nLINES = LINES;
			g_nCOLS = COLS;
		}
		else
		{
			Proc(key);
		}
	}

	delwin(_win);
}

CheckBox::CheckBox(const char* title, int n, int backColor)
	: FormWindow(title, backColor), _sel(n)
{
	_width = 25;
	int len = krstrlen(_title);
	if (_width < len) _width = len;
	_bYN = true;
}

void CheckBox::SetItem(	std::vector<std::string> 	&s,
						std::vector<bool> 	 		&vCheckItem)
{
	_item = s;
	_vCheckItem = vCheckItem;
	CalcWidth();
}

void CheckBox::CalcWidth()
{
	int title_len = krstrlen(_title);
	if ( _width < title_len )  _width = title_len;

	for (int t=0; t<_item.size(); t++)
	{
		int len = krstrlen( _item[t] );
		if ( _width < len ) _width = len;
	}

	_width += 8; // 옆의 빈칸 두개.. 선 두개.. + [1]
}

void CheckBox::Draw(WINDOW *_win)
{
	FormWindow::Draw(_win);

	for (int t=0; t<_item.size() ;++t)
	{
		if (_sel == t && _sel != -1)
			setcol(_backColor, COLOR_WHITE, _win);
		else
			setcol(COLOR_WHITE, _backColor, _win);
		wmove(_win, 3 + t, 1);
		whline(_win, ' ', _width-2);
		if (_vCheckItem[t] == true)
			wprintw(_win, " [x] %s", _item[t].c_str());
		else
			wprintw(_win, " [ ] %s", _item[t].c_str());
	}

	wmove (_win, 4+_item.size(), (_width-17)/2);
	if (_bYN == true && _sel == -1)
		wattron (_win, COLOR(_backColor, COLOR_WHITE));
	else
	{
		wattron (_win, COLOR(COLOR_WHITE, _backColor));
		wattron (_win, A_BOLD);
	}

	wprintw(_win, (char*)strmid(8, gettext("Yes")).c_str());
	wattroff(_win, A_BOLD);
	if (_bYN == false && _sel == -1)
		wattron (_win, COLOR(_backColor, COLOR_WHITE));
	else
	{
		wattron (_win, COLOR(COLOR_WHITE, _backColor));
		wattron (_win, A_BOLD);
	}	
	
	wprintw(_win, (char*)strmid(8, gettext("No")).c_str());
	wattroff(_win, A_BOLD);
	wmove (_win, 5+_item.size(), 5);	
	wrefresh(_win);
}

void CheckBox::Proc(int key)
{
	switch(key)
	{
		case ' ':
		case 13:
		case 10:
			if (_sel != -1) 
				_vCheckItem[_sel] = !_vCheckItem[_sel];
			else 
				Shutdown(true);
			break;
		case 27:
			_bYN = false;
			Shutdown(true);
			break;
		case '\t':
			if (_sel == -1 && _bYN == true) { _sel = -1; _bYN = false; break; }
			if (_sel == -1 && _bYN == false) { _sel = 0; break; }
			if (_sel != -1) { _sel = -1; _bYN = true; break; }
			break;
		case KEY_LEFT:
			if (_sel == -1) {
				_bYN = !_bYN;
				break;
			}
			_sel = 0;
			break;
		case KEY_RIGHT:
			if (_sel == -1) {
				_bYN = !_bYN;
				break;
			}
			_sel = _item.size() - 1;
			break;	
		case KEY_UP:
			if (_sel == -1) {
				_sel = 0;
				break;
			}
			_sel = (_sel - 1 + _item.size()) % _item.size();
			break;
		case KEY_DOWN:
			if (_sel == -1) {
				_sel = 0;
				break;
			}
			_sel = (_sel + 1) % _item.size();
			break;
	}
}

vector<bool> CheckBox::Do()
{
	_height = _item.size() + 6;
	FormWindow::Do();
	if (_bYN == true) 
		return _vCheckItem;
	
	_vCheckItem.clear();
	return _vCheckItem;	
}

ListBox::ListBox(const char *title, int n, int backColor)
	: FormWindow(title, backColor), _sel(n)
{
	_width = 20;

	int len = krstrlen(_title);
	if (_width < len) _width = len;
}

ListBox::ListBox(const char *title, int n, int backColor, std::vector<std::string> &v)
	: FormWindow(title, backColor), _item(v), _sel(n)
{
	_width = 20;
	CalcWidth();
}

void ListBox::SetItem(std::vector<std::string> &s)
{
	_item = s;
	CalcWidth();
}
void ListBox::SetDefault(int sel) { _sel = sel; }
void ListBox::SetMinWidth(int width)
{
	SetWidth(width);
	CalcWidth();
}

void ListBox::CalcWidth()
{
	int title_len = krstrlen(_title);
	if ( _width < title_len )  _width = title_len;

	for (int t=0; t<_item.size(); t++)
	{
		int len = krstrlen( _item[t] );
		if ( _width < len ) _width = len;
	}

	_width += 8; // 옆의 빈칸 두개.. 선 두개.. + [1]
}

void ListBox::Draw(WINDOW *_win)
{
	FormWindow::Draw(_win);

	for (int t=0; t<_item.size() ;++t)
	{
		if (_sel == t)
			setcol(_backColor, COLOR_WHITE, _win);
		else
			setcol(COLOR_WHITE, _backColor, _win);

		wmove(_win, 3 + t, 1);
		whline(_win, ' ', _width-2);

		if (t<10)
			wprintw(_win, " [%d] %s", (t+1) % 10, _item[t].c_str());
		else
			wprintw(_win, "     %s", _item[t].c_str());
	}

	wrefresh(_win);
}

void ListBox::Proc(int key)
{
	if ( '1' <= key && key <= '9' && key - '1' < _item.size())
	{
		_sel = key - '1';
		Shutdown(true);
		return;
	}

	if (key == '0' && 9 < _item.size() )
	{
		_sel = 9;
		Shutdown(true);
		return;
	}

	switch(key)
	{
	case '\t':
		_sel = (_sel + 1) % _item.size();
		break;

	case KEY_LEFT:
		_sel = 0;
		break;

	case KEY_RIGHT:
		_sel = _item.size() - 1;
		break;

	case KEY_UP:
		_sel = (_sel - 1 + _item.size()) % _item.size();
		break;

	case KEY_DOWN:
		_sel = (_sel + 1) % _item.size();
		break;

	case 10:
	case 13:
		Shutdown(true);
		break;

	case 27:
	case '`':
		Shutdown(false);
		break;	

	case KEY_MOUSE:
		MEVENT event;
		getmouse(&event);

		if (event.bstate & BUTTON1_CLICKED)
		{
			Shutdown(false);
			break;
		}		
		break;
	}
}

int ListBox::Do()
{
	_height = _item.size() + 4;
	FormWindow::Do();

	if (_confirm) return _sel;
	else return -1;
}

TextBox::TextBox(const char *title)
	: FormWindow(title, COLOR_BLUE), _nCul(0), _nFirstLine(0), _nViewSel(0), _bLineNumView(0)
{
	_width = 40;
	_height = 20;
	_file = "";
	_nTabSize = 8;
	int len = krstrlen(_title);
	if (_width < len) _width = len;
	_nFontColor = COLOR_WHITE;
}

TextBox::TextBox(const char *title, int height, int width,  int backColor)
	: FormWindow(title, COLOR_BLUE), _nCul(0), _nFirstLine(0), _nViewSel(0), _bLineNumView(0)
{
	_width = width;
	_height = height;
	_file = "";
	_nTabSize = 8;
	int len = krstrlen(_title);
	if (_width < len) _width = len;
	_nFontColor = COLOR_WHITE;
}

bool TextBox::Load(const char* file, int nTabSize)
{
	_file = file;
	ifstream in(file);
	if (!in) return false;
	if (!in.good()) return false;

	string line, var, val;
	_nCulLimit = 0;
	int		nSize = 0;

	string	sConvert, sSpaceTab;
	_nTabSize = nTabSize;
	ENCODING	eEncode = AUTO;
	_text.clear();

	do
	{
		getline(in, line);
		
		if (line == "")
		{
			_text.push_back(line);
			if (in.eof()) return true;
			continue;
		}

		sConvert = isKorCode(line, &eEncode);

		if (sConvert == "") return false;
		if (_nCulLimit < nSize)	_nCulLimit = krstrlen(sConvert);
		_text.push_back(sConvert);
	}
	while(!in.eof());
	return true;
}

void TextBox::SetText(std::vector<std::string> &s)
{
	_text = s;
	_nFirstLine = 0;
	CalcWidth();
	_nCulLimit = 0;
	_nTabSize = 8;
	for (int t=0; t<_text.size(); t++)
	{
		int len = krstrlen( _text[t] );
		if (_nCulLimit < len)	_nCulLimit = len;
	}
}

void TextBox::SetMinWidth(int width)
{
	SetWidth(width);
	CalcWidth();
}

void TextBox::CalcWidth()
{
	int title_len = krstrlen(_title);
	if ( _width < title_len )  _width = title_len;

	_width += 8; // 옆의 빈칸 두개.. 선 두개.. + [1]
}

void TextBox::Draw(WINDOW *_win)
{
	FormWindow::Draw(_win);

	int 	nViewLine;
	string 	sLineStr, sViewStr, sSpaceTab;
	char	sFormat[10];
	bool	bNextLine = true;

	nViewLine = _nFirstLine;
	if (nViewLine < 0) return;

	int nLineSize = itoa((int)_text.size()).size();
	int nViewWidth, nKrStrSize, nNum;

	if (_bLineNumView == true)
		nViewWidth = _width - (nLineSize+5);
	else
		nViewWidth = _width - 4;
	
	sSpaceTab.append(_nTabSize, ' ');

	for (int t=0; t< _height-5; t++)
	{
		setcol(_nFontColor, _backColor, _win);

		wmove(_win, 3 + t, 1);
		whline(_win, ' ', _width-2);

		if (nViewLine >= _text.size()) continue;
		
		if (bNextLine)
		{
			sLineStr = _text[nViewLine];
			sLineStr = Replace(sLineStr, "\t", sSpaceTab.c_str());
			nNum = 0;
		}
		
		if (_bNextLineView)
		{
			if (sLineStr != "")
			{
				nKrStrSize = krstrlen(sLineStr);
				if (nKrStrSize <= nViewWidth)
				{
					sViewStr = krstrncpy(sLineStr, 0, nKrStrSize);
					bNextLine = true;
				}
				else
				{
					sViewStr = krstrncpy(sLineStr, 0, nViewWidth);
					sLineStr = krstrncpy(sLineStr, nViewWidth, nKrStrSize-nViewWidth);
					bNextLine = false;
				}
			}
			else
			{
				sViewStr = " ";
				bNextLine = true;
			}
		}
		else
		{
			if (_nCul < krstrlen(sLineStr))
				sViewStr = krstrncpy(sLineStr, _nCul, nViewWidth);
			else
				sViewStr = " ";
		}

		if (_bLineNumView == true)
		{
			sprintf(sFormat, "%%%dd", nLineSize);
			setcol(COLOR_GREEN, _backColor, _win);
			mvwprintw(_win, 3+t, 1, sFormat, nViewLine+1);
			setcol(_nFontColor, _backColor, _win);
			mvwhline(_win, 3+t, nLineSize+1, VLINE, 1);
			
			if (_nViewSel == nViewLine)
				setcol(_backColor, _nFontColor, _win);
			else
				setcol(_nFontColor, _backColor, _win);
			mvwprintw(_win, 3+t, nLineSize+3, "%s", sViewStr.c_str());
		}
		else
		{
			if (_nViewSel == nViewLine)
				setcol(_backColor, _nFontColor, _win);
			else
				setcol(_nFontColor, _backColor, _win);
			mvwprintw(_win, 3+t, 2, "%s", sViewStr.c_str());
		}

		if (!_bNextLineView)
			nViewLine++;
		else
			if (bNextLine == true) nViewLine++;
	}

	_nLastLine = nViewLine-1;

	if (_bLineNumView == true)
	{
		setcol(COLOR_GREEN, _backColor, _win);
		mvwprintw(_win, _height-1, _width-20, "[ %d / %d ]",
							_nViewSel+1, (int)_text.size());
	}
	wrefresh(_win);
}

void TextBox::Proc(int key)
{
	int nLineSize = itoa((int)_text.size()).size();

	switch(key)
	{
	case KEY_HOME:
		_nViewSel = 0;
		_nFirstLine = 0;
		break;
	case KEY_END:
		_nFirstLine = _text.size()-(_nLastLine-_nFirstLine)-1;
		_nViewSel = _text.size()-1;
		break;

	// Ctrl+Left
	case 393:
		_nCul = 0;
		break;

	// Ctrl+Right
	case 402:
		if (_bLineNumView == true)
		{
			if (_nCulLimit > _width+(nLineSize+5))
				_nCul = _nCulLimit-_width+(nLineSize+5);
		}
		else
		{
			if (_nCulLimit > _width+4)
				_nCul = _nCulLimit-_width+4;
		}
		break;

	case KEY_LEFT:
		_nCul = _nCul - 1;
		if (_nCul < 0) _nCul = 0;
		break;

	case KEY_RIGHT:
		if (_bLineNumView == true)
		{			
			if (_nCulLimit > _width+(nLineSize+5))
			{
				_nCul = _nCul + 1;
				if (_nCul > _nCulLimit-_width+(nLineSize+5))
					_nCul = _nCulLimit-_width+(nLineSize+5);
			}
		}
		else
		{
			if (_nCulLimit > _width+4)
			{
				_nCul = _nCul + 1;
				if (_nCul > _nCulLimit-_width+4)
					_nCul = _nCulLimit-_width+4;
			}
		}
		break;
	case KEY_UP:
		if (_nViewSel <= _nFirstLine)
		{
			_nFirstLine--;
			if (_nFirstLine < 0) _nFirstLine = 0;
			_nViewSel = _nFirstLine;
		}
		else
		{
			if (_nViewSel > 0) _nViewSel--;
		}
		break;
	case KEY_DOWN:
		if (_nViewSel >= _nLastLine)
		{
			if (_nLastLine >= _text.size()-1)
			{
				_nFirstLine = _text.size()-(_nLastLine-_nFirstLine)-1;
				_nViewSel = _nLastLine;
			}
			else
			{
				_nFirstLine++;
				_nViewSel++;
			}
		}
		else
		{
			if (_nViewSel < _text.size()-1) _nViewSel++;
		}
		break;

	case KEY_PPAGE:
		for (int n=0; n< (_nLastLine-_nFirstLine); n++)
			Proc(KEY_UP);
		break;
	case KEY_NPAGE:
		for (int n=0; n< (_nLastLine-_nFirstLine); n++) 
			Proc(KEY_DOWN);
		break;

	case KEY_RESIZE:
		g_nLINES = LINES;
		g_nCOLS = COLS;
		break;

	case KEY_F(2):
		_bNextLineView = !_bNextLineView;
		break;

	case KEY_F(3):
		_bLineNumView = !_bLineNumView;
		break;

	case 10:
	case 13:
		Shutdown(true);
		break;

	case 27:
	case '`':
		Shutdown(false);
		break;	

	case KEY_MOUSE:
	{
		MEVENT event;
		getmouse(&event);

		LOG("Event.x [%d] Event.y [%d] Event [%u]",event.x, event.y, event.bstate);

		if (BUTTON_CLICK(event.bstate,1))
		{
			Shutdown(false);
		}
		break;
	}
	}
}

int TextBox::Do(bool bLineNumView, bool bNextLineView)
{
//	_height = 20;
	_bLineNumView = bLineNumView;
	_bNextLineView = bNextLineView;

	FormWindow::Do();

	if (_confirm) return _nFirstLine;
	else return -1;
}

///	@brief	Yes와 No가 있는 Box를 그리고 선택하게 한다.
///	@param	title		Yes No Box title
///	@param	YN			Yes나 No 기본 선택값
///	@param	back_color	배경 색상
/// @return	YN_Y/YN_N
int YNBox(const char *title, int YN, int back_color)
{
	int l_title = krstrlen(title);
	int width = l_title > 17 ? l_title : 17;

	width +=4;

	WINDOW *win = newwin(5, width, (g_nLINES-5)/2, (g_nCOLS-width)/2);

	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', width-2);

	mvwprintw(win, 1, (width - l_title)/2 , "%s", title);
	wattroff(win, A_BOLD);

	int key;

	//--Yes--- ---No---
	while(1)
	{// 출력
		wmove (win, 3, (width-17)/2);

		if (YN == YN_Y)
			wattron (win, COLOR(back_color, COLOR_WHITE));
		else
		{
			wattron (win, COLOR(COLOR_WHITE, back_color));
			wattron (win, A_BOLD);
		}

		wprintw(win, (char*)strmid(8, gettext("Yes")).c_str());
		wattroff(win, A_BOLD);

		wattron (win, COLOR(COLOR_WHITE, back_color));
		wattron (win, A_BOLD);
		waddch(win, ' ');
		wattroff(win, A_BOLD);

		if (YN == YN_N)
			wattron (win, COLOR(back_color, COLOR_WHITE));
		else
		{
			wattron (win, COLOR(COLOR_WHITE, back_color));
			wattron (win, A_BOLD);
		}

		wprintw(win, (char*)strmid(8, gettext("No")).c_str());
		wattroff(win, A_BOLD);

		wrefresh(win);

		key=getch();

		if (key == 13 || key == 10) break;

		if (key == 27)
		{
			delwin(win);
			return YN_N;
		}

		switch(key)
		{
		case '\t':
		case KEY_UP:
		case KEY_DOWN:
		case KEY_LEFT:
		case KEY_RIGHT:
			if (YN == YN_Y) YN=YN_N;
			else YN = YN_Y;
			break;
		}

	}
	delwin(win);

	return 	YN;
}

ProgressBox::ProgressBox(const char *_title, const char *_msg, int _back_color)
{
	title = _title;
	msg =_msg;
	back_color = _back_color;

	m_nCounter=0;

	win = newwin(8, 72, (g_nLINES-8)/2, (g_nCOLS-72)/2);
	redraw();

	sRightString = "";
	sLeftString = "";
}

///	@brief	상태 Box소멸자
ProgressBox::~ProgressBox()
{
	delwin(win);
}

///	@brief	상태 Box를 그리는 함수
void ProgressBox::redraw()
{
	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', 72-2);
	mvwprintw(win, 1, (72 - krstrlen(title))/2 , "%s", title);
	wattroff(win, A_BOLD);

	// msg 출력
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);
	mvwprintw(win, 3, (72 - krstrlen(msg))/2, "%s", msg);
	wattroff(win, A_BOLD);
}

void ProgressBox::show()
{
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);

	wmove(win, 5, 6);
	whline(win, ' ', 60);
	if (sLeftString != "") mvwprintw(win, 5, 6, "%s", sLeftString.c_str());
	if (sRightString != "") mvwprintw(win, 5, 72 - sRightString.size() -6, "%s", sRightString.c_str());
	wattroff(win, A_BOLD);

	//U, B 기본바탕
	wattron(win, COLOR(COLOR_WHITE, COLOR_BLACK));
	wattron(win, A_BOLD);
	wmove(win, 6, 6);
	whline(win, '.', 60);
	wattroff(win, A_BOLD);

	wattron(win, COLOR(COLOR_WHITE, COLOR_WHITE));
	wmove(win, 6, 6);
	whline(win, ' ', m_nCounter);
	wattroff(win, A_BOLD);
	wrefresh(win);
}

void ProgressBox::setCount(int nCount)
{
	if (nCount <= 0)
	{
		m_nCounter = 0;
		return;
	}

	if (nCount >= 100)
	{
		m_nCounter = 60;
		return;
	}

	m_nCounter = (int)(((double)nCount / 100.0) * 60.0);
}

///	@brief	상태 Box생성자
///	@param	_title		title 글자
///	@param	_msg		화면에 출력할 내용
///	@param	_back_color	배경 색상
StatBox::StatBox(const char *_title, const char *_msg, int _back_color)
{
	title = _title;
	msg =_msg;
	back_color = _back_color;

	UL = NULL;
	UR = NULL;
	BL = NULL;
	BR = NULL;

	U=0;
	B=0;

	win = newwin(12, 72, (g_nLINES-12)/2, (g_nCOLS-72)/2);
	redraw();
}

///	@brief	상태 Box소멸자
StatBox::~StatBox()
{
	delwin(win);
}

///	@brief	상태 Box를 그리는 함수
void StatBox::redraw()
{
	wbkgd(win, COLOR(COLOR_WHITE, back_color));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', 72-2);
	mvwprintw(win, 1, (72 - krstrlen(title))/2 , "%s", title);
	wattroff(win, A_BOLD);

	// msg 출력
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);
	mvwprintw(win, 3, (72 - krstrlen(msg))/2, "%s", msg);
	wattroff(win, A_BOLD);
}

//-
//타이틀
//
//To c:
//
//|  Autoexec
//----...
//
//<21/132>
//----...
//
//---

void StatBox::show()
{
	wattron (win, COLOR(COLOR_WHITE, back_color));
	wattron (win, A_BOLD);
	// UL UR 출력

	wmove(win, 5, 6);
	whline(win, ' ', 60);
	if (UL) mvwprintw(win, 5, 6, "%s", UL);
	if (UR) mvwprintw(win, 5, 72 - krstrlen(UR) -6, "%s", UR);

	// BL BR 출력
	wmove(win, 8, 6);
	whline(win, ' ', 60);
	if (BL) mvwprintw(win, 8, 6, "%s", BL);
	if (BR) mvwprintw(win, 8, 72 - krstrlen(BR) -6, "%s", BR);

	wattroff(win, A_BOLD);

	//U, B 기본바탕
	wattron(win, COLOR(COLOR_WHITE, COLOR_BLACK));
	wattron(win, A_BOLD);
	wmove(win, 6, 6);
	whline(win, '.', 60);
	wmove(win, 9, 6);
	whline(win, '.', 60);
	wattroff(win, A_BOLD);

	// U, B 기본이 아닌 바탕
	wattron(win, COLOR(COLOR_WHITE, COLOR_WHITE));

	wmove(win, 6, 6);
	whline(win, ' ', U);
	wmove(win, 9, 6);
	whline(win, ' ', B);

	wrefresh(win);
}

} // namespace
