/* *************************************************************************
 *   Copyright (C) 2004 by Byoungyoung, La                                 *
 *   la9527@yahoo.co.kr                                                    *
 ************************************************************************* */

/// \file	ZipReader.cpp
///	\brief	ZipReader.cpp

#include "ZipReader.h"
 
using namespace MLS;

///	\brief	생성자
ZipReader::ZipReader()
{
	m_pTargzUtil = NULL;
	
	m_sDir = "";
	m_sFilename = "";
	m_sFilePath = "";
	
	m_currentPath = "";
	m_CurrentData = 0;
	
	m_ZipCurrentDir = "";
	m_Filelist.clear();
	
	m_Count = 0;
}

///	\brief	소멸자
ZipReader::~ZipReader()
{
	if (m_pTargzUtil) delete m_pTargzUtil;
}

///	\brief	현재 directory가 '/'인지 알아보는 함수
///	\return	'/' 여부
bool ZipReader::isRoot(void)
{
	if (m_currentPath.length() == 0) return true;
	return false;
}

// file reader
///	\brief	초기화 함수
///	\param	path	zip file path	
///	\return	초기화 성공 여부
bool ZipReader::Init(const string &path)
{
	if (m_sFilename == "") return false;
	
	vector<string>		vErrorString;
	
	m_CurrentData = 0;

	noraw();
	cbreak();    // Ctrl+C 를 가능하게 하기 위해서
	
	
	string sPath = path;
	m_ZipCurrentDir = path;
	
	if (m_Count == 0)
	{
		Progress 	tProgress(gettext("Wait"), gettext("Please wait !!! - Cancel Key [Ctrl+C]"));

		try
		{
			m_pTargzUtil = new TargzUtil(m_sDir, m_sFilename, 0);
					
			m_pTargzUtil->SetCounter(0);
			tProgress.Start((Counter*)m_pTargzUtil, JOIN);
			
			if (m_pTargzUtil->FileListRead() == SUCCESS)
			{
				LOG("FileListRead success...");
				
				if (m_pTargzUtil->GetDir_Files(sPath, &m_Filelist) == ERROR)
				{
					LOG("File Count :: %d", m_Count);
					m_pTargzUtil->SetCounter(100);
					tProgress.SetRun(PTHREAD_END);
					tProgress.End();
					nocbreak();    // Ctrl+C 를 가능하게 하기 위해서
					raw();
					return false;
				}
			}
			else
			{
				throw Exception("FileListRead failure...");
			}			
		}
		catch(Exception& ex)
		{
			m_pTargzUtil->SetCounter(100);
			tProgress.SetRun(PTHREAD_END);
			tProgress.End();

			MsgBox(COLOR_RED, gettext("Error"), "%s", (char*)ex);
			nocbreak();    // Ctrl+C 를 가능하게 하기 위해서
			raw();
			return false;
		}

		m_pTargzUtil->SetCounter(100);
		tProgress.SetRun(PTHREAD_END);
		tProgress.End();

		m_sFilePath = m_sDir;
	}
	else
	{
		if (m_pTargzUtil->GetDir_Files(sPath, &m_Filelist) == ERROR)
		{
			LOG("File SIZE :: %d",  m_Filelist.size());
			nocbreak();    // Ctrl+C 를 가능하게 하기 위해서
			raw();
			return false;
		}
	}
	
	m_Count++;
	
	m_pTargzUtil->GetErrorList(vErrorString);
	/*
	if (vErrorString.size() > 0)
	{
		MsgBox(COLOR_RED, "Error", "%s", vErrorString[0].c_str());
		return ERROR;
	}
	*/
	LOG("File List :: %d", m_Filelist.size());
	nocbreak();    // Ctrl+C 를 가능하게 하기 위해서
	raw();
	return true;
}

///	\brief	다음 파일로 이동한다.
///	\return	다음 파일로 이동 성공 여부
bool ZipReader::Next(void)
{
	//LOG("m_CurrentData :: %d, %d", m_CurrentData, m_Filelist.size());
	
	if (m_CurrentData < m_Filelist.size())
	{
		m_CurrentData++;
		return true;
	}
	return false;
}

///	\brief	현재 파일 정보를 얻는다
///	\param	tFile	반환값
///	\return	파일정보 얻기 성공 여부
bool ZipReader::GetInfo(File& tFile)
{
	File*	pFile;
	
	// . 현재 선택된 파일이 유효하면
	if (m_CurrentData-1 < m_Filelist.size())
	{
		// .. filelist에서 파일 정보를 가져옴
		pFile = m_Filelist[m_CurrentData-1];
		tFile.clear();
		
		// .. filename얻기
		if (m_ZipCurrentDir == pFile->FullName)
			tFile.name = "..";
		else
			tFile.name = pFile->name;

		tFile.TarFullName = pFile->TarFullName;
		tFile.FullName = pFile->FullName;
		tFile.bDir = pFile->bDir;
		tFile.color = pFile->color;
		strncpy(tFile.date, pFile->date, 9);
		strncpy(tFile.attr, pFile->attr, 11);
		strncpy(tFile.time, pFile->time, 6);
		tFile.selected = pFile->selected;
		tFile.size = pFile->size;
		tFile.mtime = pFile->mtime;
		/*
		LOG("%s, %s, %d, %s, %s, %s, %ld", 	tFile.name.c_str(), 
										tFile.FullName.c_str(),
										tFile.bDir,
										tFile.date,
										tFile.time,
										tFile.attr,
										tFile.size);
		*/
	}
	else
		return false;
	return true;
}

///	\brief	file copy 함수
///
/// 압축된 파일 압축 풀어서 복사
///	\param	tCpyData	복사할려는 파일명
///	\param	sTarget		복사할 대상 directory
///	\return	SUCCESS	:	성공.\n
///			ERROR	:	실패.
int ZipReader::Copy(vector<string>& tCpyData, string& sTarget)
{
	noraw();
	cbreak();    // Ctrl+C 를 가능하게 하기 위해서

	try
	{
		if (m_pTargzUtil->Uncompress(tCpyData, sTarget) == ERROR)
		{
			nocbreak();
			raw();
			return ERROR;
		}
	}
	catch(Exception& ex)
	{
		MsgBox(COLOR_RED, gettext("Error"), "%s", (char*)ex);
		nocbreak();
		raw();
		return ERROR;
	}
	nocbreak();
	raw();
	return SUCCESS;
}

///	\brief	file 지우기
///
/// 압축된 파일 지우기
///	\param	tRemoveData	지울 filename
///	\return	SUCCESS	:	성공.\n
///			ERROR	:	실패.
int ZipReader::Remove(vector<string>& tRemoveData)
{
	noraw();
	cbreak();    // Ctrl+C 를 가능하게 하기 위해서
	try
	{
		if (m_pTargzUtil->Compress(tRemoveData, TAR_DELETE) == ERROR)
		{
			nocbreak();
			raw();
			return ERROR;
		}
	}
	catch(Exception& ex)
	{
		MsgBox(COLOR_RED, gettext("Error"), "%s", (char*)ex);
		nocbreak();
		raw();
		return ERROR;
	}
	nocbreak(); 
	raw();
	return SUCCESS;
}

///	\brief	Zip 파일을 보기 위한 함수
///
///	Sync를 위한 세팅
///	\param	tFile	sTmpDir 밑에 압축 풀 파일 
///	\param	sTmpDir Tmp 디렉토리에 압축 푼다.
///	\return	SUCCESS	:	성공.\n
///			ERROR	:	실패.
int ZipReader::View(File&	pFile, const string& sTmpDir)
{
	vector<File*>	vFileList;
	vFileList.push_back(&pFile);

	noraw();
	cbreak();    // Ctrl+C 를 가능하게 하기 위해서
	
	try
	{
		Progress 	tProgress(gettext("Wait"), gettext(	"Please wait !!! - Cancel Key [Ctrl+C]"));
				
		m_pTargzUtil->SetCounter(0);
		tProgress.Start((Counter*)m_pTargzUtil, JOIN);
		if (m_pTargzUtil->Uncompress(vFileList, "/tmp") == ERROR)
		{
			m_pTargzUtil->SetCounter(100);
			tProgress.SetRun(PTHREAD_END);
			tProgress.End();
			nocbreak();
			raw();
			return ERROR;
		}
		m_pTargzUtil->SetCounter(100);
		tProgress.SetRun(PTHREAD_END);
		tProgress.End();
	}
	catch(Exception& ex)
	{
		MsgBox(COLOR_RED, gettext("Error"), "%s", (char*)ex);
		nocbreak();
		raw();
		return ERROR;
	}
	nocbreak();
	raw();
	return SUCCESS;
}

///	\brief	종료 함수
void ZipReader::Destroy(void)
{
	if (m_sFilePath != "")
		chdir(m_sFilePath.c_str());
	
	if (m_pTargzUtil) delete m_pTargzUtil; m_pTargzUtil = NULL;

	m_sDir = "";
	m_sFilename = "";
	
	m_currentPath = "";
	m_CurrentData = 0;
	m_sFilePath = "";
	m_ZipCurrentDir = "";
	m_Filelist.clear();
	
	m_Count = 0;
}
