///	\file	panel_read.cpp
///	\brief	panel_read.cpp
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <string>
#include <map>
#include <algorithm>
#include <unistd.h>

#include "panel.h"
#include "strutil.h"
#include "dialog.h"
#include "ZipReader.h"

using namespace std;
using namespace strutil;
using namespace MLS;

template<class DirSort, class FileSort >
class sort_dir_adaptor 
{
public:
	bool operator() (File *a, File *b)
	{
	// dir 이면 제일 앞으로...  
	// .으로 시작하는 디렉토리는 항상 앞에..
	// ".."이 제일 앞에 오도록..
	
		if (a->bDir && b->bDir)
		{
			if (a->name == "..") return true;
			if (b->name == "..") return false;
		
			DirSort _ds;
			return _ds(a, b);
		}
		else if (a->bDir)	return true;
		else if (b->bDir)	return false;
		else
		{
			FileSort _fs;
			return _fs(a, b);
		}
	
	}		
};

///	@brief	file color를 비교하는 class
class sort_color 
{
	public:
	/// @brief	operator () 정의
						///
	/// file color로 sort시 필요한 함수로써 두개의 file color을 비교한다
						///
	/// @param	*a	첫번째 파일
	/// @param	*b	두번째 파일
	/// @return	true/false
	bool operator()(File *a, File *b)
	{		
		if (*a->color==*b->color)	return a->name < b->name;

		if (*a->color == g_Color.Default) return false;
		if (*b->color == g_Color.Default) return true;

		return *a->color < *b->color;
	}

};

/// @brief	file name으로 비교하는 class
class sort_name 
{
	public:

	/// @brief	operator () 정의
						///
	/// filename으로 sort시 필요한 함수로써 두개의 filename을 비교한다
						///
	/// @param	*a	첫번째 파일
	/// @param	*b	두번째 파일
	/// @return	true/false
	bool operator()(File *a, File *b)
	{
		return a->name < b->name;		
	}
};


///	@brief	파일 확장자로 파일을 비교하는 class
class sort_ext 
{
	public:
	/// @brief	operator () 정의
						///
	/// 확장자 sort시 필요한 함수로써 두개의 파일확장자를 비교한다
						///
	/// @param	*a	첫번째 파일
	/// @param	*b	두번째 파일
	/// @return	true/false
	bool operator()(File *a, File *b)
	{
		if (a->ext()==b->ext())	return a->name < b->name;
		return a->ext() < b->ext();
	}
};

///	@brief	file size로 비교하는  class
class sort_size
{
	public:
	/// @brief	operator () 정의
						///
	/// file size로 sort시 필요한 함수로써 두개의 file size를 비교한다
	/// @param	*a	첫번째 파일
	/// @param	*b	두번째 파일
	/// @return	true/false
	bool operator()(File *a, File *b)
	{
		if (a->size==b->size)	return a->name < b->name;
		return a->size < b->size;	
	}
};

///	@brief	파일을 최근 수정 시간을 비교하는 class
class sort_time 
{
	public:
	/// @brief	operator () 정의
						///
	/// file mtime(time of last modification)으로 sort시 필요한 함수로써 두개의 
	/// file mtime를 비교한다
	/// @param	*a	첫번째 파일
	/// @param	*b	두번째 파일
	/// @return	true/false
	bool operator()(File *a, File *b)
	{
		if (a->mtime==b->mtime)	return a->name < b->name;
		return a->mtime < b->mtime;
	}
};


// read interface
/*
 * (기본) 파일 리더.
 * file:// 파일 리더
 * arc://
 * ftp:// 
 */
 
/// \brief	초기화 함수
void Panel::Init(void)
{
	_cur      = 0;
	_page     = 0;
	_sel_size = 0;
	_sel_num  = 0;	

	_nfile    = 0;
	_ndir     = 0;
	_dir_size = 0;	
	
	_file_list.clear();
	_pool.Clear();	
}

/// \brief	파일명에 해당 되는 색상을 구함
/// \param	t		파일
/// \return	색상
ColorEntry *Panel::GetColor(File &t)
{
	// 이름
	map<string, ColorEntry>::iterator namei = g_Color.mapName.find( tolower(t.name) );
	
	if (namei != g_Color.mapName.end() ) 
		return &(*namei).second;
	
	// 확장자
	map<string, ColorEntry>::iterator exti = g_Color.mapExt.find( tolower(t.ext()) );
	
	if (exti != g_Color.mapExt.end() ) 
		return &(*exti).second;
	
	// 퍼미션
	int mode=0;
	
	for (int z=1; z<10; z++)
	{
		mode <<= 1;
		mode +=	t.attr[z] == '-' ? 0 : 1; 		
	}	
	for (map<int , ColorEntry>::iterator mi = g_Color.mapMask.begin(); mi != g_Color.mapMask.end(); ++mi)
		if (mode & (*mi).first) return &(*mi).second;				
	
	
	return &g_Color.Default;	
}

void Panel::Sort(void)
{
	pFileIterator  begin = _file_list.begin(),
		           end   = _file_list.end();
		
	if (_bDirSort)
	{
		switch(_eSortMethod)
		{
			case SORT_NAME:
				sort(begin, end, sort_dir_adaptor<sort_name, sort_name>()); break;
			case SORT_EXT: 
				sort(begin, end, sort_dir_adaptor<sort_name, sort_ext>()); break;
			case SORT_SIZE: 
				sort(begin, end, sort_dir_adaptor<sort_name, sort_size>()); break;
			case SORT_TIME: 
				sort(begin, end, sort_dir_adaptor<sort_name, sort_time>()); break;
			case SORT_COLOR: 
				sort(begin, end, sort_dir_adaptor<sort_name, sort_color>()); break;
			case SORT_NONE:
			default:
				return;
		}
	}
	else
	{
		switch(_eSortMethod)
		{
			case SORT_NAME: sort(begin, end, sort_name()); break;
			case SORT_EXT: sort(begin, end, sort_ext()); break;
			case SORT_SIZE: sort(begin, end, sort_size()); break;
			case SORT_TIME: sort(begin, end, sort_time()); break;
			case SORT_COLOR: sort(begin, end, sort_color()); break;
			case 0:
			default:
				return;
		}
	}
}

void Panel::Refresh(bool bReload)
{
	int tmp;
	string cur_name = _file_list[_cur]->name;
	
	if (isZip() == true)
	{
		string sTmpPath = GetZipCurrentPath();
		sTmpPath = sTmpPath.substr(1, sTmpPath.size()-1);

		if (bReload == true)
		{
			string sTmpZipFile = _sZipFile;
			
			// new된 Zip 을 삭제
			_tReader->Destroy();
			_nZip = 0;
	
			Read(GetCurrentPath());
	
			_nZip = 1;
			_sZipFile = sTmpZipFile;
			_zipPath = _sZipFile + " :: /";
			if (Read("") == false)	_nZip = 0;
		}
		
		if (Read(sTmpPath) == false)
		{
			sTmpPath = "/" + sTmpPath;
			Read(sTmpPath);
		}
	}
	else
	{
		// GetCurrentPath로 하게되면 링크된 디렉토리가 유지되지 않습니다.
		Read(".");
	}
	
	_cur = Search(cur_name);
	if (_cur == -1) _cur = 0;
}

File Panel::GetCurrentFile(void)
{
	return *_file_list[_cur];
}

string Panel::ViewCurrentPath(void)
{
	if (_nZip == 0)
		return _sTmpPath;
	else
		return _zipPath;
}

/// \brief	현재 dir 절대 path를 구한다.
/// \return	현재 dir 절대 path
string Panel::GetCurrentPath(void)
{
	string 	s_path;
	
	// . 현재 dir name을 구한다.
	char *tmpdir  = getcwd(NULL, 0);
	if (tmpdir)
	{
		s_path = tmpdir;
		free(tmpdir);
			
		// . 끝에 '/'를 추가한다.
		if (s_path != "/") s_path += '/';
	}
	else
	{
		// 현재 디렉토리 읽기 오류면 home 디렉토리 지정
		struct passwd *pw = getpwuid(getuid()); // getuid이냐 geteuid냐...;
		s_path = s_path + pw->pw_dir + '/';
	}
	return s_path;
}

/// \brief	상대경로는 절대경로로 바꾼다.
/// \param	str		상대경로 path명
///	\return	절대경로명
string Panel::GetRealPath(const string &str)
{
	string s_path;
	
	s_path = str;
	
	// . zip 파일 검색이 아니면
	if (_nZip == 0)
	{
		// ~ 인지 먼저 판단.
		if (s_path[0]=='~')
		{			
			if (s_path == "~" || s_path[1] == '/')
			{
				// . home 디렉토리 지정
				struct passwd *pw = getpwuid(getuid()); // getuid이냐 geteuid냐...;
				string home  = pw->pw_dir;
				home += '/';
				string s_tmp = s_path.substr(1);
				string s_tmp2 = "";
				if (s_tmp.find('/') != -1)
					s_tmp2 = s_tmp.substr(s_tmp.find('/')+1);
				s_path = home + s_tmp2;
			}
			else
			{	
				string s_tmp = s_path.substr(1);
				string s_tmp2 = "", s_tmp3 = "";
				if (s_tmp.find('/') != -1)
					s_tmp2 = s_tmp.substr(0, s_tmp.find('/'));
				else
					s_tmp2 = s_tmp;
				if (s_tmp.find('/') != -1)
					s_tmp3 = s_tmp.substr(s_tmp.find('/')+1);
				struct passwd *pw = getpwnam(s_tmp2.c_str());
				if (pw == NULL) return "";
				string home = pw->pw_dir;
				home += '/';
				s_path = home + s_tmp3;
			}
		}
		else if (s_path[0]!='/') // .. /(root) dir이 아니면
		{
			// ... 현재 dir를 얻어야 한다면 현재 dir절대경로를 찾는다. 
			if (s_path == ".") s_path = _path.empty() ? GetCurrentPath() : _path;
			// ... 상위 dir를 얻어야 한다면
			else if (s_path == "..")
			{
				// .... /(root)dir이 아니면 상위 절대경로 dir명을 구함
				if (_path != "/")
				{
					string::size_type p = _path.rfind('/', _path.size()-2);
					s_path = _path.substr(0, p+1);
				}			
				/// .... /(root)dir이면 절대경로에 / 를 넣음
				else s_path = _path;
			}			
			else // ... 그외 dir은 path에 추가하여 절대경로를 구한다.
			{
				s_path = _path + s_path;
			}
		}
	}
	// . zip 파일 이면
	else
	{
		// . 현재 절대경로를 구한다.
		s_path = GetCurrentPath();
	}
	if (s_path.substr(s_path.size()-1, 1) != "/") s_path += '/';
	_sTmpPath = s_path;
	return s_path;
}

/// \brief	디렉토리에서 파일이름을 읽는다.
///
///  path 뒤에는 항상 /가 붙는 것을 원칙으로 합니다. (예 /home/iddd/ )
/// \param	dir_name	읽을 디렉토리명
///	\return	성공여부
bool Panel::Read(const string &dir_name)
{
	string prev_dir;
	string strPath;
	
	// . 읽을 디렉토리가 상위 디렉토리이고 현재 path가 / 가아니면
	if (dir_name == ".."  && _path != "/")
	{
		// .. 현재 path를 prev_dir에 저장
		string::size_type p = _path.rfind('/', _path.size()-2);
		prev_dir = _path.substr(p+1, _path.size()-p-2);
	}

	// . 상대경로를 절대경로로 바꾼다.
	strPath  = GetRealPath(dir_name);
	
	//MsgBox("Error", strPath.c_str());
	// 절대경로에 맞는 Reader를 얻어온다
	_tReader = GetReader(strPath);
	
	if (_tReader == NULL) return false;
	
	// . zip 파일이 아니면
	if (_nZip == 0)
	{
		if (!_tReader->Init(strPath)) return false;// .. Init 은 항상 절대경로로.
	}
	// . zip 파일이면
	else
	{
		LOG("Zip File INIT :: %s", dir_name.c_str());
		
		if (!_tReader->Init(dir_name)) // Zip 은 dir_name으로	
		{
			//MsgBox("ERROR", "Zip file reading failure!!!");
			return false;
		}
	}

	_bRoot = _tReader->isRoot();
	
	_path  = strPath;
	
	// . History에 현재 dir name 저장
	_tHistory.AddEntry(_path);
	
	// . panel init
	Init();
	
	while(_tReader->Next())
	{
		File &t = _pool.Get();
		
		if (!_tReader->GetInfo(t)) continue;
		
		if (_nZip == 0)
		{
			if (t.name == ".")	continue;
			if (_bRoot && t.name == "..") continue;
			if (!_bShowHidden && t.name[0] == '.' && t.name != ".." ) continue;
		}
		else
		{
			//if (!e_tConf.ShowHidden && t.name[0] == '.' && t.name != ".." ) continue;
		}
		
		_dir_size += t.size;
		
		t.selected = false;
		t.color    = GetColor(t);
		
//		LOG("FileList Data :: %s", t.name.c_str());
		_file_list.push_back(&t);
		
		if (t.bDir) _ndir++;
		else _nfile++;
	}
	
	Sort();	
	
	// . 바로 전 dir 에 커서를 위치시킨다.
	if (!prev_dir.empty()) // ..에 해당하는 디렉토리를 찾는다.
	{
		int tmp;
		if ((tmp = Search(prev_dir)) != -1) _cur = tmp;
	}
	
	return true;
}
