/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "utils.h"
#include <osip/sdp.h>
#include "osipmanager.h"
#include "resolver.h"
#include "uatransaction.h"
#include <sys/param.h>



void
respond_to_request (osip_t * config, transaction_t * transaction, int code)
{
	sip_t *response;
	int i;
	i = osip_dialog_generate_response_default( NULL,
						  code, transaction->orig_request,&response);
	if (i!=0)
	  return ;
	osip_send_response(def_manager,transaction,response);
}


void
osip_send_response (OsipManager * mgr, transaction_t * transaction,
		    sip_t * response)
{
	via_t *via;
	sipevent_t *sipevent;
	/* we must check if first via has x.x.x.x address. If not, we must resolve it */
	msg_getvia (response, 0, &via);
	if (via == NULL)
	{
		osip_trace (OSIP_WARNING,
			    ("Cannot send response because it has no via field"));
		msg_free(response);
		sfree(response);
		return;
	}

	sipevent = osip_new_outgoing_sipmessage (response);
	sipevent->transactionid = transaction->transactionid;

	/*
	  AMD: the receiver of SIP REQUESTs checks the via header when
	  the message is received. If the Via in the host is not the same
	  than the sender at the UDP message layer, then the 'received' 
	  parameter is added with the ip from the UDP layer. Then,
	  we are sure to send the response to the receiver and we
	  are sure we know the ip in the SIP request.  :)
	  if (inet_addr (host) == -1)
	  {
	  async_resolv_and_send (def_manager, transaction, sipevent);
	  return;
	  }
	*/
	ua_transaction_execute(transaction,sipevent);
}



int
async_resolv_and_send (OsipManager * manager, transaction_t * transaction,
		       sipevent_t * sipevent)
{
	/* put the transaction,event on the fifo of the resolver thread */
	async_resolv_t *te;

	osip_trace (OSIP_INFO1, ("info: Name resolution requested.\n"));
	if (manager->resolv_run_cond == 0)
		osip_manager_start_resolver (manager);
	te = smalloc (sizeof (async_resolv_t));
	te->transaction = transaction;
	te->event = sipevent;
	te->dest = NULL;
	te->port = 0;
	te->ackmsg = NULL;
	fifo_add (manager->resolv_fifo, (void *) te);
	return transaction->transactionid;
}

int
async_resolv_and_send_ack (OsipManager * manager, dialog_t *dialog, char *dest, int port,
			   sip_t * ackmsg)
{
	/* put the transaction,event on the fifo of the resolver thread */
	async_resolv_t *te;

	osip_trace (OSIP_INFO1, ("info: Name resolution requested.\n"));
	if (manager->resolv_run_cond == 0)
		osip_manager_start_resolver (manager);
	te = smalloc (sizeof (async_resolv_t));
	te->transaction = NULL;
	te->event = NULL;
	te->dest = sgetcopy (dest);
	te->port = port;
	te->ackmsg = ackmsg;
	te->dialog=dialog;
	fifo_add (manager->resolv_fifo, (void *) te);
	return 0;
}



/*search element el in list, and remove it*/
int
list_remove_el (list_t * list, void *el)
{
	int pos = 0;
	int cond = 0;
	node_t *nd;

	nd = list->node;
	while ((pos < list->nb_elt) && (!cond))
	{
		if (nd->element == el)
			cond = 1;
		else
		{
			pos++;
			nd = nd->next;
		}
	}
	if (cond)
		list_remove (list, pos);
	else
		return (-1);
	return (0);
}

void
callid_getcopy (call_id_t * dest, call_id_t * source)
{
	dest->number = smalloc (strlen (source->number) + 1);
	strcpy (dest->number, source->number);
	dest->host = smalloc (strlen (source->host) + 1);
	strcpy (dest->host, source->host);
}

void
list_free (list_t * li, void (*destructor) (void *))
{
	int pos = 0;
	void *elem;
	while (!list_eol (li, pos))
	{
		elem = list_get (li, pos);
		list_remove (li, pos);
		if (destructor != NULL)
			destructor (elem);
	}
}



char *
make_message (const char *fmt, ...)
{
	/* Guess we need no more than 100 bytes. */
	int n, size = 100;
	char *p;
	va_list ap;
	if ((p = smalloc (size)) == NULL)
		return NULL;
	while (1)
	{
		/* Try to print in the allocated space. */
		va_start (ap, fmt);
		n = vsnprintf (p, size, fmt, ap);
		va_end (ap);
		/* If that worked, return the string. */
		if (n > -1 && n < size)
			return p;
		//printf("Reallocing space.\n");
		/* Else try again with more space. */
		if (n > -1)	/* glibc 2.1 */
			size = n + 1;	/* precisely what is needed */
		else		/* glibc 2.0 */
			size *= 2;	/* twice the old size */
		if ((p = srealloc (p, size)) == NULL)
			return NULL;
	}
}


/* this function check is *url has a sip: prefix. If it has not, then it add ones in a newly allocated string. If no correction is made, it just returns
the initial string */
char *
check_sipurl (char *url)
{
	char *newstr;
	from_t *from;
	int err;

	/* try to parse to see if the address in correct */
	from_init (&from);
	err = from_parse (from, url);
	from_free (from);
	sfree (from);
	if (err == 0)
		return url;	/* OK */


	if (strncmp (url, "sip:", 4) == 0)
		return NULL;
	if (strncmp (url, "<sip:", 5) == 0)
		return NULL;

	newstr = smalloc (strlen (url) + 5);
	strcpy (newstr, url);
	sclrspace (newstr);
	if (strncmp (newstr, "sip:", 4) == 0)
		return newstr;
	if (strncmp (newstr, "<sip:", 5) == 0)
		return newstr;
	/* if we are here, there is no "sip:" at the begginning, so add it */

	if (newstr[0] == '<')
		sprintf (newstr, "<sip:%s", url + 1);	/* do a <sip:machin@truc> */
	else
		sprintf (newstr, "sip:%s", url);	/* do a sip:machin@truc */
	sclrspace (newstr);
	/* try to parse to see if the address in correct */
	from_init (&from);
	err = from_parse (from, newstr);
	from_free (from);
	sfree (from);
	if (err == 0)
		return newstr;	/* OK */
	else
		return (NULL);
}


int
msg_getbody_by_mime (sip_t * sipmsg, char *mime, body_t ** body)
{
	int i = 0;
	body_t *tmpbody;
	while (1)
	{
		msg_getbody (sipmsg, i, &tmpbody);
		if (strcmp (tmpbody->content_type->type, mime) == 0)
		{
			*body = tmpbody;
			return 0;
		}
		i++;
	}
	*body = NULL;
	return -1;
}

char *
content_type_get_type (content_type_t * ct)
{
	char *p;
	int len = 0;
	if (ct->type != NULL)
		len += strlen (ct->type);
	else
	{
		return NULL;
	}
	if (ct->subtype != NULL)
		len += strlen (ct->subtype);
	else
		return NULL;
	p = smalloc (len + 2);
	sprintf (p, "%s/%s", ct->type, ct->subtype);
	return p;
}

char * int_2char(int a)
{
	char *p=smalloc(16);
	snprintf(p,16,"%i",a);
	return p;
}

/* return the value of attr "field" for payload pt at line pos (field=rtpmap,fmtp...)*/
char *sdp_a_attr_value_get_with_pt(sdp_t *sdp,int pos,int pt,char *field)
{
	int i,tmppt=0,scanned=0;
	char *tmp;
	sdp_attribute_t *attr;
	for (i=0;(attr=sdp_attribute_get(sdp,pos,i))!=NULL;i++){
		if (keywordcmp(field,attr->a_att_field)==0){
			sscanf(attr->a_att_value,"%i %n",&tmppt,&scanned);
			if (pt==tmppt){
				tmp=attr->a_att_value+scanned;
				if (strlen(tmp)>0)
					return tmp;
				else {
					osip_trace(OSIP_WARNING,("sdp has a strange a= line."));
					continue;
				}
			}
		}
	}
	return NULL;
}

int sdp_b_bandwidth_get_with_pt(sdp_t *sdp,int pos,int pt)
{
	int i,tmppt=0,bwvalue=0;
	sdp_bandwidth_t *bw;
	
	for (i=0;(bw=sdp_bandwidth_get(sdp,pos,i))!=NULL;i++){
		if (keywordcmp("AS",bw->b_bwtype)==0){
			sscanf(bw->b_bandwidth,"%i %i",&tmppt,&bwvalue);
			if (pt==tmppt){
				return bwvalue;
			}
		}
	}
	return 0;
}

void *scalloc(int n, int size)
{
	return calloc(n,size);
}
//void sfree(void *ptr);
void *srealloc(void *addr, int size)
{
	return realloc(addr,size);
}


int guess_local_address(char *address_to_reach,char **loc){
	int err,tmp;
	socklen_t s;
	struct addrinfo hints;
	struct addrinfo *res=NULL;
	struct sockaddr_storage addr;
	int sock;

	*loc=NULL;

	memset(&hints,0,sizeof(hints));
	hints.ai_family=PF_UNSPEC;
	hints.ai_socktype=SOCK_DGRAM;
	//hints.ai_flags=AI_NUMERICHOST|AI_CANONNAME;
	err=getaddrinfo(address_to_reach,"5060",&hints,&res);
	if (err!=0){
		osip_trace(OSIP_ERROR,("Error in getaddrinfo for %s: %s\n",address_to_reach,gai_strerror(err)));
		return -1;
	}
	if (res==NULL){
		osip_trace(OSIP_ERROR,("getaddrinfo reported nothing !"));
		abort();
		return -1;
	}
	sock=socket(res->ai_family,SOCK_DGRAM,0);
	tmp=1;
	err=setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,&tmp,sizeof(tmp));
	if (err<0){
		osip_trace(OSIP_ERROR,("Error in setsockopt: %s\n",strerror(errno)));
		abort();
		return -1;
	}
	err=connect(sock,res->ai_addr,res->ai_addrlen);
	if (err<0) {
		osip_trace(OSIP_ERROR,("Error in connect: %s\n",strerror(errno)));
		abort();
		return -1;
	}
	freeaddrinfo(res);
	res=NULL;
	s=sizeof(addr);
	err=getsockname(sock,(struct sockaddr*)&addr,&s);
	if (err<0) {
		osip_trace(OSIP_ERROR,("Error in getsockname: %s\n",strerror(errno)));
		close(sock);
		return -1;
	}
	*loc=smalloc(MAXHOSTNAMELEN);
	err=getnameinfo((struct sockaddr*)&addr,s,*loc,MAXHOSTNAMELEN,NULL,0,NI_NUMERICHOST);
	if (err<0){
		osip_trace(OSIP_ERROR,("getnameinfo error:%s",strerror(errno)));
		abort();
		return -1;
	}
	close(sock);
	osip_trace(OSIP_INFO1,("Outgoing interface to reach %s is %s.\n",address_to_reach,*loc));
	return 0;
}
