/*
 *
 * Linup - Uptimes Project Client for Linux
 * Copyright (C) 2002  Joao Sena Ribeiro
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "linup.h"
#include <sys/utsname.h>
#include <sys/sysinfo.h>
#include <limits.h>

/* Number to divide the number we get from sysinfo(), regarding *
 * the load average. In my system, it's 65535 (corresponding to *
 * USHRT_MAX).                                                  */
#define LOADDIV USHRT_MAX

/* get_most_info(char*, long int*, float*, char*, char*, char*) -> (void)
 *
 * Gets (through uname() and sysinfo()) the most information
 * possible (idle time is not available through these system
 * calls), and stores it in the pointers it receives).
 *
 */
void get_most_info(char *progname, long *uptime, float *load, char *os,
    char *oslevel, char *cpu) {

  struct sysinfo data;
  struct utsname name;

  if(sysinfo(&data) == -1 || uname(&name) == -1)
    crash(progname, "sysinfo() or uname()");

  *uptime = data.uptime / 60;
  *load = (float) data.loads[0] / (float) LOADDIV;
  strncpy(os, name.sysname, (MAXINFOLENGTH - 1));
  strncpy(oslevel, name.release, (MAXINFOLENGTH - 1));
  strncpy(cpu, name.machine, (BUFFSIZE - 1));
}

/* get_idle(char*, int*) -> (void)
 *
 * Gets (reading a file in the /proc FS) the idle time
 * information. Stores it in the pointer it receives.
 *
 * Idle time information is not available through uname() or
 * sysinfo(), so we need a separate function for this.
 *
 */
void get_idle(char *progname, int *idle) {
  FILE *fp;
  char c, buff[BUFFSIZE];
  int i;
  long int uptime;
  double cpu_idle;

  if((fp = fopen(PROC_UPTIME, "r")) == NULL)
    crash(progname, PROC_UPTIME);

  for(i = 0; ((c = fgetc(fp)) != EOF) && (i < (sizeof(buff) - 1)); i++)
    buff[i] = c;
  buff[i] = '\0';

  sscanf(buff, "%ld.%*d %lf", &uptime, &cpu_idle);

  /* The idle formula is:   *
   *   (idle/uptime)*100    *
   * everything in seconds. */
  *idle = (int) ((cpu_idle / uptime) * 100.0);

  fclose(fp);
}
