/*
 *
 * Linup - Uptimes Project Client for Linux
 * Copyright (C) 2002  Joao Sena Ribeiro
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "linup.h"
#include <ctype.h>

/* read_auth(char*, char*, int, char*) -> (void)
 *
 * Reads a string from a file (the last argument is a pointer to
 * a string containing the filename) and stores it in the pointer
 * passed to it. This string is the key to use in authenticating
 * with the server. Finally, checks if the key is valid (through
 * check_key()).
 *
 */
void read_auth(char *progname, char *auth, int maximum, char *authfile) {
  FILE *fp;
  char c;
  int i;

  if((fp = fopen(authfile, "r")) == NULL)
    crash(progname, authfile);

  for(i = 0; ((c = fgetc(fp)) != EOF) && (i < (maximum - 1)); i++)
    auth[i] = c;
  auth[i] = '\0';

  check_key(progname, auth);

  fclose(fp);
}

/* set_auth(char*, char*, int, char*) -> (void)
 *
 * Checks if the key is valid (through check_key()), and sets the
 * second char pointer it receives to be the key.
 *
 */
void set_auth(char *progname, char *auth, int maximum, char *authstring) {

  check_key(progname, authstring);

  strncpy(auth, authstring, maximum);
}

/* check_key(char*, char*) -> (void)
 *
 * Checks if a key is valid. If it isn't, exit.
 *
 */
void check_key(char *progname, char *key) {
  int i;

  if(strlen(key) != 32) {
    fprintf(stderr,
        "%s: Authorization key must have a length of 32 characters\n",
        progname);
    exit(EXIT_FAILURE);
  }

  for(i = 0; i < strlen(key); i++)
    if(!isxdigit(key[i])) {
      fprintf(stderr,
          "%s: Character number %d (%c) is not valid in an auth key\n",
          progname, (i + 1),
          ((isgraph(key[i]) || isblank(key[i])) ? key[i] : '?'));
      exit(EXIT_FAILURE);
    }
}
