/*
 * Copyright (c) 2013 Samsung Electronics Co., Ltd.
 *		http://www.samsung.com
 *
 * Jungseok Lee <jays.lee@samsung.com>
 *
 * EXYNOS5440 - SOC graceful poweroff support
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/workqueue.h>

#define XMU_EVTEN			0x4
#define XMU_IRQEN			0x8
#define XMU_IRQ_STATUS			0xC

#define XMU_PWRDN_REQ_EN		0x1
#define XMU_PWRDN_REQ_SHIFT		4

struct exynos_xmu_data {
	struct device *dev;
	void __iomem *base;
	int irq;
	struct work_struct irq_work;
};

static void exynos_xmu_work(struct work_struct *work)
{
	struct exynos_xmu_data *data = container_of(work,
				struct exynos_xmu_data, irq_work);
	unsigned int irq_status;

	irq_status = __raw_readl(data->base + XMU_IRQ_STATUS);

	if ((irq_status >> XMU_PWRDN_REQ_SHIFT) & 0x1) {
		__raw_writel(XMU_PWRDN_REQ_EN << XMU_PWRDN_REQ_SHIFT,
			data->base + XMU_IRQ_STATUS);
		orderly_poweroff(true);
	}

	enable_irq(data->irq);
}

static irqreturn_t exynos_xmu_irq(int irq, void *id)
{
	struct exynos_xmu_data *data = id;

	disable_irq_nosync(irq);
	schedule_work(&data->irq_work);

	return IRQ_HANDLED;
}

static const struct of_device_id exynos_xmu_match[] = {
	{
		.compatible = "samsung,exynos5440-xmu",
	},
	{},
};
MODULE_DEVICE_TABLE(of, exnoys_xmu_match);

static int exynos_xmu_probe(struct platform_device *pdev)
{
	struct device_node *np;
	struct resource res;
	struct exynos_xmu_data *data;
	int ret;

	np = of_find_compatible_node(NULL, NULL, "samsung,exynos5440-xmu");
	if (!np) {
		dev_err(&pdev->dev, "Failed to find device node\n");
		return -ENODEV;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(struct exynos_xmu_data),
					GFP_KERNEL);
	if (!data) {
		dev_err(&pdev->dev, "Failed to allocate driver structure\n");
		return -ENOMEM;
	}

	data->dev = &pdev->dev;

	ret = of_address_to_resource(np, 0, &res);
	if (ret)
		goto err_put_node;

	data->base = devm_ioremap(data->dev, res.start, resource_size(&res));
	if (IS_ERR(data->base)) {
		ret = PTR_ERR(data->base);
		goto err_put_node;
	}

	/*
	 * XMU interrupt number is 0
	 * Thus, it is impossible to add error check code
	 */
	data->irq = irq_of_parse_and_map(np, 0);

	INIT_WORK(&data->irq_work, exynos_xmu_work);
	ret = devm_request_irq(data->dev, data->irq, exynos_xmu_irq,
		IRQF_TRIGGER_RISING|IRQF_SHARED, dev_name(&pdev->dev), data);
	if (ret) {
		dev_err(data->dev, "Failed to register IRQ\n");
		goto err_put_node;
	}

	__raw_writel(XMU_PWRDN_REQ_EN << XMU_PWRDN_REQ_SHIFT,
		data->base + XMU_IRQEN);
	__raw_writel(XMU_PWRDN_REQ_EN << XMU_PWRDN_REQ_SHIFT,
		data->base + XMU_EVTEN);

	pr_info("XMU interrupt handler registered\n");

	return 0;

err_put_node:
	dev_err(data->dev, "Failed to initialize XMU driver\n");
	return ret;
}

static int exynos_xmu_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver exynos_xmu_driver = {
	.driver = {
		.name = "exynos5440-xmu",
		.owner = THIS_MODULE,
		.of_match_table = exynos_xmu_match,
	},
	.probe = exynos_xmu_probe,
	.remove = exynos_xmu_remove,
};

module_platform_driver(exynos_xmu_driver);

MODULE_DESCRIPTION("EXYNOS5440 XMU Driver");
MODULE_AUTHOR("Jungseok Lee<jays.lee@samsung.com>");
MODULE_LICENSE("GPL");
