/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SerialPort.h"
#include "Exception.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include <wx/debug.h>
#include <wx/log.h>

CSerialPort::CSerialPort(const wxString& device) :
CPTTPort(),
m_device(device),
m_fd(-1)
{
}

CSerialPort::CSerialPort() :
CPTTPort(),
m_device(wxEmptyString),
m_fd(-1)
{
}

CSerialPort::~CSerialPort()
{
}

void CSerialPort::open(const wxString& device)
{
	m_device = device;

	open();
}

void CSerialPort::open()
{
	wxASSERT(m_fd == -1);

	m_fd = ::open(m_device.mb_str(), O_RDWR, 0);
	if (m_fd < 0) {
		wxString text;
		text.Printf(wxT("Cannot open the serial port - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	if (::isatty(m_fd) == 0) {
		close();
		throw CException(wxT("Device is not a serial port"));
	}

	unkeyTX();
}

void CSerialPort::keyTX()
{
	wxASSERT(m_fd != -1);

	unsigned int y;
	if (::ioctl(m_fd, TIOCMGET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	y |= TIOCM_RTS;
	y |= TIOCM_DTR;

	if (::ioctl(m_fd, TIOCMSET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}
}

void CSerialPort::unkeyTX()
{
	wxASSERT(m_fd != -1);

	unsigned int y;
	if (::ioctl(m_fd, TIOCMGET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot unkey the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	y &= ~TIOCM_RTS;
	y &= ~TIOCM_DTR;

	if (::ioctl(m_fd, TIOCMSET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot unkey the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}
}

void CSerialPort::close()
{
	wxASSERT(m_fd != -1);

	unkeyTX();

	::close(m_fd);
	m_fd = -1;
}
