package Lire::Aggregator;

use strict;

use base qw/ Lire::ReportOperator /;

use Lire::ReportOperator;

use Carp;

=pod

=head1 NAME

Lire::Aggregator - Base class for all aggregator operators

=head1 SYNOPSIS

    use base qw/ Lire::Aggregator /;

=head1 DESCRIPTION

The Lire::Aggregator is the base class for all the aggregator
operators available in Lire. It implements behavior common to all the
aggregator as well as defining some methods that need to be
implemented by subclasses.

In Lire, an aggregator is an operator which will group DLF records.
Other operators like avg or sum will then compute values in these
groups of DLF records. Aggregators can be nested to compute values on
hierarchical groups.

=head1 METHODS

=head2 ops( [$new_ops] )

Returns the operators contained in this aggregator. The returned value
is a reference to an array of Lire::ReportOperator objects.

If the $new_ops is used, it changes the content of this aggragagor to
these new values. $new_ops should be a reference to an array
containing Lire::ReportOperator objects.

=cut

sub ops {
    my ( $self, $ops ) = @_;

    if ( @_ == 2 ) {
	croak "$ops isn't an array reference"
	  unless UNIVERSAL::isa( $ops, "ARRAY" );

	croak "fields array is empty"
	  if @$ops == 0;

	foreach my $op ( @$ops ) {
	    croak "$op isn't of type Lire::ReportOperator"
	      unless UNIVERSAL::isa( $op, "Lire::ReportOperator" );
	}

	$self->{'ops'} = $ops;
    }

    return $self->{'ops'};
}

=pod

=head2 op_by_name( $name )

Returns the operator named $name in this aggregator. An exception is
thrown if there is no such operator.

=cut

sub op_by_name {
    my ( $self, $name ) = @_;

    # Check in ops
    foreach my $op ( @{$self->{'ops'}} ) {
	return $op if $op->name eq $name;
    }

    croak "No operation named $name\n";
}

=pod

=head2 is_name_defined( $name )

Returns true if this aggregator contains an operator named $name.

=cut

sub is_name_defined {
    my ( $self, $name ) = @_;

    return 1 if $name eq $self->name();

    # Check in ops and aggregator's children
    foreach my $p ( @{$self->{'ops'}} ) {
	return 1 if $p->name() eq $name;
	if ( $p->isa( 'Lire::Aggregator' ) ) {
	    return 1 if $p->is_name_defined( $name );
	}
    }

    return 0;
}

=pod

=head1 METHODS FOR SUBCLASSES

=cut

#------------------------------------------------------------------------
# Method init( %params )
#
# Initialize the ops attribute.
sub init {
    my ($self, %params) = @_;

    $self->{'ops'} = [];

    $self->SUPER::init( %params );

    return;
}


=pod

=head2 print( $fh, $pfx )

This methods implements the print() method required by
Lire::ReportOpetor. It prints the XML element named after op() and
takes care of writing the XML representation of all the children
operation. It also takes care of writing the name and label attribute.
Other attributes can be added to the XML element by overriding the
xml_attrs() method. Other children elements could be added to the
output stream by overriding the print_content() method.

=cut

sub print {
    my ( $self, $fh, $pfx ) = @_;
    $fh  ||= \*STDOUT;
    $pfx ||= 0;

    my $prefix = " " x $pfx;
    my $attrs = $self->xml_attrs;

    print $fh $prefix, '<lire:', $self->op;
    print $fh ' ', $attrs
      if length $attrs;
    print $fh qq{ label="$self->{'label'}"}
      if $self->{'label'};
    print $fh ">\n";

    $self->print_content( $fh, $pfx + 1);

    print $fh $prefix, "</lire:", $self->op, ">\n";
}

=pod

=head2 xml_attrs()

This method can be used to write additional XML attributes. The
returned string will be output in the XML element.

=cut

sub xml_attrs {
    return "";
}

=pod

=head2 print_content( $fh, $pfx )

This method prints the operators contained in this aggregator. It can
be overriden to add some other elements.

=cut

sub print_content {
    my ( $self, $fh, $pfx ) = @_;

    foreach my $o ( @{$self->{'ops'}} ) {
	$o->print( $fh, $pfx );
    }
}

=pod

=head2 create_group_info( $info )

FIXME

Subclasses have to override the create_categorical_info() method for
this implementation.

=cut

sub create_group_info {
    my ( $self, $info ) = @_;

    $info = $self->maybe_nest_group_info( $info );

    $self->create_categorical_info( $info );

    foreach my $op ( @{$self->ops} ) {
        if ( $op->isa( 'Lire::Aggregator' ) ) {
            $op->create_group_info( $info );
        } else {
            $op->create_numerical_info( $info );
        }
    }
}

=pod

=head2 create_categorical_info( $info )

This method is used by the implementation of create_group_info() to add
the categorical ColumnInfo provided by the aggregator.

=cut

sub create_categorical_info {
    croak "unimplemented create_categorical_info() in ", ref $_[0];
}

#------------------------------------------------------------------------
# Method maybe_nest_group_info( $info )
#
# This method should be used by subclasses in their create_group_info() 
# implementation to select the GroupInfo to which they should add their
# ColumnInfo.
#
# This method takes care of creating a GroupInfo if the aggregator has a
# parent. It returns the GroupInfo object to which the aggregator should
# add its ColumnInfo objects.
#
sub maybe_nest_group_info {
    my ( $self, $info ) = @_;

    if ( $self->parent ) {
	return $info->create_group_info( $self->name );
    } else {
	return $info;
    }
}

=pod

=head2 create_entries( $subreport )

This method is used by Lire::ReportSpec to fill the
Lire::Report::Subreport with the entries when creating the subreport.

The $subreport parameter contains the Subreport object to which the
subreport's entries should get added.

This method will only be called on the top-level aggregator in the
report. Default implementation creates an empty report by not adding
anything to the subreport or computes the report on the ReportSpec's
store.

=cut

sub create_entries {
    my ( $self, $subreport ) = @_;

    if ( $self->{'store'} ) {
        my $query = new Lire::DlfQuery( $self->report_spec()->schema()->id() );
        $self->build_query( $query );
        my $expr = $self->report_spec()->filter_spec();
        if ( $expr ) {
            $query->set_filter_clause( $expr->sql_expr(),
                                       @{$expr->sql_params()} );
        }
        $self->build_table( $self->{'store'}, $query, $subreport );
    }

    return;
}


=pod

=head2 build_query( $query )

FIXME

=cut

sub build_query {
    my ($self, $query ) = @_;

    $query->add_aggr_field( '_lr_nrecords', 'count(*)' )
      unless grep { $_->isa( 'Lire::Aggregator' ) } @{$self->ops()};

    foreach my $op ( @{ $self->ops() }) {
        $op->build_query( $op->isa( 'Lire::Aggregator' )
                          ? $query->create_nested_query()
                          : $query );
    }
}

sub build_table_summary {
    my ( $self, $store, $query, $table ) = @_;

    my $result = $query->execute_summary( $store );
    my $summary = $result->next_row();

    $table->nrecords( $summary->{'_lr_nrecords'} );
    $self->set_group_summary( $table, $summary );
}

=pod

=head2 set_group_summary( $group, $row )

FIXME

=cut

sub set_group_summary {
    my ( $self, $group, $row ) = @_;

    $group->nrecords( $row->{'_lr_nrecords'} );

    $self->_set_aggregate_summary_values( $group, $row );
}

sub _set_aggregate_summary_values {
    my ( $self, $group, $row ) = @_;

    foreach my $op ( @{$self->ops()} ) {
        if ( $op->isa( 'Lire::Aggregator' ) ) {
            $op->_set_aggregate_summary_values( $group, $row );
        } else {
            my $parent = $group->parent_entry()
              ? $group->parent_entry()->group() : $group;
            my $value = $op->create_value( $parent, $row );
            $group->set_summary_value( $op->name(), %$value );
        }
    }
}

sub _set_store {
    my ( $self, $store )  = @_;

    $self->{'store'} = $store;

    return;
}

sub build_table {
    my ( $self, $store, $query, $table ) = @_;

    $self->build_table_summary( $store, $query, $table )
      unless $self->parent();

    my $result = $query->execute( $store );
  ROW:
    while (defined (my $row = $result->next_row() ) ) {
        my $group;
        if ( $self->parent() ) {
            my $p_name = defined $self->parent()->parent()
              ? $self->parent()->name() : 'table';
            my $p_entry = $table->find_entry( $p_name, $row );
            next ROW unless defined $p_entry;
            $group = $p_entry->data_by_name( $self->name() );
        } else {
            $group = $table;
        }


        my $entry = $self->create_entry( $group, $row );
        next ROW unless defined $entry;

        foreach my $op ( @{$self->ops()} ) {
            if ( $op->isa( 'Lire::Aggregator' ) ) {
                my $group = $entry->create_group();
                $op->set_group_summary( $group, $row );
            } else {
                my $value = $op->create_value( $entry->group(), $row );
                $entry->add_value( $value );
            }
        }
    }

    my $i=0;
    my $nqueries = $query->nested_queries();
    foreach my $op ( grep { $_->isa( 'Lire::Aggregator' ) } @{$self->ops()}) {
        $op->build_table( $store, $nqueries->[$i++], $table );
    }
}

=pod

=head2 create_entry( $group, $row )

FIXME

=cut
sub create_entry {
    my ( $self, $group, $row ) = @_;

    croak( "Unimplemented create_entry() in ", ref $self );
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::GroupOp(3pm), Lire::ReportOperator(3pm),
Lire::Group(3pm), Lire::Timegroup(3pm), Lire::Timeslot(3pm),
Lire::Rangegroup(3pm)

=head1 VERSION

$Id: Aggregator.pm,v 1.24 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

