package Lire::AsciiDlf::Timeslot;

use strict;

# We need to inherit from AsciiDlf::Aggregator first, otherwise
# perl will look into Lire::Aggregator before than
# Lire::AsciiDlf::Aggregator
use base qw/ Lire::AsciiDlf::Aggregator Lire::Timeslot /;

use Lire::DataTypes qw/ :time /;
use Carp;

use vars qw/ @SLOT_UNIT @MAX_SLOTS @SLOT_INDEX @SLOT_FMT /;

BEGIN {
    # SLOT_INDEX: where in a localtime(3) struct we can find relevant entry;
    #  since there is no weeknumber represented in localtime, we use a `wk'
    #  as placeholder
    #
    #              mday, wk, wday, hour,  min, sec

    @SLOT_UNIT  = qw/1M  1w   1d    1h    1m    1s/;
    @MAX_SLOTS  = qw/12  52    7    24    60    60/;
    @SLOT_INDEX = qw/4   wk    6     2     1     0/;
    @SLOT_FMT   = qw/mo  wk   dw    %H:00 00:%M 00:00:%S/;
}

#------------------------------------------------------------------------
# Method init_merge($period_start, $period_end)
#
# Method required by Lire::AsciiDlf::ReportOperator
sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;

    $self->init_common;

    $self->SUPER::init_merge( $period_start, $period_end );

    return $self;
}

#------------------------------------------------------------------------
# Method init_common()
#
# Method that does initialization common to init_merge() and init_report()
sub init_common {
    my ( $self ) = @_;

    my $unit = $self->unit;
    if ( $unit =~ /^\$/ ) {
	$unit = substr $unit, 1;
	$unit = $self->report_spec->param( $unit )->value;
    }
    my $unit_sec = duration2sec( $unit );

    # Determine if we have a valid duration value for a timeslot
    my $unit_idx = 5;
    if ( monthly_duration( $unit ) ) {
	$unit_idx = 0;
    } elsif ( weekly_duration( $unit ) ) {
	$unit_idx = 1;
    } elsif ( daily_duration( $unit ) ) {
	$unit_idx = 2;
    } elsif ( hourly_duration( $unit ) ) {
	$unit_idx = 3;
    } elsif ( minutely_duration( $unit ) ) {
	$unit_idx = 4
    } else {
	$unit_idx = 5
    }

    my $unit_ok = 0;
    my $sec = duration2sec( $SLOT_UNIT[$unit_idx] );
    if ( $unit_sec >= $sec && !( $unit_sec % $sec ) ) {
	croak "$unit is a too big muliple of $SLOT_UNIT[$unit_idx]. ",
	  "Max is $MAX_SLOTS[$unit_idx]"
	    if $unit_sec > $MAX_SLOTS[$unit_idx] * $sec;

	$self->{'slot_index'} = $SLOT_INDEX[$unit_idx];
	$self->{'time_fmt'}   = $SLOT_FMT[$unit_idx];
	$self->{'multiplier'} = $unit_sec / $sec;

	$unit_ok = 1;
    }
    croak "invalid timeslot value: must be an even multiple of 1M, 1wk, 1d, 1h, 1m or 1s"
      unless $unit_ok;

    return;
}

#------------------------------------------------------------------------
# Method init_aggregator_data()
#
# Method required by Lire::AsciiDlf::Aggregator
sub init_aggregator_data {
    return [];
}

#------------------------------------------------------------------------
# Method merge_aggregator_data( $group, $data )
#
# Method required by Lire::AsciiDlf::Aggregator
sub merge_aggregator_data {
    my ( $self, $group, $timeslots ) = @_;

    foreach my $e ( $group->entries ) {
	my @names = $e->names;
	die "wrong number of names for a timeslot subreport: ",
	  scalar @names, "\n" unless @names == 1;
	my $slot = $names[0]{'value'};
	my $mult = $names[0]{'range'};

	# Are the multipliers compatible?
	# They are only if the target multiplier is a multiple 
	# of the source one.
	# Non-multiple would need interpolation, which would be hard
	# to achieve for other operation than sum and count
	die "incompatible unit multiplier: $self->{'multiplier'} isn't a multiple of $mult\n"
	  unless $self->{'multiplier'} % $mult == 0;

	# Are the used slot compatible?
	my $index;
	if ( $self->{'time_fmt'} eq 'mo' ) {
	    croak "incompatible timeslot unit: $slot\n"
	      unless $slot =~ /^M(\d+)$/;
	    $index = $1;
	} elsif ( $self->{'time_fmt'} eq 'dw' ) {
	    croak "incompatible timeslot unit: $slot\n"
	      unless $slot =~ /^D(\d+)$/;
	    $index = $1;
	} elsif ( $self->{'time_fmt'} eq 'wk' ) {
	    croak "incompatible timeslot unit: $slot\n"
	      unless $slot =~ /^W(\d+)$/;
	    $index = $1;
	} else {
	    my $type;
	    ( $type, $index ) = $slot =~ /^(\d)-(\d+)$/
	      or croak "invalid timeslot's value attribute: $slot\n";

	    croak "incompatible timeslot unit: $slot\n"
	      if $type != $self->{'slot_index'};
	}

	# Map index when target multiplier is greater than the original one.
	# For example: original = 2h, target = 4h
	# Original:  0 2 4 6 8 10 12 14 16 18 20 22 -> 12 indices
	# Target:    0   4   8    12    16    20    -> 6  indices

	my $idx = int( $index * $mult / $self->{'multiplier'} );

	my $data = $timeslots->[$idx];
	unless ( defined $data ) {
	    $data = [];

	    my $i = 0;
	    foreach my $op ( @{$self->ops} ) {
		$data->[$i++] = $op->init_group_data();
	    }

	    $timeslots->[$idx] = $data;
	}

	my $i = 0;
	foreach my $op ( @{$self->ops} ) {
	    my $value = $e->data_by_name( $op->name );
	    my $op_data = $data->[$i++];
	    $op->merge_group_data( $value, $op_data )
	      if ( $value );
	}
    }

    return $self;
}

#------------------------------------------------------------------------
# Method end_aggregator_data($data)
#
# Method required by Lire::AsciiDlf::Aggregator
sub end_aggregator_data {
    my ( $self, $timeslots ) = @_;

    # Finalize each timeslice
    # Either create empty one or call end_group_data on them
    my $last_idx;
    if ( $self->{'slot_index'} eq 'wk' ) {
	$last_idx  = 52;
    } else {
	$last_idx  = ( 59, 59, 23, 0, 11, 0, 6 )[$self->{'slot_index'}];
    }
    $last_idx = int($last_idx / $self->{'multiplier'});

    my $i = 0;
    while ( $i <= $last_idx) {
	if ( $timeslots->[$i]) {
	    my $data = $timeslots->[$i];
	    my $j = 0;
	    foreach my $op ( @{$self->ops} ) {
		$op->end_group_data( $data->[$j++] );
	    }
	} else {
	    # Create empty set
	    my $data =[];
	    my $j = 0;
	    foreach my $op ( @{$self->ops} ) {
		$data->[$j] = $op->init_group_data();
		$op->end_group_data( $data->[$j++] );
	    }

	    $timeslots->[$i] = $data;
	}
	$i++;
    }

    return $self;
}


#------------------------------------------------------------------------
# Method create_group_entries( $group, $data)
#
# Method required by Lire::AsciiDlf::Aggregator
#
sub create_group_entries {
    my ( $self, $group, $timeslots ) = @_;

    for ( my $i=0; $i < @$timeslots; $i++ ) {
        my $row = { $self->name() => $i * $self->{'multiplier'} };
	my $entry = $self->create_entry( $group, $row );

	my $j = 0;
	foreach my $op ( @{$self->ops} ) {
	    $op->add_entry_value( $entry, $timeslots->[$i][$j++] );
	}

    }

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Timeslot - handle lire:timeslot stuff

=head1 SYNOPSIS

use Lire::AsciiDlf::Timeslot;

=head1 DESCRIPTION

This module offers support for lire:timeslot elements, as defined in lrsml.xml,
and used in Lire report definition files.

=head1 VERSION

$Id: Timeslot.pm,v 1.33 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
