package Lire::Config::List;

use base qw/ Lire::Config::Value /;

use Carp;

use Lire::Utils qw/ check_param check_object_param /;

=pod

=head1 NAME

Lire::Config::List - List Value Object.

=head1 DESCRIPTION

Used for configuration parameters which contains a list of values.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my $self = $class->SUPER::new(@_);

    check_object_param( $self->{'spec'}, 'spec', 'Lire::Config::ListSpec' );

    $self->{'elements'} = [];

    return $self;
}

sub as_value {
    return [ map { $_->as_value() } @{$_[0]->{'elements'}} ];
}

sub has_value {
    return defined $_[0]->{'elements'} && @{$_[0]->{'elements'}};
}

=pod

=head2 elements()

Returns this list's values as an array of Lire::Config::Value object.

=cut

sub elements {
    return @{$_[0]->{'elements'}};
}

=pod

=head2 get( $idx )

Returns the value stored at index $idx of this list. An exception is
thrown if the index is out-of-bounds.

=cut

sub get {
    my ( $self, $idx ) = @_;

    check_param( $idx, 'idx', qr/^-?\d+$/,
                 "'idx' parameter should be an integer" );

    croak "index out of bounds: $idx"
      unless $idx >= 0 ? ( $idx < @{$self->{'elements'}} ):
                         ( abs($idx)-1 < @{$self->{'elements'}} );

    return $self->{'elements'}[$idx];
}

=head2 remove( $idx )

Removes the value stored at index $idx of this list. An exception is
thrown if the index is out-of-bounds. It returns the deleted element.

=cut

sub remove {
    my ( $self, $idx ) = @_;

    check_param( $idx, 'idx', qr/^-?\d+$/,
                 "'idx' parameter should be an integer" );

    croak "index out of bounds: $idx"
      unless ( $idx >= 0
               ? ( $idx < @{$self->{'elements'}} )
               : ( abs($idx)-1 < @{$self->{'elements'}} ) );

    return splice(@{$self->{'elements'}}, $idx, 1);
}

=pod

=head2 append( $val )

Adds $val to the end of this list.

=cut

sub append {
    my ( $self, $value ) = @_;

    check_object_param( $value, 'value', 'Lire::Config::Value' );

    my $value_name = $value->name();
    croak $self->name(), " cannot contains ", $value_name, " parameters"
      unless $self->spec()->has_component( $value_name );

    croak "$value is not a valid instance for component ", $value_name
      unless $self->spec()->get( $value_name )->is_instance( $value );

    push @{$self->{'elements'}}, $value;

    return;
}

=pod

=head2 set( $idx, $value )

Replaces value at index $idx with $value.

=cut

sub set {
    my ( $self, $idx, $value ) = @_;

    check_param( $idx, 'idx' );
    check_param( $value, 'value' );

    croak "index out of bounds: $idx"
      unless $idx >= 0 ? ( $idx < @{$self->{'elements'}} ):
                         ( abs($idx)-1 < @{$self->{'elements'}} );

    $self->{'elements'}[$idx] = $value;

    return;
}


sub save_value {
    my $self = shift;
    my ($fh, $indent) = @_;

    foreach my $val (@{$self->{'elements'}}) {
        $val->save_xml($fh, $indent+1);
    }

    return;
}

1; # whine, whine

__END__

=pod

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

=head1 VERSION

$Id: List.pm,v 1.5 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
