package Lire::Config::Plugin;

use strict;

use base qw/Lire::Config::Dictionary/;

use Lire::Config;
use Lire::Config::RecordSpec;
use Lire::Config::Dictionary;

use Carp;

=pod

=head1 NAME

Lire::Config::Plugin - Value object for plugin configuration.

=head1 SYNOPSIS

  use Lire::Config::Plugin;

=head1 DESCRIPTION

This configuration object hold the name of a selected plugin as
well as its options.

=cut

sub new {
    my $self = shift->SUPER::new( @_ );

    my %args = @_;

    $self->set_plugin( $args{'value'} );

    return $self;
}

=pod

=head2 get_plugin()

Returns the currently selected plugin.

=cut

sub get_plugin {
    return $_[0]{'_plugin'};
}

=pod

=head2 set_plugin( $plugin )

Changes the selected plugin. If the selected plugin is changed, a
new property set will be created from the plugin's defined properties.

=cut
sub set_plugin {
    my ( $self, $plugin ) = @_;

    my $old_plugin = $self->{'_plugin'};
    $self->{'_plugin'} = $plugin;
    my $spec = $self->get_properties_spec();
    $self->{'_properties'} = $spec->instance()
      unless ( defined $plugin && defined $old_plugin
               && $plugin eq $old_plugin );

    return;
}

=pod

=head2 get_properties_spec()

Returns the TypeSpec that is used to specify the plugin's properties.
This method will return an empty RecordSpec when the plugin didn't
define any properties. One can also use the has_properties() method to
check if the Plugin defined configuration properties.

=cut

sub get_properties_spec {
    my $self = $_[0];

    return new Lire::Config::RecordSpec( 'name' => $self->name() )
      unless ( defined $self->{'_plugin'} );

    my $name = $self->{'_plugin'} . '_properties';
    my $spec = Lire::Config->config_spec();
    return $spec->get( $name ) if $spec->has_component( $name );

    return new Lire::Config::RecordSpec( 'name' => $name );
}

=pod

=head2 has_properties()

Returns a boolean value indicating whether the current plugin is configurable.

=cut
sub has_properties {
    return  $_[0]->get_properties_spec()->components() > 0;
}

=pod

=head2 get_properties()

Returns the Lire::Config::Dictionary object which hold the
plugin configuration.

=cut
sub get_properties {
    return $_[0]{'_properties'};
}

=pod

=head2 as_value()

Returns an hash reference with two keys : 'plugin' which contains
the selected plugin and 'properties' which is an hash reference containing
the plugin's properties.

=cut
sub as_value {
    return { 'plugin' => $_[0]->{'_plugin'},
             'properties' => $_[0]->{'_properties'}->as_value() };
}

=pod

=head2 get( $name )

Delegates to the Dictionary which contains the plugin's properties.

=cut
sub get {
    return shift->{'_properties'}->get( @_ );
}

=pod

=head2 set( $name, $value )

Delegates to the Dictionary which contains the plugin's properties.

=cut
sub set {
    return shift->{'_properties'}->set( @_ );
}

=pod

=head2 is_set( $name )

Delegates to the Dictionary which contains the plugin's properties.

=cut
sub is_set {
    return shift->{'_properties'}->is_set( @_ );
}

sub save_xml {
    my ( $self, $fh, $indent ) = @_;

    return unless ( defined $self->{'_plugin'} );

    print $fh '  ' x $indent, '<param name="', $self->name(), '" value="',
      $self->{'_plugin'}, '">', "\n";
    my $spec = $self->get_properties_spec();
    foreach my $comp ( $spec->components() ) {
        $self->get( $comp->name() )->save_xml( $fh, $indent + 1 );
    }
    print $fh '  ' x $indent, "</param>\n";
}

1;
