package Lire::Config::Scalar;

use strict;

use base qw/Lire::Config::Value/;

use Lire::Utils qw/ xml_encode shell_quote/;

use Carp;

=pod

=head1 NAME

Lire::Config::Scalar

=head1 SYNOPSIS

  use Lire::Config::Scalar

=head1 DESCRIPTION

This package contains the classes use to represent an actual
configuration.

=head1 Lire::Config::Scalar

Value container for scalar values. Implements simple methods to get()
and set() string data.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my $self = $class->SUPER::new(@_);
    my %args = @_;

    $self->{'value'} = '';
    $self->{'warned'} = 0;

    $self->set( $args{'value'} )
        if exists $args{'value'};

    return $self;
}

sub as_value {
    my $self = $_[0];
    my $val = $self->spec->normalize($self->{'value'});

    return $val if $self->spec->is_valid($val);

    unless($self->{'warned'}) {
        carp "invalid value for parameter '", $self->name, "': ",
          ( defined $val ? $val : '(undef)' );
        $self->{'warned'} = 1;
    }

    return wantarray ? () : undef;
}

sub as_label {
    my $label = $_[0]->as_value();
    $label =~ s/(^\s*|\s*$)//g;

    return $label;
}

sub as_shell_var {
    my $self = $_[0];

    return ( $self->spec->is_valid( $self->{'value'} )
             ? $self->name . "=" . shell_quote( $self->as_value )
             : '' );
}

=pod

=head2 has_value()

Returns true if this variable's value was set.

=cut

sub has_value {
    return defined $_[0]->{'value'};
}

=pod

=head2 get()

Returns this scalar's value.

=cut

sub get {
    return $_[0]{'value'};
}

=pod

=head2 set( $val )

This method can be used to change this scalar's value.

=cut

sub set {
    my ($self, $val) = @_;

    $self->{'warned'} = 0;

    $self->{'value'} = $val;
}

=pod

=head2 is_valid()

Checks with the parameter's spec that the current value is valid.

=cut

sub is_valid {
    return $_[0]->{'spec'}->is_valid( $_[0]->{'value'} );
}

sub save_xml {
    my $self = shift;
    my ($fh, $indent) = @_;
    if(defined $self->{'value'}) {
        print $fh '  'x$indent, '<param name="', $self->name, "\">";
        print $fh xml_encode($self->{'value'}), "</param>\n";
    }
}

1; # whine, whine

__END__

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

=head1 VERSION

$Id: Scalar.pm,v 1.4 2004/03/28 23:08:25 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
