package Lire::Records;

use strict;

use base qw/ Lire::Aggregator /;

use Carp;
use POSIX qw/ strftime /;

use Lire::DataTypes qw/ is_time_type is_quantity_type format_numeric_type /;
use Lire::Utils qw/ check_param /;

=pod

=head1 NAME

Lire::Records - Base class for implementation of the records operator.

=head1 SYNOPSIS

    use Lire::Records;

=head1 DESCRIPTION

This module is the base class for implementation of the records
operator. The records operator isn't an aggregator nor a group
operation. It only collects information on each DLF records which will
be included in the report.

=head1 CONSTRUCTOR

=head2 new( %params )

Creates a new instance of a records operator. In addition to the
normal report operator parameters, the records operator takes one
additional parameter:

=over

=item fields

A reference to an array containing the names of the DLF field that should be included in the report.

=back

=cut

sub new {
    my ( $class, %args ) = @_;

    check_param( $args{'fields'}, 'fields' );

    my $self = bless {}, $class;
    $self->SUPER::init( %args,
			'op' => 'records' );

    $self->fields( $args{'fields'} );

    return $self;
}

#------------------------------------------------------------------------
# Method print( $fh, $prefix )
#
# Implementaion of the method required by Lire::ReportOperator.
sub print {
    my ($self,$fh, $prefix) = @_;
    $fh	    ||= \*STDOUT;
    $prefix ||= 0;

    my $pfx = " " x $prefix;

    print $fh $pfx, '<lire:records fields="', join( " ", @{$self->{'fields'}}),
      qq{"/>\n};
}

=pod

=head1 METHOD

=head2 fields( [$new_fields] )

Returns a reference to an array containing the DLF field names that
will be included in the report.

You can use the $new_fields parameter to change that value.

=cut

sub fields {
    my ($self, $fields) = @_;

    if ( @_ == 2 ) {
	if ( defined $fields ) {
	    croak "$fields isn't an array reference"
	      unless UNIVERSAL::isa( $fields, 'ARRAY' );
	    croak "fields cannot be empty\n"
	      unless @$fields;

	    foreach my $f ( @$fields ) {
		croak "$f isn't a defined field in the ",
		  $self->report_spec->schema->id, " schema"
		    unless $self->report_spec->schema->has_field( $f );
	    }
	} else {
	    croak "undefined fields\n";
	}
	$self->{'fields'} = $fields;
    }
    $self->{'fields'};
}

=pod

=head2 ops()

FIXME

=cut

sub ops {
    return [] if (@_ == 1);
    croak "records cannot contain any children";
}

# Implements Lire::ReportOperator::name()
sub name {
    return 'records:' . @{$_[0]->fields()};
}

# Implements Lire::Aggregator::create_categorical_info
sub create_categorical_info {
    my ( $self, $group_info ) = @_;

    foreach my $field ( @{$self->fields} ) {
	my $dlf_field = $self->report_spec->schema->field( $field );
	$group_info->create_column_info( $field, 'categorical',
					 $dlf_field->type,
					 $dlf_field->label, );
    }
}

# Overrides Lire::Aggregator::build_query
sub build_query {
    my ($self, $query ) = @_;

    $self->SUPER::build_query( $query );

    foreach my $field ( @{$self->{'fields'}} ) {
        $query->add_field( $field );
    }
    $query->set_sort_spec( $self->report_spec()->schema()->timestamp_field()->name()) ;
    return;
}

sub create_entry {
    my ( $self, $group, $row ) = @_;

    my $entry = $group->create_entry();

    for ( my $i=0; $i < @{$self->{'fields'}}; $i++ ) {
        my $value = $row->{$self->{'fields'}[$i]};
        my $type = $group->group_info()->info_by_index( $i )->type();
	if ( is_time_type( $type ) ) {
            $entry->add_name( strftime( '%Y-%m-%d %H:%M:%S',
                                        localtime $value ),
                              $value );
	} elsif ( is_quantity_type( $type ) ) {
            $entry->add_name( format_numeric_type( $value, $type ), $value );
	} else {
            $entry->add_name( $value );
	}
    }

    return $entry;
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::Aggregate(3pm), Lire::ReportOperator(3pm),
Lire::Group(3pm), Lire::Timegroup(3pm), Lire::Timeslot(3pm),
Lire::Rangegroup(3pm)

=head1 VERSION

$Id: Records.pm,v 1.19 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

