package Lire::Report::TableInfo;

use strict;

use base qw/Lire::Report::GroupInfo/;

use Carp;

use Lire::Report::GroupInfo;

=pod

=head1 NAME

Lire::Report::TableInfo - Object that holds all the GroupInfo and ColumnInfo for one subreport

=head1 SYNOPSIS

    my $info = $subreport->table_info;
    my $col_info = $info->colum_info( "request_total" );

=head1 DESCRIPTION

The Lire::Report::TableInfo object holds the ColumnInfo and GroupInfo
objects for one subreport..

=head1 CONSTRUCTOR

=head2 new()

Creates a new Lire::Report::TableInfo object.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my $self = bless {
                      'name'       => "table",
                      'children'   => [],
                      'row_idx'    => 0,
                      'parent'     => undef,
                     }, $class;

    $self->index( 0 );

    return $self;
}

=pod

=head2 column_info_by_name( $name )

Returns the Lire::Report::ColumnInfo object that has the name $name.

Contrary to the method in Lire::Report::GroupInfo, this one will
throw an exception if there is no column named $name.

=cut

sub column_info_by_name {
    my ( $self, $name ) = @_;

    my $col = $self->SUPER::column_info_by_name( $name );
    croak "There is no column named $name"
      unless $col;
    return $col;
}

=pod

=head2 column_info_by_col_start( $idx )

Returns the Lire::Report::ColumnInfo object that is starts at column
index $idx.

Contrary to the method in Lire::Report::GroupInfo, this one will throw
an exception if this column is out of bounds.

=cut

sub column_info_by_col_start {
    my ( $self, $idx ) = @_;

    my $col = $self->SUPER::column_info_by_col_start( $idx );
    croak "There is no column index $idx"
      unless $col;

    return $col;
}

=pod

=head2 group_info( $name )

Returns the Lire::Report::GroupInfo object that has the name $name.

Contrary to the method in Lire::Report::GroupInfo, this one will
throw an exception if there is no column named $name.

=cut

sub group_info {
    my ( $self, $name ) = @_;

    # Cache the result of the base lookup
    unless ( exists $self->{'group_info'}{$name} ) {
        $self->{'group_info'}{$name} = $self->SUPER::group_info( $name );
    }

    croak "There is no group named $name"
      unless $self->{'group_info'}{$name};

    $self->{'group_info'}{$name};
}

=pod

=head2 ncols()

Returns the number of columns there is in this table.

=cut

sub ncols {
    my ( $self ) = @_;

    return scalar $self->column_infos;
}

=pod

=head2 column_infos()

Returns an array containing all the columns of the table. Each element
is a Lire::Report::ColumnInfo object.

=cut

sub column_infos {
    my ( $self ) = @_;

    my @cols = $self->column_children;
    my @groups = $self->group_children;
    while ( @groups ) {
        my $g = shift @groups;
        push @cols, $g->column_children;
        push @groups, $g->group_children;
    }
    sort { $a->col_start <=> $b->col_start } @cols;
}

=pod

=head2 groups()

Returns an array containing all the groups of the table. Each element
is a Lire::Report::GroupInfo object.

=cut

sub groups {
    my ( $self ) = @_;

    my @groups = ( $self );
    my @children = ( $self );
    while ( @children ) {
        my $g = shift @children;
        push @groups, $g->group_children;
        push @children, $g->group_children;
    }
    @groups;
}

=pod

=head2 columns_width()

Returns an array where each element represents the suggested columns'
width in characters.

=cut

sub columns_width {
    my ( $self ) = @_;

    map { $_->col_width } $self->columns;
}

# ------------------------------------------------------------------------
# Method compute_columns_width()
#
# Try to determine the width of all the columns
sub compute_columns_width {
    my ( $self ) = @_;

    $self->assign_columns_width;
    $self->correct_spanning_column_width( $self );
}

#-----------------------------------------------------------------------
# Methods write_report( $fh, $index )
sub write_report {
    my ( $self, $fh, $indent ) = @_;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    print $fh $pfx, qq{<lire:table-info>\n};
    foreach my $i ( @{$self->{'children'}} ) {
        $i->write_report( $fh, $indent + 1 );
    }
    print $fh $pfx, "</lire:table-info>\n";
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::Report::Subreport(3pm) Lire::Report::ColumnInfo(3pm)

=head1 VERSION

$Id: TableInfo.pm,v 1.19 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
