package Lire::ReportConfig;

use strict;

use Carp;
use Text::ParseWords qw/ shellwords /;

use Lire::DataTypes qw/ check_superservice /;
use Lire::ReportSpec;
use Lire::FilterSpec;
use Lire::I18N qw/ set_fh_encoding /;
use Lire::Report;
use Lire::ReportSection;
use Lire::ReportSpecFactory;

=pod

=head1 NAME

Lire::ReportConfig - API to report configuration file

=head1 SYNOPSIS

    use Lire::ReportConfig;

    my $cfg = eval { new_from_file Lire::ReportConfig };
    $cfg->print();

=head1 DESCRIPTION

This class parses and writes report configuration file. All methods
will die on error.

=head1 CONFIGURATION FILE SYNTAX

Report configuration files are made of a series of section. Sections
are introduced by the C<=section> marker:

    =section Section's Title

Each section is made up of a series of filter specifications that
should be shared by all the report specifications that follow.

Filter specifications are introduced by the C<|> character followed by
the filter specification's id. The rest of the line is interpreted as
parameter assigment. Here is an example:

    |select-url url=http://www.logreport.org/

This will make all report specifications of this section use the
C<select-url> filter specification with the C<url> parameter set to the
C<http://www.logreport.org/> value.

Report specification lines start with the specification's id; the
rest of the line is interpreted as parameters' assigments like in the
filter specification's case. Here is an example:

    top-urls-by-client_host     url_to_show=5  client_to_show=10

=head1 CONSTRUCTORS

=head2 new($superservice, [$factory] )

This creates a new Lire::ReportConfig object for the superservice
$superservice. The $factory parameter should be an instance of a
Lire::ReportSpecFactory which would be created to instantiate the
filter and report specification objects related to this configuration
file. If this parameter is omitted, a new object of the class
Lire::ReportSpecFactory will be created.

The created report configuration object doesn't contain any
section, report or filter specifications.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $super, $factory ) = @_;

    croak "invalid superservice: $super"
      unless check_superservice( $super );

    $factory ||= new Lire::ReportSpecFactory;
    croak "factory parameter isn't a Lire::ReportSpecFactory: $factory"
      unless UNIVERSAL::isa( $factory, "Lire::ReportSpecFactory" );

    bless { 'superservice' => $super,
            'factory'      => $factory,
            'sections'     => [],
            '_encoding'    => undef,
            '_filename'    => undef,
           }, $class;
}

=pod

=head2 new_from_file( $superservice, $report_cfg, [$factory] )

This will create a new report configuration object for the
$superservice superservice based on the report configuration file
$report_cfg. The $factory parameter gives the Lire::ReportSpecFactory
object which should be used to create the report and filter
specifications contained in this report configuration. If omitted, a
default one will be used.

=cut

sub new_from_file {
    my $proto   = shift;
    my $class   = ref $proto || $proto;
    my ($superservice, $report_cfg, $factory) = @_;

    my $self = $class->new( $superservice, $factory );

    $self->load_from_file( $report_cfg );

    return $self;
}

=pod

=head1 OBJECT METHODS

=head2 superservice()

Returns the report configuration's superservice.

=cut

sub superservice {
    my ( $self ) = @_;

    $self->{'superservice'};
}

=pod

=head2 factory()

Returns the factory object used by this report configuration.

=cut

sub factory {
    return $_[0]{'factory'};
}

=pod

=head2 filename()

Returns the filename from which this ReportConfig was loaded. It will
return undef if the ReportConfig wasn't loaded from a file.

=cut

sub filename {
    return $_[0]{'_filename'};
}

sub load_from_file {
    my ( $self, $report_cfg, $factory ) = @_;

    # Reset the sections
    $self->{'sections'} = [];

    # Format of the configuration file is
    # ((=section <title>)
    #  (|filter_id <param>)*
    #  (report_id  <param>))*)+

    $self->{'_curr_section'}  = undef;

    # Load the report configuration file
    open my $fh, $report_cfg
      or croak "can't open report configuration file $report_cfg: $!\n";
    $self->{'_fh'} = $fh;

    my $line;
    while ( defined( $line = <$fh> ) ) {
        next if $line =~ /^\s*#/; # Skip comments
        next if $line =~ /^\s*$/; # Skip blank lines

        chomp $line;

        my $first_char = substr( $line, 0, 1 );
        if ( $first_char eq '=' ) {
            if ( $line =~ /^=encoding/ ) {
                $self->_parse_encoding_line( $line );
            } elsif ( $line =~ /^=section/ ) {
                $self->_parse_section_line( $line );
            } else {
                warn "Unknown directive at line $.: $line\n";
            }
        } elsif ( $first_char eq '|' ) {
            $self->_parse_filter_line( $line );
        } else {
            $self->_parse_report_line( $line );
        }
    }
    close $fh;

    delete $self->{'_fh'};
    delete $self->{'_curr_section'};

    $self->{'_filename'} = $report_cfg;
    return;
}

sub _parse_encoding_line {
    my ( $self, $line ) = @_;

    croak "'encoding' directive requires perl >= 5.8.0, at line $."
      unless $Lire::I18N::USE_ENCODING;

    croak "'encoding' directive must be the first directive, at line $."
      if ( defined $self->{'_curr_section'} );
    croak "only one 'encoding' directive allowed, at line $."
      if ( defined $self->{'_encoding'} );

    $line =~ /^=encoding\s+([-\w.]+)$/;
    croak "invalid 'encoding' directive, at line $."
      unless ( defined $1 );
    $self->{'_encoding'} = $1;

    set_fh_encoding( $self->{'_fh'}, $1 );

    return;
}

sub _parse_section_line {
    my ( $self, $line ) = @_;

    unless ( $line =~ /^=section (.*)$/ ) {
        warn "invalid section directive at line $.: $line";
        return;
    }

    $self->{'_curr_section'} = new Lire::ReportSection( $self->superservice, $1 );
    $self->add_section( $self->{'_curr_section'} );

    return;
}

sub _parse_filter_line {
    my ( $self, $line ) = @_;

    my ( $id, $params ) = parse_param_line( substr( $line, 1 ) );
    eval {
        die "filter specification before any =section directive\n"
          unless $self->{'_curr_section'};
        die"filter specification should come before report specifications\n"
          if $self->{'_curr_section'}->reports();

        my $spec = Lire::FilterSpec->load( $self->superservice, $id,
                                           $self->{'factory'} );

        while ( my ($name, $value) = each %$params ) {
            $spec->param( $name )->value( $value );
        }
        $self->{'_curr_section'}->add_filter( $spec );
    };
    if ( $@ ) {
        warn( "error at line $.: $@\n" );
        warn( "Omitting filter $id defined at line $.\n" );
    }

    return;
}

sub _parse_report_line {
    my ( $self, $line ) = @_;

    my ( $id, $params ) = parse_param_line( $line );

    eval {
        die "report specification before any =section directive\n"
          unless $self->{'_curr_section'};

        my $report_spec =
          Lire::ReportSpec->load( $self->superservice, $id,
                                  $self->{'factory'} );

        while ( my ($name, $value) = each %$params ) {
            $report_spec->param( $name )->value( $value );
        }
        $self->{'_curr_section'}->add_report( $report_spec );
    };
    if ( $@ ) {
        warn( "error at line $.: $@\n" );
        warn( "Omitting report $id defined at line $.\n" );
    }

    return;
}

sub parse_param_line {
    my ( $id, @p ) = shellwords( $_[0] );
    my %params = ();
    foreach my $param_str ( @p ) {
        my ( $param, $value ) = $param_str =~ /^([-.\w]+)=(.*)$/;
        unless ( defined $param ) {
            warn( "error parsing parameter $param_str at line $.. Ignoring parameter.\n" );
            next;
        }

        $value = "" unless defined $value;
        $params{$param} = $value;
    }

    return ( $id, \%params );
}

sub create_param_line {
    my ( $id, $spec ) = @_;

    my @line = ( $id );
    foreach my $name ( $spec->param_names ) {
        my $value = $spec->param( $name )->value;
        $value =~ s/\\/\\\\/g;  # Escape backslashes
        $value =~ s/"/\\"/g;    # and double quotes

        push @line, $name . '="' . $value . '"';
    }
    return join( " ", @line );
}

=pod

=head2 sections()

Return's this report configuration's sections as an array of
Lire::Section objects

=cut

sub sections {
    my ( $self ) = @_;

    return @{$self->{'sections'}};
}

=pod

=head2 add_section( $section )

Adds a section to this report configuration. The $section parameter
should be a Lire::ReportSection object.

=cut

sub add_section {
    my ( $self, $section ) = @_;

    croak ( "section should be of type Lire::ReportSection (not $section)" )
      unless UNIVERSAL::isa( $section, "Lire::ReportSection" );

    croak( "section's superservice ", $section->superservice,
           " is different than this report configuration's one:",
           $self->superservice )
      if $self->superservice ne $section->superservice;

    push @{$self->{'sections'}}, $section;
}

=pod

=head2 merge_filters()

Calling this method will make sure that all report specifications
take into account their section's filter specification.

This method will modify all report specifications. After this their
object representation won't be identical to the one in the XML report
specification.

=cut

sub merge_filters {
    my ( $self ) = @_;

    my $factory = $self->factory;
    foreach my $section ( $self->sections ) {
        my @filters = map { $_->filter_spec } $section->filters;
        my @reports = $section->reports;

        foreach my $r ( @reports ) {
            if ( @filters ) {
                my $expr;
                if ( $r->filter_spec || @filters > 1) {
                    $expr = $factory->create_and_expr( 'container' => $r );
                    if ( $r->filter_spec ) {
                        $expr->expr( [ @filters, $r->filter_spec ] );
                    } else {
                        $expr->expr( [@filters] );
                    }
                } else {
                    $expr = $filters[0];
                }
                $r->filter_spec( $expr );
            }
        }
    }
}

=pod

=head2 print( [$fh] )

Prints the report configuration on the $fh filehandle. If the $fh
parameter is omitted, the report configuration will be printed on
STDOUT.

=cut

sub print {
    my ( $self, $fh ) = @_;

    $fh ||= \*STDOUT;

    if ( defined $self->{'_encoding'} ) {
        set_fh_encoding( $fh, $self->{'_encoding'} );
        print $fh "=encoding ", $self->{'_encoding'}, "\n\n";
    }
    foreach my $section ( $self->sections() ) {
        print $fh "=section ", $section->title(), "\n";

        foreach my $filter ( $section->filters() ) {
            print $fh create_param_line( "|" . $filter->id(), $filter ), "\n";
        }

        foreach my $report ( $section->reports() ) {
            print $fh create_param_line( $report->id(), $report ), "\n";
        }

        # Empty line
        print $fh "\n"
    }
}

=pod

=head2 create_report( $timespan_start, $timespan_end )

Returns a Lire::Report object based on this report's configuration file.
The $timespan_start and $timespan_end attribute will be used to initiate
the Report object.

Called in lr_dlf2xml(1) and lr_xml_merge(1).

=cut

sub create_report {
    my ( $self, $timespan_start, $timespan_end) = @_;

    my $report = new Lire::Report( $self->superservice, $timespan_start,
                                   $timespan_end,
                                 );

    foreach my $s ( $self->sections ) {
        $s->create_report_section( $report );
    }

    return $report;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportSection(3pm) Lire::ReportSpec(3pm) Lire::FilterSpec(3pm)
Lire::Report(3pm) Lire::ReportSpecFactory(3pm)

=head1 VERSION

$Id: ReportConfig.pm,v 1.20 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

