package Lire::ReportSchedule;

use strict;

use Carp;

use Lire::DlfSchema;
use Lire::ReportConfig;
use Lire::AsciiDlf::AsciiDlfFactory;
use Lire::FilterSpec;
use Lire::FilterExpr;
use Lire::ReportGenerator;
use Lire::ReportMerger;
use Lire::ReportSchedule;
use Lire::Utils qw/ check_param check_object_param /;
use Time::Local;
use Lire::WeekCalculator;

use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::ReportSchedule - Object which reprents one periodical report generation.

=head1 SYNOPSIS

  use Lire::ReportJob;
  use Lire::ReportSchedule;

  my $store = Lire::DlfStore->open( 'aStore' );
  my $job = new Lire::ReportJob( "webServer", "www" );
  $job->add_schedule( new Lire::ReportSchedule( 'daily', 'www.cfg' );
  $job->run( 'daily', $store );

=head1 DESCRIPTION

Lire::ReportSchedule objects together with Lire::OutputJob objects
make it possible to configure all kind of periodical report
generation. Whereas the Lire::OutputJob is used to represent the
formatting of one generated report, the Lire::ReportSchedule
represents such a generation. It will be use to generate periodically
one XML report in a DlfStore using one report configuration file.

ReportSchedules grouped together in one ReportJob are assumed to
generate compatible reports as far as merging is concerned. The idea
is that previous reports generated in one ReportJob (according to
different schedules) could be used as a data source when the DlfStore
doesn't contain all the DLF records for the whole period. For example,
if you don't keep one year of www DLF in your store, you could still
generate an yearly report using the daily reports generated during the
year.

=head2 new( $period, $cfg_file );

Creates a new Lire::ReportSchedule that will generates an XML report
using the $cfg_file report configuration file on a $period period.
$period should be one of 'hourly', 'daily', 'weekly', 'monthly' or
'yearly'.

=cut

sub new {
    my ( $class, $period, $cfg_file ) = @_;

    check_param( $period, 'period', qr/^(hourly|daily|weekly|monthly|yearly)$/,
                 "'period' parameter should be one of 'hourly', 'daily', 'weekly', 'monthly' or 'yearly'" );
    check_param( $cfg_file, 'report_cfg' );

    my $self = bless { '_period' => $period,
                       '_cfg_file' => $cfg_file,
                       '_output_jobs' => [],
                     }, $class;

    return $self;
}

=pod

=head2 period()

Returns the period of this ReportSchedule.

=cut

sub period { $_[0]{'_period'} };


=pod

=head2 period_range( [ $time ] )

Returns the starting and ending boundaries of the schedule period
which includes $time. (Defaults to now). The ending boundary is
excluded from the period and the starting boundary is included.

=cut

sub period_range {
    my ($self, $time) = @_;

    my ( $hour, $day, $month, $year ) = (localtime( $time || time() ))[2..5];
    if ( $self->{'_period'} eq 'hourly' ) {
        my $start = timelocal( 0, 0, $hour, $day, $month, $year );
        return [ $start, $start + 3600 ];
    } elsif ( $self->{'_period'} eq 'daily' ) {
        my $start = timelocal( 0, 0, 0, $day, $month, $year );
        return [ $start, $start + 3600*24 ];
    } elsif ( $self->{'_period'} eq 'weekly' ) {
        my $calc = new Lire::WeekCalculator();
        my $week_no = $calc->week_number( $time );
        my $start = $calc->week_start( $year, $week_no );
        return [ $start, $start + 86400*7 ];
    } elsif ( $self->{'_period'} eq 'monthly' ) {
        my $start = timelocal( 0, 0, 0, 1, $month, $year );
        if ( $month == 11 ) {
            return [ $start, timelocal( 0, 0, 0, 1, 0, $year+1 ) ]
        } else {
            return [ $start, timelocal( 0, 0, 0, 1, $month + 1, $year ) ]
        }
    } elsif ( $self->{'_period'} eq 'yearly' ) {
        return [ timelocal( 0, 0, 0, 1, 0, $year ),
                 timelocal( 0, 0, 0, 1, 0, $year+1 ) ]
    }
}

=pod

=head2 config_file()

Returns the report configuration file that should be used to generate
the report.

=cut

sub config_file {
    return $_[0]{'_cfg_file'};
}

=pod

=head2 init_report_config( $superservice, $merging, [ $time ] )

Returns a Lire::ReportConfig object loaded from the report
configuration file associated to this schedule. The ReportConfig is
initialized for the $superservice schema. If $merging is set, a
ReportConfig object that can handle merging is created.

Additionanlly, a FilterSpec will have been added to each section of
the ReportConfig object which will restrict the reporting on the
period defined for the scedule. The boundaries of the period are
determined according to $time which defaults to now.

=cut

sub init_report_config {
    my ( $self, $superservice, $merging, $time ) = @_;

    check_param( $superservice, 'superservice' );
    $time ||= time();
    my $factory = $merging ? new Lire::AsciiDlf::AsciiDlfFactory() : undef;

    my $cfg = new_from_file Lire::ReportConfig( $superservice,
                                                $self->{'_cfg_file'},
                                                $factory );

    my $range = $self->period_range( $time );
    my $filter_title = $self->_filter_title( $time );
    foreach my $sect ( $cfg->sections() ) {
        my $section_schema = ( $sect->reports() )[0]->schema();
        my $field = '$' . $section_schema->timestamp_field()->name();
        my $spec = new Lire::FilterSpec();
        $spec->superservice( $section_schema->superservice() );
        $spec->schema( $section_schema->id() )
          if $section_schema->superservice() ne $section_schema->id();
        $spec->display_title( $filter_title );

        my $expr = new Lire::FilterExpr::And( 'container' => $spec, );
        $spec->filter_spec( $expr );
        my $start = new Lire::FilterExpr::Ge( 'container' => $spec,
                                              'arg1' => $field,
                                              'arg2' => $range->[0] );
        my $end = new Lire::FilterExpr::Lt( 'container' => $spec,
                                            'arg1' => $field,
                                            'arg2' => $range->[1] );
        $expr->expr( [ $start, $end ] );
        $sect->add_filter( $spec );
    }

    return $cfg;
}

sub _filter_title {
    my ( $self, $time ) = @_;

    my $week_calc = new Lire::WeekCalculator();
    my $timefmt;
    if ( $self->{'_period'} eq 'hourly' ) {
        $timefmt = '%Y-%m-%d %H:%M';
    } elsif ( $self->{'_period'} eq 'daily' ) {
        $timefmt = '%Y-%m-%d';
    } elsif ( $self->{'_period'} eq 'weekly' ) {
        $timefmt = $week_calc->strformat();
    } elsif ( $self->{'_period'} eq 'monthly' ) {
        $timefmt = '%Y-%m';
    } elsif ( $self->{'_period'} eq 'yearly' ) {
        $timefmt = '%Y';
    }
    return __x( 'DLF records for {time}.',
                'time' => $week_calc->strfdate( $timefmt, localtime( $time ) ));
}
=pod

=head2 add_output_jobs( $job, ... )

Adds one or more Lire::OutputJob to the ReportSchedule object.

=cut

sub add_output_job {
    my ( $self, @jobs ) = @_;

    croak "missing one or more 'output_job' parameters"
      unless @jobs;

    foreach my $job ( @jobs ) {
        check_object_param( $job, 'output_job', 'Lire::OutputJob' );
        push @{$self->{'_output_jobs'}}, $job;
    }

    return;
}

=pod

=head2 output_jobs()

Returns the Lire::OutputJobs related to this object.

=cut

sub output_jobs {
    return @{$_[0]{'_output_jobs'}};
}

=pod

=head2 run( $store, $report_job, [$time]  )

Generate a XML report and save it in the $store Lire::DlfStore. The
report will be generated either using the DlfStreams or previously
generated reports. Preferences is giving to generating the report
using the Dlf data For more details, consult the documentation
of find_report_source() in Lire::DlfStore(3pm).

The period for which the report will be generated is done using the
$time parameter which defaults to now. For example, a 'daily' report
will generate a report for the whole day (midnight-midnidht based on
the day that $time represents).

Once the report is generated, all registered OutputJob will be run
with the new report.

=cut

sub run {
    my ( $self, $store, $report_job, $time ) = @_;

    my $source = $store->find_report_source( $report_job, $self, $time );
    return if $source->{'source'} eq 'none';

    my $cfg = $self->init_report_config( $report_job->superservice(),
                                         $source->{'source'} eq 'merging',
                                         $time );
    my $report;
    if ( $source->{'source'} eq 'merging' ) {
        my $merger = new Lire::ReportMerger( $cfg, @{$source->{'reports'}} );
        $report = $merger->merge_reports();
    } else {
        my $generator = new Lire::ReportGenerator( $cfg, $store );
        $report = $generator->generate_report();
    }

    my $report_file = $store->put_report( $report_job, $self, $report );
    foreach my $job ( $self->output_jobs() ) {
        $job->run( $report_file, $time );
    }
    return;
}

sub new_from_config {
    my ( $pkg, $dict ) = @_;

    my $sched = new Lire::ReportSchedule( $dict->{'period'},
                                          $dict->{'report_cfg'});
    $sched->add_output_job( @{$dict->{'output_jobs'}} )
      if @{$dict->{'output_jobs'}};

    return $sched;
}

1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfStore(3pm) Lire::DlfConverter(3pm)

=head1 VERSION

$Id: ReportSchedule.pm,v 1.5 2004/04/07 18:15:37 flacoste Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

