package Lire::UI::CompoundWidget;

use strict;

use base qw/ Curses::UI::Container Lire::UI::Widget /;

use Carp;
use Curses;
use Curses::UI::Common;
use Locale::TextDomain 'lire';

use Lire::Utils qw/ check_object_param /;

use vars qw/ @CARP_NOT %bindings %routines /;

@CARP_NOT = qw/ Curses::UI::Container /;

%bindings = ( KEY_PPAGE() => 'pageup',
              KEY_NPAGE() => 'pagedown' );
%routines = ( 'pageup' => \&_help_pageup,
              'pagedown' => \&_help_pagedown );

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase(\%userargs);

    check_object_param( $userargs{'value'}, 'value',
                        'Lire::Config::Dictionary' );

    my $self = $class->Curses::UI::Container::new( %userargs,
                                                   '-bindings' => \%bindings,
                                                   '-routines' => \%routines );

    $self->_setup_value_widgets();
    $self->_setup_help_viewer();

    return $self;
}

sub _setup_empty_label {
    my $self = $_[0];

    $self->add('empty_label', 'Label',
               '-text'=>__('-- This option does not have any attributes --'));
}

sub _setup_value_widgets {
    my $self = $_[0];

    my @components = $self->{'value'}->spec()->components();

    unless ( @components ) {
        $self->_setup_empty_label();
        return;
    }

    $self->{'_label_width'} = $self->_max_label_size();
    $self->{'_widget_width'} = ( $self->canvaswidth() - $self->{'_label_width'}
                                 - 32 );
    $self->{'_y'} = 0;

    foreach my $comp ( @components ) {
        my $name = $comp->name();
        $self->add( "${name}_label", 'Label',
                    '-text' => $name, '-bold' => 1,
                    '-textalignment' => 'right',
                    '-x' => 0, '-y' => $self->{'_y'},
                    '-width' => $self->{'_label_width'},
                  );
        my $widget = $self->add( "${name}_widget", 'Lire::UI::Widget',
                                 'value' => $self->{'value'}->get( $name ),
                                 '-onfocus' => \&_child_focus_cb,
                                 '-x' => $self->{'_label_width'} + 1,
                                 '-y' => $self->{'_y'},
                                 '-width' => $self->{'_widget_width'},
                               );
        $self->{'_y'} += $widget->{'-height'} + 1;
    }

    return;
}

# FIXME: this is hackish
sub _help_onfocus_cb {
    my $help = $_[0];

    my $self = $help->parent();
    if ( $help->{'_tag'} ) {
        $self->focus_next();
        $help->{'_tag'} = 0;
    } else {
        $self->parent()->focus_next();
        $help->{'_tag'} = 1;
    }

    return;
}

sub _setup_help_viewer {
    my $self = $_[0];

    return unless $self->{'value'}->spec()->components();

    my $x = $self->{'_label_width'} + $self->{'_widget_width'} + 2;
    $self->add( 'help_summary', 'Label', '-bold' => 1,
                '-x' => $x, '-y' => 0, '-width' => 30 );
    my $viewer = $self->add( 'help_viewer', 'TextViewer',
                             '_tag' => 0,
                             '-wrapping' => 1, '-vscrollbar' => 1,
                             '-x' => $x, '-y' => 2, '-width' => 30 );
    $viewer->onFocus( \&_help_onfocus_cb );

    return;
}

sub _help_pageup {
    my $self = $_[0];

    my $help_viewer = $self->getobj( 'help_viewer' );
    $help_viewer->cursor_pageup();
    $help_viewer->draw();

    return;
}

sub _help_pagedown {
    my $self = $_[0];

    my $help_viewer = $self->getobj( 'help_viewer' );
    $help_viewer->cursor_pagedown();
    $help_viewer->draw();

    return;
}

sub refresh_view {
    my $self = $_[0];

    foreach my $comp ( $self->{'value'}->spec()->components() ) {
        my $name = $comp->name();
        die "$name: $self" unless $self->isa( 'Lire::UI::CompoundWidget' );
        $self->getobj( "${name}_widget" )->refresh_view();
    }

    return;
}

sub _max_label_size {
    my $self = $_[0];

    my @labels = map { $_->name() } $self->{'value'}->spec()->components();

    my $max = 0;
    foreach my $label ( @labels ) {
        my $length = length( $label );
        $max = $length if ( $max < $length );
    }

    return ( $max < 20 ? $max : 20 );
}

# callbacks
sub _child_focus_cb {
    my $widget = $_[0];

    my $self = $widget->parent();

    my $value = $widget->{'value'};

    $self->getobj( 'help_summary' )->text( $value->summary() );
    $self->getobj( 'help_viewer' )->text( $value->text_description()
                                          || __( 'No help available.' ) );

    return;
}

1;
