package Lire::UI::Utils;

use strict;

use base qw/ Exporter /;

use Carp;

use Lire::Utils qw/ check_param /;

use vars qw( @EXPORT_OK );

BEGIN {
    @EXPORT_OK	= qw/ button_box_width text_for_width /;
}

=pod

=head1 NAME

Lire::UI::Utils - Utility functions for UI management

=head1 SYNOPSIS

  use Lire::UI::Utils;

=head1 DESCRIPTION

XXX FIXME

=head2 button_box_width( $buttons )

Computes the width required to draw the button box described by the 'buttons'
parameter. This parameter is a hash reference.

=cut

sub button_box_width {
    my $buttons = $_[0];

    my $width = scalar ( @$buttons ) - 1;
    foreach my $button ( @$buttons ) {
        $width += length( $button->{'-label'} );
    }

    return $width;
}

=pod

=head2 text_for_width( $text, $width )

Returns a stripped-down representation of 'text', ensuring its length is
shorter or equal to 'width'. If the original text already fits the given
width, it is returned unchanged, otherwise, it is shortened and '...' is put
in the middle to indicate the cut.

=cut

sub text_for_width {
    my ( $text, $width ) = @_;

    check_param( $text, 'text' );
    check_param( $width, 'width',
                 sub { return ( $_[0] =~ m/^[0-9]+$/
                                && $_[0] >= 5 ) },
                 "'width' should be a positive, greater than or equal to 5, integer" );

    my $length = length( $text );
    return $text if $length <= $width;

    my $segment = ( $width - 3 ) / 2;
    my $start = substr( $text, 0, $segment );
    $start =~ s/\s*$//g;
    my $end = substr( $text, $length - $segment );
    $end =~ s/^\s*//g;

    return $start . '...' . $end;
}

1;
__END__

=pod

=head1 SEE ALSO

Lire::UI(3pm)

=head1 VERSION

$Id: Utils.pm,v 1.3 2004/03/30 14:49:04 wsourdeau Exp $

=head1 AUTHORS

Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
