package tests::AsciiDlfTimegroupTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use vars qw/$VERSION /;

use Lire::DlfSchema;
use Lire::Report;
use Lire::Report::Subreport;
use Lire::ReportSpec;
use Lire::AsciiDlf::Timegroup;
use Lire::AsciiDlf::GroupOp;

use Time::Local;

use File::Basename qw/dirname/;
use Cwd qw/realpath/;

our ($VERSION) = '$Revision: 1.9 $' =~ m!Revision: ([.\d]+)!;

sub set_up {
    my ( $self ) = shift->SUPER::set_up();

    $self->{'cfg'}{'lr_scale_numbers'} = 0;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    $self->{'schemasdir'} = realpath( dirname(__FILE__) . "/schemas" );

    # Make sure the Lire::DlfSchema can find our test schema
    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'schemasdir'} ];

    # Create a simple report spec containing a Timegroup and a count
    # operation
    $self->{'report_spec'} =  new Lire::ReportSpec();
    $self->{'report_spec'}->id( 'test-timegroup' );
    $self->{'report_spec'}->superservice( 'test' );
    $self->{'report_spec'}->title( 'Test Timegroup Report Specification' );
    $self->{'report_spec'}->display_title( 'Test Timegroup Report' );

    $self->{'timegroup'} =
            new Lire::AsciiDlf::Timegroup( 'report_spec' => $self->{'report_spec'},
                                           'period' => '1d',
                                         );
    $self->{'report_spec'}->calc_spec( $self->{'timegroup'} );

    $self->{'count_op'} =
            new Lire::AsciiDlf::GroupOp::Count( 'report_spec' => $self->{'report_spec'},
                                                'parent' => $self->{'timegroup'},
                                                'name' => 'test_count',
                                         );
    $self->{'timegroup'}->ops( [ $self->{'count_op'} ] );
}

sub test_init_common {
    my $self = $_[0];

    my $timegroup = $self->{'timegroup'};
    $timegroup->init_common( undef, undef );
    $self->assert_not_null( $timegroup->{'_start_time_na'},
                            "_start_time_na wasn't set" );
}

sub test_merge_2days {
    my ( $self ) = @_;

    my $report_spec = $self->{'report_spec'};
    my $tg = $self->{'timegroup'};
    my $count = $self->{'count_op'};

    # Merge two days of data 2002-12-03 and 2002-12-04
    my $start = timelocal( 0, 0, 0, 3, 11, 102 );
    my $end   = $start + 2 * 86400;

    my $dec3 = 1200;
    my $dec4 = 657;

    my $report    = new Lire::Report( 'test', $start, $end );
    my $subreport1 = new Lire::Report::Subreport( $report, $report_spec->id );
    $subreport1->table_info( $report_spec->create_table_info );
    my $entry1 = $subreport1->create_entry;
    $entry1->add_name( '2002-12-03', $start, 86400 );
    $entry1->add_value( 'content' => $dec3 );
    $subreport1->set_summary_value( $count->name, 'content' => $dec3 );

    my $subreport2 = new Lire::Report::Subreport( $report, $report_spec->id );
    $subreport2->table_info( $report_spec->create_table_info );
    my $entry2 = $subreport2->create_entry;
    $entry2->add_name( '2002-12-04', $start + 86400, 86400 );
    $entry2->add_value( 'content' => $dec4 );
    $subreport2->set_summary_value( $count->name, 'content' => $dec4 );

    my $merged = new Lire::Report::Subreport( $report, $report_spec->id );
    $merged->table_info( $report_spec->create_table_info );

    $tg->init_merge( $start, $end );
    $tg->merge_subreport( $subreport1 );
    $tg->merge_subreport( $subreport2 );
    $tg->end_merge;
    $tg->create_entries( $merged );

    my @entries = $merged->entries;
    $self->assert_equals( 2, scalar @entries );

    my $sum = $merged->get_summary_value( $count->name );
    $self->assert_not_null( $sum, "summary value is undefined" );
    $self->assert_equals( $dec3 + $dec4, $sum->{'value'} );

    my ($dec3_n) = $entries[0]->names;
    $self->assert_not_null( $dec3_n, "dec3_n is undefined" );
    $self->assert( $start, $dec3_n->{'value'} );

    my ($dec4_n) = $entries[1]->names;
    $self->assert_not_null( $dec4_n, "dec4_n is undefined" );
    $self->assert( $start + 86400, $dec4_n->{'value'} );

    my $dec3_v = $entries[0]->data_by_name( $count->name );
    $self->assert_not_null( $dec4_n, "dec3_v is undefined" );
    $self->assert_equals( $dec3, $dec3_v->{'value'} );

    my $dec4_v = $entries[1]->data_by_name( $count->name );
    $self->assert_not_null( $dec4_n, "dec4_v is undefined" );
    $self->assert_equals( $dec4, $dec4_v->{'value'} );
}

1;
