package tests::ConfigValuesTest;

use strict;

use base qw/Test::Unit::TestSuite/;

use Lire::Config::TypeSpec;
use Lire::Config::Value;
use Carp;

sub name {
    return "Configuration Values Tests";
}

sub include_tests {
    return qw/tests::ConfigValuesScalarTest tests::ConfigValuesListTest
              tests::ConfigValuesDictionaryTest /;
}

package tests::ConfigValuesValueTest;

use base qw/Lire::Test::TestCase/;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'config_spec'} = new Lire::Config::ConfigSpec();
    $self->{'list_spec'} = new Lire::Config::ListSpec( 'name' => "list" );
    $self->{'config_spec'}->add( $self->{'list_spec'} );
    $self->{'int_spec'} = new Lire::Config::IntegerSpec( 'name' => "int" );
    $self->{'config_spec'}->add( $self->{'int_spec'} );
    $self->{'string_spec'} = new Lire::Config::StringSpec( 'name' => "string" );
    $self->{'config_spec'}->add( $self->{'string_spec'} );
    $self->{'list_spec'}->add( $self->{'int_spec'} );
    $self->{'select_spec'} = new Lire::Config::SelectSpec( 'name' => "select" );
    $self->{'config_spec'}->add( $self->{'select_spec'} );
    $self->{'list_spec'}->add( $self->{'select_spec'} );
    $self->{'select_spec'}->add( new Lire::Config::OptionSpec( 'name' => "option_1" ));
    $self->{'select_spec'}->add( new Lire::Config::OptionSpec( 'name' => "option_2" ));

    return $self;
}

sub test_new {
    my $self = $_[0];

    my $value = $self->type->new( 'spec' => $self->spec );
    $self->assert_not_null( $value, "new() returned undef" );
    $self->assert( UNIVERSAL::isa( $value, $self->type ),
                   "new() returned wrong type $value" );
    $self->assert_equals( $self->spec->name, $value->name );

    $self->assert_died( sub { $self->type->new },
                        qr/missing 'spec' parameter/ );

    $self->assert_died( sub { $self->type->new( 'spec' => $self ) },
                        qr/'spec' parameter.*should be a.*instance/ );
}

sub test_as_shell_var {
    my $self = $_[0];

    my $value = $self->type->new( 'spec' => $self->spec );
    my $name = $value->name;
    $self->assert_matches( qr/$name=.*not supported/, $value->as_shell_var );
}

sub test_clone {
    my $self = $_[0];

    my $value = $self->type->new( 'spec' => $self->spec );
    my $clone = $value->clone();

    $self->assert_str_not_equals( $value, $clone );
    $self->assert_deep_equals( $value, $clone );
    $self->assert_str_equals( $value->{'spec'}, $clone->{'spec'} );
}

package tests::ConfigValuesScalarTest;

use base qw/tests::ConfigValuesValueTest/;

sub type {
    return "Lire::Config::Scalar";
}

sub spec {
    return $_[0]{'int_spec'};
}

sub test_new {
    my $self = $_[0];

    $self->SUPER::test_new;

    my $value = new Lire::Config::Scalar( 'spec' => $self->{'int_spec'},
                                          'value' => 10,
                                        );

    $self->assert_equals( 10, $value->as_value );
}


sub test_as_value {
    my $self = $_[0];

    my $value = new Lire::Config::Scalar( 'spec' => $self->{'int_spec'} );
    $value->set( 10 );
    $self->assert_equals( 10, $value->as_value );

    $value = new Lire::Config::Scalar( 'spec' => $self->{'select_spec'} );
    $value->set( "OPTION_1" );
    $self->assert_equals( "OPTION_1", $value->get );
    $self->assert_equals( "option_1", $value->as_value );

    my $warning = '';
    local $SIG{'__WARN__'} = sub { $warning .= join "", @_ };
    $value = new Lire::Config::Scalar( 'spec' => $self->{'select_spec'} );
    $value->{'value'} = "no_such_option";
    my @array_context = $value->as_value;
    $self->assert_matches( qr/invalid value for parameter 'select':/,
                           $warning );
    $self->assert_deep_equals( [], \@array_context );

    # Second as_value() shouldn't emit another warning
    $warning = '';
    $self->assert_null( scalar $value->as_value,
                        "should return undef in scalar context" );
    $self->assert_equals( '', $warning );
}

sub test_as_shell_var {
    my $self = $_[0];

    my $int = $self->type->new( 'spec' => $self->{'int_spec'}, value => 10 );
    $self->assert_equals( "int='10'", $int->as_shell_var );

    my $undefined_int = $self->type->new( 'spec' => $self->{'int_spec'} );
    $self->assert_equals( "", $undefined_int->as_shell_var );

    my $s = q{> Long string with shell `metacharacters`, 'quote', "double"};
    my $quoted = q{> Long string with shell `metacharacters`, '\''quote'\'', "double"};
    my $string = $self->type->new( 'spec' => $self->{'string_spec'}, value => $s );
    $self->assert_equals( "string='$quoted'", $string->as_shell_var );

    local $SIG{'__WARN__'} = sub { $self->annotate( join "", @_ ) };
    my $bad_option = $self->type->new( 'spec' => $self->{'select_spec'},
                                       'value' => "bad_option" );
    $self->assert_equals( '', $bad_option->as_shell_var );
}

sub test_set {
    my $self = $_[0];

    my $value = new Lire::Config::Scalar( 'spec' => $self->{'int_spec'} );

    $value->set( 10 );
    $self->assert_equals( 10, $value->get );

    my $warning = undef;
    my $oldwarn = $SIG{'__WARN__'};
    $SIG{'__WARN__'} = sub { $warning = join(' ', @_); };
    $value->set('file');
    $SIG{'__WARN__'} = $oldwarn;
    $self->assert_null($warning, "set() with invalid value should NOT warn");
}

package tests::ConfigValuesListTest;

use base qw/tests::ConfigValuesValueTest/;

sub type {
    return "Lire::Config::List";
}

sub spec {
    return $_[0]{'list_spec'};
}

sub test_as_value {
    my $self = $_[0];

    local $SIG{'__WARN__'} = sub { $self->annotate( join "",  @_ ) };
    my $list = $self->{'list_spec'}->instance;
    $self->assert_deep_equals( [], $list->as_value );
    $list->append( $self->{'int_spec'}->instance( 'value' => 10 ) );
    $list->append( $self->{'select_spec'}->instance( 'value' => "option_1" ) );
    $list->append( $self->{'select_spec'}->instance( 'value' => "no_such_option" ) );
    $self->assert_deep_equals( [ 10, "option_1" ],
                               $list->as_value,
                             );
}

sub test_get {
    my $self = $_[0];

    my $list = $self->{'list_spec'}->instance;
    $self->assert_died( sub { $list->get( 0 ) },
                        qr/index out of bounds/);

    my $int = $self->{'int_spec'}->instance( 'value' => 10 );
    $list->append( $int );
    $self->assert_equals( $int, $list->get( 0 ) );
    $self->assert_equals( $int, $list->get( -1 ) );

    $self->assert_died( sub { $list->get( "string" ) },
                        qr/'idx' parameter should be an integer/);
}

sub test_set {
    my $self = $_[0];

    my $list = $self->{'list_spec'}->instance;
    $self->assert_died( sub { $list->set() },
                        qr/missing 'idx' parameter/ ); 
    my $int = $self->{'int_spec'}->instance( 'value' => 10 );
    $self->assert_died( sub { $list->set( 0, $int ) },
                        qr/index out of bounds: 0/ );
    $list->append( $int );
    $self->assert_died( sub { $list->set( 0, undef ) },
                        qr/missing 'value' parameter/ ); 

    $int = $self->{'int_spec'}->instance( 'value' => 5 );
    $list->set( 0, $int );
    $self->assert_str_equals( $int, $list->{'elements'}[0] );
}

sub test_append {
    my $self = $_[0];

    my $list = $self->{'list_spec'}->instance;
    $self->assert_equals( 0, scalar $list->elements );

    my $select = $self->{'select_spec'}->instance;
    my $int = $self->{'int_spec'}->instance( 'value' => 10 );
    $list->append( $select );
    $self->assert_equals( 1, scalar $list->elements );
    $list->append( $int );
    $self->assert_deep_equals( [ $select, $int ], [$list->elements] );

    $self->assert_died( sub { $list->append },
                        qr/missing 'value' parameter/ );
    $self->assert_died( sub { $list->append( $self->{'config_spec'}->instance )},
                        qr/cannot contains config parameters/ );

    my $bad_select = new Lire::Config::SelectSpec( 'name' => $self->{'select_spec'}->name );
    $self->assert_died( sub { $list->append( $bad_select->instance ) },
                        qr/is not a valid instance for component/ );
}

package tests::ConfigValuesDictionaryTest;

use base qw/tests::ConfigValuesValueTest/;

sub type {
    return "Lire::Config::Dictionary";
}

sub spec {
    return $_[0]{'config_spec'};
}

sub test_as_value {
    my $self = $_[0];

    local $SIG{'__WARN__'} = sub { $self->annotate( join "", $@ ) };
    my $dict = $self->{'config_spec'}->instance;
    $self->assert_deep_equals( { 'select' => '',
                                 'int'    => 0,
                                 'list'   => [],
                                 'string' => '',
                               },
                               $dict->as_value,
                             );

    $dict->set( $self->{'select_spec'}->instance( 'value' => "option_1" ) );
    $dict->set( $self->{'int_spec'}->instance( 'value' => "10" ) );
    $self->assert_deep_equals( {
                                'select' => "option_1",
                                'int'    => 10,
                                'list'   => [],
                                'string' => '',
                               },
                               $dict->as_value,
                             );
}

sub test_is_set {
    my $self = $_[0];

    my $dict = $self->{'config_spec'}->instance;

    $self->assert( !$dict->is_set( "int" ), "is_set() returned true" );
    $dict->get( 'int' );
    $self->assert( $dict->is_set( "int" ), "is_set() returned false" );

    $self->assert_died( sub { $dict->is_set },
                      qr/missing 'name' parameter/ );

    ;
    $self->assert_died( sub { $dict->is_set( "no_such_param") },
                        qr/no parameter no_such_param defined/ );
}

sub test_get {
    my $self = $_[0];

    my $dict = $self->{'config_spec'}->instance();
    my $int = $dict->get( "int" );
    $self->assert_not_null( $int, "get() should create the instance" );
    $self->assert_equals( "int", $int->name() );
    $self->assert_str_equals( '', $int->get() );

    $self->assert_equals( $int, $dict->get( "int" ) );

    $self->assert_died( sub { $dict->get() },
                        qr/missing 'name' parameter/ );

    $self->assert_died( sub { $dict->get( "bad_param" ) },
                      qr/no parameter bad_param defined in/);
}

sub test_set {
    my $self = $_[0];

    my $dict = $self->{'config_spec'}->instance;

    $self->assert_died( sub { $dict->set }, qr/missing 'value' parameter/ );

    $self->assert_died( sub{ $dict->set( $self->{'config_spec'} ) },
                        qr/\'value\' parameter should be a \'Lire::Config::Value\' instance, not/);

    my $int = $self->{'int_spec'}->instance;
    $dict->set( $int );
    $self->assert_equals( $int, $dict->get( 'int' ) );

    my $select = new Lire::Config::SelectSpec( 'name' => 'int' )->instance;
    ;
    $self->assert_died( sub { $dict->set( $select ) },
                        qr/is not a valid instance for parameter/ );
}

1;
