package tests::DlfStoreTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture /;

use Lire::DlfStore;
use Lire::Utils qw/create_file tempdir create_file /;
use Lire::Report;
use Lire::ReportJob;
use Lire::ReportSchedule;

use File::Path qw/mkpath/;
use Time::Local;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    $self->{'workingdir'} = tempdir( __PACKAGE__ . "XXXXXX",
                                     'CLEANUP' => 1, TMPDIR => 1,
                                   );

    my $spec = new Lire::Config::ConfigSpec();
    my $spec_string = new Lire::Config::StringSpec( 'name' => 'String' );
    $spec->add( $spec_string );
    $self->{'spec'} = $spec;

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'cfg'}{'lr_week_numbering'} = 'ISO';
    $self->{'cfg'}{'_lr_config_spec'} = $self->{'spec'};

    $self->set_up_tz( 'EST' );

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->tear_down_test_store()
      if $self->{'store'};

    return;
}

sub test_open_create {
    my $self = $_[0];

    my $path  = $self->{'workingdir'} . "/test_open_create";
    my $store = Lire::DlfStore->open( $path, 1 );
    $self->assert_not_null( $store, "Lire::DlfStore->open returned undef" );

    $self->assert_equals( $path, $store->path() );

    $self->assert( -d $path, "open() didn't create the directory" );
    $self->assert_not_null( $store->{'_dbh'}, "_dbh attribute is undef" );
    $self->assert( -f "$path/dlf.db",
                   "DLF database '$path/dlf.db' wasn't created" );

    $self->assert( -f "$path/lock", "open() didn't create the lock file" );
    $self->_check_dbh( $store );

    $self->assert( UNIVERSAL::isa( $store->{'_config'},
                                   'Lire::Config::ConfigFile' ),
                   "'Lire::Config::ConfigFile' expected: $store->{'_config'}");
    $self->assert_str_equals( "$path/config.xml",
                              $store->{'_config'}->filename());

    local $SIG{'__WARN__'} = sub { $self->annotate( @_ ) };
    $self->assert_deep_equals( { 'String' => undef },
                               $store->{'_config'}->as_value() );

    $store->close();
}

sub test_open {
    my $self = $_[0];

    my $registered_functions = 0;
    my $registered_aggregates = 0;

    no warnings 'redefine';
    local *Lire::SQLExt::Registry::register_functions =
      sub {
          $registered_functions = 1;
      };

    local *Lire::SQLExt::Registry::register_aggregates =
      sub {
          $registered_aggregates = 1;
      };

    my $bad_path = $self->{'workingdir'} . "/bad_store";
    mkdir $bad_path;
    $self->assert_died( sub { Lire::DlfStore->open( $bad_path ) },
                        qr/Invalid DlfStore: \'$bad_path\'/ );

    my $path = $self->{'workingdir'} . "/test_open";
    $self->assert_died( sub { Lire::DlfStore->open( $path ) },
                        qr/DlfStore \'$path\' doesn't exist/ );

    my $store = Lire::DlfStore->open( $path, 1 );
    $self->assert_not_null( $store, "open() returned undef" );

    $store->{'_config'}->get( 'String' )->set( 'wawa string' );
    $store->close();
    $store = Lire::DlfStore->open( $path );
    $self->assert_str_equals( 'wawa string',
                              $store->{'_config'}->get( 'String' )->get() );
    $self->assert_not_null( $store, "open() returned undef" );
    $self->assert_equals( $path, $store->path );
    $self->_check_dbh( $store );
    $store->close();

    $self->assert( $registered_functions,
                   'open() should register SQLExt functions' );
    $self->assert( $registered_aggregates,
                   'open() should register SQLExt aggregates' );
}

sub _check_dbh {
    my ($self, $store ) = @_;

    $self->assert( $store->{'_dbh'}, "_dbh attribute is undef" );
    $self->assert_equals( "SQLite", $store->{'_dbh'}{'Driver'}{'Name'} );
    $self->assert( $store->{'_dbh'}{'RaiseError'}, 
                   "RaiseError isn't enabled on _dbh"  );
    $self->assert( !$store->{'_dbh'}{'AutoCommit'},
                   "AutoCommit isn't turn off on _dbh"  );
}

sub test_close {
    my $self = $_[0];

    my $path = $self->{'workingdir'} . "/test_close";
    my $store = Lire::DlfStore->open( $path, 1);
    $self->assert_not_null( $store, "open()) returned undef" );
    my $warnings = "";
    local $SIG{'__WARN__'} = sub { $warnings .= join "", @_ };
    $self->assert( ! -e "$path/config.xml" );
    $store->close();
    $self->annotate( $warnings );
    $self->assert( ! -f "$path/lock", "close()) didn't remove the lock file" );
    $self->assert_null( $store->{'_dbh'}, "_dbh wasn't closed?" );
    $self->assert( !$warnings, "warnings were generated during close()" );
    $self->assert( -f "$path/config.xml",
                   'Expected config.xml to be created' );
}

sub test_locked_open {
    my $self = $_[0];

    my $path = $self->{'workingdir'} . "/test_locked_open";
    my $store = Lire::DlfStore->open( $path, 1);
    $store->close();
    create_file( "$path/lock", getppid . "\n" );
    $self->annotate( <<EORANT );
assert_died() is hosed because \$@ is lost in this particular case, ask
the "did-too-much-acid-in-their-time" perl's authors for a plausible
reason.
EORANT
    $self->assert_died( sub { Lire::DlfStore->open( $path ) },
                        qr/DlfStore '$path' is locked by process/ );
}

sub test_stale_lock {
    my $self = $_[0];

    my $path = $self->{'workingdir'} . "/test_state_lock";
    my $store = Lire::DlfStore->open( $path, 1);
    $self->assert_not_null( $store, "open()) returned undef" );
    $store->close;

    open LOCK, "> $path/lock"
      or die "can't create bogus lock file\n";
    print LOCK "-2\n";
    close LOCK;

    $store = Lire::DlfStore->open( $path );
    $self->assert_not_null( $store, "open() returned undef" );
    $store->close;
}

sub test_open_dlf_stream {
    my $self = $_[0];

    my $path = $self->{'workingdir'} . "/test_open_stream";
    my $store = Lire::DlfStore->open( $path, 1 );
    $self->assert_not_null( $store, "open() returned undef" );

    my @streams = $store->dlf_streams;
    $self->assert_equals( 0, scalar @streams );
    $self->assert( ! $store->has_dlf_stream( "test" ),
                   "store shouldn't have a test DLF stream." );

    $self->assert_died( sub { $store->open_dlf_stream( "test", "r" ) },
                        qr/no DLF stream 'test' in this store/ );
    my $s = $store->open_dlf_stream( "test", "w" );
    $self->assert( $s->isa( "Lire::DlfStream"),
                   "open_dlf_stream() didn't return a Lire::DlfStream instance: $s");

    my $stream_r = $store->open_dlf_stream( "test", "r" );
    $self->assert( $stream_r->isa( "Lire::DlfStream" ),
                   "open_dlf_stream() didn't return a Lire::DlfStream instance: $stream_r");
    $self->assert_equals( 'r', $stream_r->mode() );

    my $stream_w2 = $store->open_dlf_stream( "test", "w" );
    $self->assert( $stream_w2->isa( "Lire::DlfStream" ),
                   "open_dlf_stream() didn't return a Lire::DlfStream instance: $stream_w2");
    $self->assert_equals( 'w', $stream_w2->mode() );
    $s->close();
    $stream_r->close();
    $stream_w2->close();

    $s = $store->open_dlf_stream( "test", "r" );
    $self->assert( $s->isa( "Lire::DlfStream" ), "open_dlf_stream() failed" );

    $self->assert( $store->has_dlf_stream( "test"),
                   "has_dlf_stream() should have succeeded",
                 );
    @streams = $store->dlf_streams();
    $self->assert_deep_equals( [ "test"], \@streams );

    $s->close();
}

sub test_report_filename {
    my $self = $_[0];

    my $path = "$self->{'workingdir'}/test_report_filename";
    my $store = bless { '_store_path' => $path  }, 'Lire::DlfStore';

    $self->assert_dies( qr/'period' parameter should be one of 'hourly', 'daily', 'weekly', 'monthly' or 'yearly'/,
                        sub { $store->_report_filename( 'test', 'unique',
                                                        time ) } );
    my $jan7_2004 = timelocal( 0, 0, 0, 7, 0, 2004 );
    $self->assert_str_equals( "$path/hourly_reports/test/200401/07/00.xml",
                              $store->_report_filename( 'test', 'hourly',
                                                         $jan7_2004 ) );
    $self->assert_str_equals( "$path/daily_reports/test/200401/07.xml",
                              $store->_report_filename( 'test', 'daily',
                                                        $jan7_2004 ) );
    $self->assert_str_equals( "$path/weekly_reports/test/2004/02.xml",
                              $store->_report_filename( 'test', 'weekly',
                                                         $jan7_2004 ) );
    $self->assert_str_equals( "$path/monthly_reports/test/2004/01.xml",
                              $store->_report_filename( 'test', 'monthly',
                                                         $jan7_2004 ) );
    $self->assert_str_equals( "$path/yearly_reports/test/2004.xml",
                              $store->_report_filename( 'test', 'yearly',
                                                         $jan7_2004 ) );
}

sub test_put_report {
    my $self = $_[0];

    my $path = "$self->{'workingdir'}/test_put_report";
    my $store = Lire::DlfStore->open( $path, 1 );
    my $feb14_2004 = timelocal( 0, 5, 12, 14, 1, 2004 );
    my $report = new Lire::Report( 'test', $feb14_2004, $feb14_2004 + 3600 );

    my $job = new Lire::ReportJob( 'aTest', 'test' );
    my $sched = new Lire::ReportSchedule( 'hourly', 'report.cfg' );
    my $file = $store->put_report( $job, $sched, $report );
    $self->assert( -s $file, "report doesn't exists in store" );
    $self->assert_str_equals( "$path/hourly_reports/aTest/200402/14/12.xml",
                              $file );
}

sub test_find_report_source_dlf_only {
    my $self = $_[0];

    my $jan25_2003_noon = timelocal( 0, 5, 12, 25, 0, 2003 );
    $self->set_up_test_store();

    my $store = $self->{'store'};
    my $job = new Lire::ReportJob( 'aJob', 'test' );
    my $sched = new Lire::ReportSchedule( 'hourly', 'report.cfg' );
    $self->assert_deep_equals( { 'source'   => 'dlf', 
                                 'start'    => 1043514000,
                                 'end'      => 1043514000 + 3600,
                                 'coverage' => 100
                               }, $store->find_report_source( $job, $sched,
                                                              $jan25_2003_noon,
                                                            ) );

    my $jan25_2003_1600 = timelocal( 0, 5, 16, 25, 0, 2003 );
    $sched = new Lire::ReportSchedule( 'hourly', 'report.cfg' );
    $self->assert_deep_equals( { 'source'   => 'none',
                                 'coverage' => 0,
                               },
                               $store->find_report_source( $job, $sched, 
                                                           $jan25_2003_1600 ));
    $sched = new Lire::ReportSchedule( 'daily', 'report.cfg' );
    $self->assert_deep_equals( { 'source'   => 'dlf',
                                 'start'    => 1043514000,
                                 'end'      => 1043526410,
                                 'coverage' => 14,
                               }, $store->find_report_source( $job, $sched,
                                                              $jan25_2003_noon,
                                                            ) );
}

sub test_find_report_source {
    my $self = $_[0];

    my $jan25_2003_noon = timelocal( 0, 5, 12, 25, 0, 2003 );
    $self->set_up_test_store();

    my $store = $self->{'store'};
    my $job = new Lire::ReportJob( 'aJob', 'test' );
    my $sched = new Lire::ReportSchedule( 'weekly', 'report.cfg' );
    $self->assert_deep_equals( { 'source'   => 'dlf',
                                 'start'    => 1043514000,
                                 'end'      => 1043526410,
                                 'coverage' => 2,
                               }, $store->find_report_source( $job, $sched,
                                                              $jan25_2003_noon,
                                                            ) );

    my $dir = "$self->{'store'}{'_store_path'}/daily_reports/aJob/200301";
    mkpath( $dir, 0, 0755);
    for my $day ( 23, 24, 26, 27 ) {
        create_file( "$dir/$day.xml", '' );
    }
    $self->assert_deep_equals( { 'source'   => 'merging',
                                 'start'    => 1043298000,
                                 'end'      => 1043643600,
                                 'coverage' => 57,
                                 'reports'  => [ "$dir/23.xml", "$dir/24.xml",
                                                 "$dir/26.xml" ],
                                 'days' => [ '2003-01-23', '2003-01-24',
                                             '2003-01-26' ],
                               }, $store->find_report_source( $job, $sched,
                                                              $jan25_2003_noon,
                                                            ) );

    for my $day ( 19, 20, 21, 22, 25 ) {
        create_file( "$dir/$day.xml", '' );
    }
    $self->assert_deep_equals( { 'source'   => 'merging',
                                 'start'    => 1043038800,
                                 'end'      => 1043643600,
                                 'coverage' => 100,
                                 'reports'  => [ "$dir/20.xml", "$dir/21.xml",
                                                 "$dir/22.xml", "$dir/23.xml",
                                                 "$dir/24.xml", "$dir/25.xml",
                                                 "$dir/26.xml" ],
                                 'days' => [ '2003-01-20', '2003-01-21',
                                             '2003-01-22', '2003-01-23',
                                             '2003-01-24', '2003-01-25',
                                             '2003-01-26',
                                           ],
                               }, $store->find_report_source( $job, $sched,
                                                              $jan25_2003_noon,
                                                            ) );
}

1;
