package tests::OutputJobTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::OutputJob;
use Lire::OutputFormat;
use Class::Inner;
use Time::Local;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'spec'} = $self->lire_default_config_spec();

    $self->{'mock_output'} =
      new Class::Inner( 'parent' => 'Lire::OutputFormat',
                        'methods' =>
                        {
                         'new' => sub { return bless {}, shift },
                         'xml2mail_params' => sub { '-i -o html' },
                         'output_to_file' =>
                             sub { $_[0]{'output_to_file'} = [ @_ ] },
                        } );
    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'cfg'}{'_lr_config_spec'} = $self->{'spec'};

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_new_invalid {
    my $self = $_[0];

    $self->assert_dies( qr/'type' parameter should be one of 'email' or 'file'/,
                        sub { new Lire::OutputJob( 'type', 'type' ) } );
}

sub test_new_file {
    my $self = $_[0];

    my $file = '/var/www/report-%Y-%m-%d';
    my $job = new Lire::OutputJob( 'file', $self->{'mock_output'},
                                   'file' => $file );
    $self->assert_isa( 'Lire::FileOutputJob', $job );
    $self->assert_str_equals( $self->{'mock_output'}, $job->{'_format'} );
    $self->assert_str_equals( $self->{'mock_output'}, $job->format() );
    $self->assert_str_equals( $file, $job->{'_file'} );
    $self->assert_str_equals( $file, $job->file() );
}

sub test_output_file {
    my $self = $_[0];

    my $jan7_2004 = timelocal( 0, 0, 0, 7, 0, 2004 );
    my $file = '/var/www/report-%Y-%m-%d';
    my $job = new Lire::OutputJob( 'file', $self->{'mock_output'},
                                   'file' => $file );
    $self->assert_str_equals( '/var/www/report-2004-01-07',
                              $job->output_file( $jan7_2004 ) );
}

sub test_new_email {
    my $self = $_[0];

    my $emails = [ 'flacoste@logreport.org', 'wolfgang@logreport.org' ];
    my $job = new Lire::OutputJob( 'email', $self->{'mock_output'},
                                   'emails' => $emails,
                                   'subject' => 'A subject',
                                 );
    $self->assert_isa( 'Lire::EmailOutputJob', $job );
    $self->assert_str_equals( $self->{'mock_output'}, $job->{'_format'} );
    $self->assert_str_equals( $self->{'mock_output'}, $job->format() );
    $self->assert_str_equals( 'A subject', $job->{'_subject'} );
    $self->assert_str_equals( 'A subject', $job->subject() );
    $self->assert_deep_equals( $emails, $job->{'_emails'} );
    $self->assert_deep_equals( $emails, [ $job->emails() ] );
}

sub test_email_run {
    my $self = $_[0];

    my $job = new Lire::OutputJob( 'email', $self->{'mock_output'},
                                   'emails' => [ 'flacoste@logreport.org' ],
                                   'subject' => "LogReport's \\ \"subject\""
                                 );

    $self->assert_str_equals( 'lr_xml2mail -i -o html -s \'LogReport\'\\\'\'s \\ "subject"\' -x \'report.xml\' \'flacoste@logreport.org\'',
                              $job->_shell_cmd( 'report.xml' ) );
}

sub test_file_run {
    my $self = $_[0];

    my $jan7_2004 = timelocal( 0, 0, 0, 7, 0, 2004 );
    my $file = '/var/www/report-%Y-%m-%d';
    my $job = new Lire::OutputJob( 'file', $self->{'mock_output'},
                                   'file' => $file );
    $job->run( 'report.xml', $jan7_2004 );
    $self->assert_deep_equals( [ $self->{'mock_output'},
                                 'report.xml',
                                 '/var/www/report-2004-01-07' ],
                               $self->{'mock_output'}{'output_to_file'} );
}

sub test_new_from_config {
    my $self = $_[0];

    my $job = $self->{'spec'}->get( 'report_jobs' )->get( 'report_job' )->get( 'schedules' )->get( 'schedule' )->get( 'output_jobs' )->get( 'output_job' )->instance();
    $job->get( 'name' )->set( 'myJob' );
    $job->get( 'format' )->set_plugin( 'txt' );
    $job->get( 'format' )->get( 'encoding' )->set( 'UTF-8' );

    $job->get( 'destination' )->set_plugin( 'file' );
    $job->get( 'destination' )->get( 'file' )->set( 'report.txt' );

    my $jobObject = $job->as_value();
    $self->assert_isa( 'Lire::FileOutputJob', $jobObject );
    $self->assert_isa( 'Lire::OutputFormat::Text', $jobObject->{'_format'} );
    $self->assert_str_equals( 'UTF-8', $jobObject->format()->encoding() );
}

1;
