package tests::ReportJobTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture /;

use Lire::ReportJob;
use Lire::ReportSchedule;
use Class::Inner;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    $self->{'spec'} = $self->lire_default_config_spec();
    $self->{'test_cfg'} = "$self->{'testdir'}/data/test.cfg";

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();
    $self->set_up_test_store();
    $self->{'cfg'}{'_lr_config_spec'} = $self->{'spec'};

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->tear_down_test_store();

    return;
}

sub test_new {
    my $self = $_[0];

    my $job = new Lire::ReportJob( "aTest", 'test' );
    $self->assert_isa( 'Lire::ReportJob', $job );

    $self->assert_str_equals( 'aTest', $job->{'_name'} );
    $self->assert_str_equals( 'aTest', $job->name() );
    $self->assert_str_equals( 'test', $job->{'_superservice'} );
    $self->assert_str_equals( 'test', $job->superservice() );
    $self->assert_deep_equals( [], $job->{'_schedules'} );
    $self->assert_deep_equals( [], [ $job->schedules() ] );

    $self->assert_dies( qr/invalid 'superservice' parameter: 'aTest'/,
                        sub { new Lire::ReportJob( 'aTest', 'aTest' ) } );
}

sub test_add_shedule {
    my $self = $_[0];

    my $job = new Lire::ReportJob( "aTest", 'test' );
    my $schedule = bless {}, 'Lire::ReportSchedule';

    $self->assert_dies( qr/missing one or more 'schedule' parameters/,
                        sub { $job->add_schedule() } );
    $job->add_schedule( $schedule, $schedule );
    $self->assert_deep_equals( [ $schedule, $schedule ],
                               $job->{'_schedules'} );
}

sub test_new_from_config {
    my $self = $_[0];

    my $job = $self->{'spec'}->get( 'report_jobs' )->get( 'report_job' )->instance();
    $job->get( 'name' )->set( 'myReport' );
    $job->get( 'superservice' )->set( 'test' );

    my $jobObject = $job->as_value();
    $self->assert_isa( 'Lire::ReportJob', $jobObject );
    $self->assert_str_equals( 'myReport', $jobObject->{'_name'} );
    $self->assert_str_equals( 'test', $jobObject->{'_superservice'} );
    $self->assert_deep_equals( [], $jobObject->{'_schedules'} );

    my $sched = $job->spec()->get( 'schedules' )->get( 'schedule' )->instance();
    $sched->get( 'period' )->set( 'daily' );
    $sched->get( 'report_cfg' )->set( $self->{'test_cfg'} );
    $job->get( 'schedules' )->append( $sched );

    $sched = $sched->spec()->instance();
    $sched->get( 'period' )->set( 'hourly' );
    $sched->get( 'report_cfg' )->set( $self->{'test_cfg'} );
    $job->get( 'schedules' )->append( $sched );

    $jobObject = $job->as_value();
    my @sched = $jobObject->schedules();
    $self->assert_num_equals( 2, scalar @sched );
    $self->assert_isa( 'Lire::ReportSchedule', $sched[0] );
    $self->assert_isa( 'Lire::ReportSchedule', $sched[1] );
}

sub test_run {
    my $self = $_[0];

    my $job = new Lire::ReportJob( 'myReport', 'test' );
    my $daily = 
      new Class::Inner( 'parent' => 'Lire::ReportSchedule',
                        'args' => [ 'daily', $self->{'test_cfg'} ],
                        'methods' => { 'run' => sub { $_[0]{'params'} = [@_] } } );
    my $weekly =
      new Class::Inner( 'parent' => 'Lire::ReportSchedule',
                        'args' => [ 'weekly', $self->{'test_cfg'} ],
                        'methods' => { 'run' => sub { $_[0]{'params'} = [@_] } } );

    $job->add_schedule( $daily, $weekly );

    my $one_hour_ago = time - 3600;
    $job->run( $self->{'store'}, 'daily', $one_hour_ago );
    $self->assert_null( $weekly->{'params'}, 'weekly should not have been processed' );
    $self->assert_deep_equals( [ $daily, $self->{'store'}, $job,$one_hour_ago],
                               $daily->{'params'} );

    $daily->{'params'} = undef;
    my $now = time;
    $job->run( $self->{'store'}, 'weekly' );
    $self->assert_null( $daily->{'params'}, 'daily should not have been processed' );
    $self->assert_not_null( $weekly->{'params'}, 'weekly should have been processed' );
    $self->assert( $weekly->{'params'}[3] <= $now,
                   'default time should be now' );
}

1;
