package tests::ReportMergerTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::ReportConfig;
use Lire::ReportMerger;
use Lire::ReportParser::ReportBuilder;
use Lire::AsciiDlf::AsciiDlfFactory;
use File::Basename qw/dirname/;
use Cwd qw/realpath/;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'dir'} = realpath( dirname( __FILE__ ));

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_tz( 'EST' );

    $self->{'cfg'}{'lr_reports_path'} = [ $self->{'dir'} . "/reports" ];
    $self->{'cfg'}{'lr_filters_path'} = [ $self->{'dir'} . "/filters" ];
    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'dir'} . "/schemas" ];

    $self->{'cfg'}{'lr_scale_numbers'} = 0;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    $self->{'report_cfg'} =
      new_from_file Lire::ReportConfig( "test", $self->{'dir'} . "/data/test.cfg",
                                        new Lire::AsciiDlf::AsciiDlfFactory() );
    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];

    $self->assert_dies( qr/'report_cfg' parameter should be a 'Lire::ReportConfig' instance/,
                        sub { new Lire::ReportMerger( 'test' ) } );
    $self->assert_dies( qr/'report_cfg' parameter should have been initialized with a Lire::AsciiDlf::AsciiDlfFactory/,
                        sub { new Lire::ReportMerger( new Lire::ReportConfig( 'test' ), 'test.xml' ) } );
    $self->assert_dies( qr/new\(\) requires one or more 'file' parameters/,
                        sub { new Lire::ReportMerger( $self->{'report_cfg'} ) } );
}

sub test_self_merge_report {
    my $self = $_[0];

    my $parser = new Lire::ReportParser::ReportBuilder();

    my $report_e = $parser->parsefile( "$self->{'dir'}/data/test-merge.xml" );

    my $gen = new Lire::ReportMerger( $self->{'report_cfg'},
                                      $self->{'dir'} . "/data/test-sqlite.xml",
                                    );
    my $report_t = $gen->merge_reports();

    # The difference in these fields are normal
    $report_t->{'date'} = $report_e->{'date'};
    $report_t->generator( $report_e->generator() );

    $self->assert_deep_equals( $report_e, $report_t );
}

sub test_doublemerge_report {
    my $self = $_[0];

    my $parser = new Lire::ReportParser::ReportBuilder();

    my $report_e = $parser->parsefile( "$self->{'dir'}/data/test-doublemerge.xml" );

    my $gen = new Lire::ReportMerger( $self->{'report_cfg'},
                                      $self->{'dir'} . "/data/test-sqlite.xml",
                                      $self->{'dir'} . "/data/test-sqlite.xml",
                                    );
    my $report_t = $gen->merge_reports();

    # The difference in these fields are normal
    $report_t->{'date'} = $report_e->{'date'};
    $report_t->generator( $report_e->generator() );

    $self->assert_deep_equals( $report_e, $report_t );
}

sub test_merge_with_empty_report {
    my $self = $_[0];

    my $warnings = "";
    local $SIG{'__WARN__'} = sub { $warnings .= join "", @_ };

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report_e = $parser->parsefile( "$self->{'dir'}/data/test-merge.xml" );

    my $gen = new Lire::ReportMerger( $self->{'report_cfg'},
                                      $self->{'dir'} . "/data/test-sqlite.xml",
                                      $self->{'dir'} . "/data/test-empty-sqlite.xml",
                                    );
    my $report_t = $gen->merge_reports();
    $self->annotate( $warnings );
    $self->assert( !$warnings, "There were warnings" );
    # The difference in these fields are normal
    $report_t->{'date'} = $report_e->{'date'};
    $report_t->generator( $report_e->generator() );

    $self->assert_deep_equals( $report_e, $report_t );
}

sub test_two_empty_report {
    my $self = $_[0];

    my $warnings = "";
    local $SIG{'__WARN__'} = sub { $warnings .= join "", @_ };

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report_e = $parser->parsefile( "$self->{'dir'}/data/test-empty-merge.xml" );

    my $gen = new Lire::ReportMerger( $self->{'report_cfg'},
                                      $self->{'dir'} . "/data/test-empty-sqlite.xml",
                                      $self->{'dir'} . "/data/test-empty-sqlite.xml",
                                    );
    my $report_t = $gen->merge_reports();
    $self->annotate( $warnings );
    $self->assert( !$warnings, "There were warnings" );

    # The difference in these fields are normal
    $report_t->{'date'} = $report_e->{'date'};
    $report_t->generator( $report_e->generator() );

    $self->assert_deep_equals( $report_e, $report_t );
}

1;
