package tests::UIListWidgetTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::Config::TypeSpec;
use Lire::Utils qw/ item_index /;

use Curses::UI;
use File::Basename qw/ dirname basename /;

use Lire::Test::CursesUIDriver;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'driver'} = new Lire::Test::CursesUIDriver();
    $self->{'driver'}->setup_curses_ui();

    $self->{'ui'} = new Curses::UI();
    $self->{'driver'}->set_curses_ui( $self->{'ui'} );
    $self->{'window'} = $self->{'ui'}->add( 'window', 'Window' );

    my $spec = new Lire::Config::ConfigSpec();
    my $list_param = new Lire::Config::ListSpec( 'name' => 'list_param' );
    my $poly_list = new Lire::Config::ListSpec( 'name' => 'poly_list' );
    my $comp_list = new Lire::Config::ListSpec( 'name' => 'comp_list' );
    $spec->add( $list_param );
    $spec->add( $poly_list );
    $spec->add( $comp_list );

    my $string_param = new Lire::Config::StringSpec( 'name' => 'string' );
    $list_param->add( $string_param );
    $poly_list->add( new Lire::Config::StringSpec( 'name' => 'string' ) );
    $poly_list->add( new Lire::Config::BooleanSpec( 'name' => 'boolean' ) );

    my $compound = new Lire::Config::RecordSpec( 'name' => 'compound',
                                                 'label' => 'label' );
    my $label_spec = new Lire::Config::StringSpec( 'name' => 'label' );
    my $string1_spec = new Lire::Config::StringSpec( 'name' => 'string1' );
    my $string2_spec = new Lire::Config::StringSpec( 'name' => 'attribute2' );
    $compound->add( $label_spec );
    $compound->add( $string1_spec );
    $compound->add( $string2_spec );
    $comp_list->add( $compound );

    $self->{'spec'} = $spec;

    my $list_value = $self->{'list_value'} = $list_param->instance();
    $list_value->append( $string_param->instance( 'value' => 'string1' ) );
    $list_value->append( $string_param->instance( 'value' => 'string2' ) );
    $list_value->append( $string_param->instance( 'value' => 'string3' ) );
    $self->{'poly_value'} = $poly_list->instance();

    my $comp_value = $self->{'comp_value'} = $comp_list->instance();
    my $comp_inst = $compound->instance();
    $comp_inst->set( $label_spec->instance( 'value' => 'compvalue1' ) );
    $comp_inst->set( $string1_spec->instance( 'value' => 'wawa string' ) );
    $comp_inst->set( $string2_spec->instance( 'value' => 'wawa attr' ) );
    $comp_value->append( $comp_inst );

    $comp_inst = $compound->instance();
    $comp_inst->set( $label_spec->instance( 'value' => 'compvalue2' ) );
    $comp_inst->set( $string1_spec->instance( 'value' => 'test string' ) );
    $comp_inst->set( $string2_spec->instance( 'value' => 'wawatest attr' ) );
    $comp_value->append( $comp_inst );

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->{'driver'}->teardown_curses_ui();

    return;
}

sub test_new {
    my $self = $_[0];

    my $win = $self->{'window'};
    $self->assert_dies( qr/missing \'value\' parameter/,
                        sub { my $widget = $win->add( 'list_widget',
                                                      'Lire::UI::ListWidget' ) } );
    $self->assert_dies( qr/\'value\' parameter should be a 'Lire::Config::List' instance, not \'HASH/,
                        sub { my $widget = $win->add( 'list_widget',
                                                      'Lire::UI::ListWidget',
                                                      'Value' => {} ) } );
    $self->assert_dies( qr/\'poly_list\' contains more than one component/,
                        sub { my $widget = $win->add( 'list_widget',
                                                      'Lire::UI::ListWidget',
                                                      'Value' => $self->{'poly_value'} ) } );

    my $widget = $win->add( 'list_widget', 'Lire::UI::ListWidget',
                            'value' => $self->{'list_value'} );
    $self->assert( UNIVERSAL::isa( $widget, 'Lire::UI::ListWidget' ),
                   "not a Lire::UI::ListWidget instance: $widget" );
    $self->assert_str_equals( ($self->{'list_value'}->spec()->components())[0],
                              $widget->{'component'} );
    my $values_list = $widget->getobj( 'list' );
    $self->assert_not_null( $values_list, "missing 'list'" );
    $self->assert_str_equals( \&Lire::UI::ListWidget::_value_change_cb,
                              $values_list->{'-onchange'} );
    $self->assert_num_equals( 3, scalar @{$values_list->{'-values'}} );
    $self->assert_str_equals( 'string1', $values_list->{'-labels'}{ $self->{'list_value'}->get(0)} );

    my $value_widget = $widget->getobj( 'value_widget' );
    $self->assert_not_null( $value_widget, "missing 'value_widget'" );
    $self->assert_str_equals( 'string1', $value_widget->{'value'}->get() );

    my $buttons = $widget->getobj( 'buttons' );
    $self->assert_not_null( $buttons, "missing 'buttons'" );
    $self->assert_deep_equals( [ { '-label' => '[Add]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_add_cb },
                                 { '-label' => '[Apply]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_apply_cb },
                                 { '-label' => '[Delete]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_delete_cb }, ],
                               $buttons->{'-buttons'} );


    return;
}

sub test_new_compound {
    my $self = $_[0];

    my $win = $self->{'window'};
    my $widget = $win->add( 'list_widget',
                            'Lire::UI::ListWidget',
                            'value' => $self->{'comp_value'} );
    $self->assert_null( $widget->getobj( 'value_widget' ) );
    my $buttons = $widget->getobj( 'buttons' );
    $self->assert_not_null( $buttons, "missing 'buttons'" );
    $self->assert_deep_equals( [ { '-label' => '[Add]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_add_cb },
                                 { '-label' => '[Edit]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_edit_cb },
                                 { '-label' => '[Delete]',
                                   '-onpress' =>\&Lire::UI::ListWidget::_delete_cb }, ],
                               $buttons->{'-buttons'} );

    my $values_list = $widget->getobj( 'list' );
    $self->assert_num_equals( 2, scalar @{$values_list->{'-values'}} );
    $self->assert_deep_equals( [ 'compvalue1', 'compvalue2' ],
                               [ map { $values_list->{'-labels'}{$_} } @{$values_list->{'-values'}} ] );

    return;
}

sub test_refresh_view {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'list_value'} );

    # Add an empty value
    $self->{'list_value'}->append( $self->{'list_value'}->spec()->get( 'string' )->instance() );

    my $values_list = $widget->getobj( 'list' );
    $values_list->{'-values'} = [];
    $values_list->{'-labels'} = {};

    $widget->refresh_view();
    $self->assert_deep_equals( [ $self->{'list_value'}->get(0),
                                 $self->{'list_value'}->get(1),
                                 $self->{'list_value'}->get(2),
                                 $self->{'list_value'}->get(3) ],
                               $values_list->{'-values'} );

    $self->assert_deep_equals( { $self->{'list_value'}->get(0) => 'string1',
                                 $self->{'list_value'}->get(1) => 'string2',
                                 $self->{'list_value'}->get(2) => 'string3',
                                 $self->{'list_value'}->get(3) => '' },
                               $values_list->{'-labels'} );
    # deep_equals() thinks that '' and undef are equals
    $self->assert_str_equals( '',
                              $values_list->{'-labels'}{$self->{'list_value'}->get(3)} );
}

sub test_edit_cb {
    my $self = $_[0];

    return;
}

sub test_add_cb {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'list_value'} );

    my $buttons = $widget->getobj( 'buttons' );
    my $values_list = $widget->getobj( 'list' );
    my $value_widget = $widget->getobj( 'value_widget' );
    $value_widget->{'value'}->set( 'A string' );
    Lire::UI::ListWidget::_add_cb( $buttons );
    $self->assert_num_equals( 4, scalar $self->{'list_value'}->elements() );
    $self->assert_num_equals( 4, scalar @{$values_list->{'-values'}} );
    my $new_value = $self->{'list_value'}->get( 3 );
    $self->assert_str_equals( 'A string', $new_value->get() );
    $self->assert_str_equals( 'A string',
                              $values_list->{'-labels'}{$new_value} );
    $self->assert_str_not_equals( $new_value, $value_widget->{'value'} );
    $self->assert_str_equals( $new_value, $values_list->{'-values'}[3] );
}

sub test_edit_cb_compound {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'comp_value'} );

    my $buttons = $widget->getobj( 'buttons' );
    my $values_list = $widget->getobj( 'list' );

    $values_list->{'-selected'} = 1;
    $self->assert_deep_equals( [ 'compvalue1', 'compvalue2' ],
                               [ map { $values_list->{'-labels'}{$_} } @{$values_list->{'-values'}} ] );
    my $list_widget = '<undef>';
    my $value = '<undef>';

    no warnings 'redefine';
    local *Lire::UI::ListWidget::_edit_compound = sub {
        ( $list_widget, $value ) = @_;
        foreach my $elem ( $value->spec()->components() ) {
            $value->set( $elem->instance( 'value' => 'anotherval' ) );
        }
        return 1;
    };

    Lire::UI::ListWidget::_edit_cb( $buttons );
    $self->assert_deep_equals( [ 'compvalue1', 'anotherval' ],
                               [ map { $values_list->{'-labels'}{$_} } @{$values_list->{'-values'}} ] );
}

sub test_edit_compound {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'comp_value'} );
    my $compound = $self->{'comp_value'}->get( 0 );
    $self->assert_isa( 'Lire::Config::CompoundSpec',
                       $compound->spec() );

    my $ok_sub  = sub {
        my $dialog = $_[1];

        my $buttons = $dialog->getobj( 'buttons' );
        $buttons->{'-buttons'}[1]{'-onpress'}->();
    };

    $self->{'driver'}->add_event_loop_handler( $ok_sub );
    my $value = Lire::UI::ListWidget::_edit_compound( $widget, $compound );
    $self->assert_num_equals( 1, $value );

    my $cancel_sub  = sub {
        my $dialog = $_[1];

        my $buttons = $dialog->getobj( 'buttons' );
        $buttons->{'-buttons'}[0]{'-onpress'}->();
    };

    $self->{'driver'}->add_event_loop_handler( $cancel_sub );
    $value = Lire::UI::ListWidget::_edit_compound( $widget, $compound );
    $self->assert_num_equals( 0, $value );

    return;
}

sub test_add_cb_compound {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'comp_value'} );

    my $buttons = $widget->getobj( 'buttons' );
    my $values_list = $widget->getobj( 'list' );

    $self->assert_num_equals( 2, scalar @{$values_list->{'-values'}} );
    $self->assert_deep_equals( [ 'compvalue1', 'compvalue2' ],
                               [ map { $values_list->{'-labels'}{$_} } @{$values_list->{'-values'}} ] );
    my $list_widget = '<undef>';
    my $value = '<undef>';

    no warnings 'redefine';
    local *Lire::UI::ListWidget::_edit_compound = sub {
        ( $list_widget, $value ) = @_;
        foreach my $elem ( $value->spec()->components() ) {
            $value->set( $elem->instance( 'value' => 'anotherval' ) );
        }
        return 1;
     };

    Lire::UI::ListWidget::_add_cb( $buttons );

    $self->assert_num_equals( 3, scalar @{$values_list->{'-values'}} );
    $self->assert_deep_equals( [ 'compvalue1', 'compvalue2', 'anotherval' ],
                               [ map { $values_list->{'-labels'}{$_} } @{$values_list->{'-values'}} ] );
}

sub test_delete_cb {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'list_value'} );

    my $buttons = $widget->getobj( 'buttons' );
    my $values_list = $widget->getobj( 'list' );
    $self->assert_num_equals( 3, scalar @{$values_list->{'-values'}} );
    $values_list->{'-focusable'} = 0;
    Lire::UI::ListWidget::_delete_cb( $buttons );
    $self->assert_num_equals( 3, scalar @{$values_list->{'-values'}} );

    my $new_elements = [ $self->{'list_value'}->get( 0 ),
                         $self->{'list_value'}->get( 2 ) ];
    $values_list->{'-focusable'} = 1;
    $values_list->{'-ypos'} = 1;
    Lire::UI::ListWidget::_delete_cb( $buttons );
    $self->assert_deep_equals( $new_elements,
                               [ $self->{'list_value'}->elements() ] );
    $self->assert_deep_equals( $new_elements, $values_list->{'-values'} );
}

sub test_apply_cb {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'list_value'} );
    my $buttons = $widget->getobj( 'buttons' );
    my $values_list = $widget->getobj( 'list' );
    my $value_widget = $widget->getobj( 'value_widget' );
    $value_widget->{'value'}->set( 'New value' );
    $values_list->{'-selected'} = undef;
    Lire::UI::ListWidget::_apply_cb( $buttons );
    $self->assert_str_equals( 'string1', $widget->{'value'}->get(0)->get() );
    $self->assert_str_equals( 'string2', $widget->{'value'}->get(1)->get() );
    $self->assert_str_equals( 'string3', $widget->{'value'}->get(2)->get() );

    $values_list->{'-selected'} = 1;
    Lire::UI::ListWidget::_apply_cb( $buttons );
    my $value = $widget->{'value'}->get(1);
    $self->assert_str_equals( 'New value', $value->get() );
    $self->assert_str_equals( 'New value', $values_list->{'-labels'}{$value} );
}

sub test_value_change_cb {
    my $self = $_[0];

    my $widget = $self->{'window'}->add( 'list_widget', 'Lire::UI::ListWidget',
                                         'value' => $self->{'list_value'} );
    my $values_list = $widget->getobj( 'list' );
    my $value_widget = $widget->getobj( 'value_widget' );
    my $value = $value_widget->{'value'};
    $values_list->{'-selected'} = 1;
    Lire::UI::ListWidget::_value_change_cb( $values_list );
    $self->assert_str_equals( 'string2', $value_widget->{'value'}->get() );
    $self->assert_str_equals( 'string2', $value_widget->text() );

    $values_list->{'-selected'} = 0;
    Lire::UI::ListWidget::_value_change_cb( $values_list );
    $self->assert_str_equals( 'string1', $value_widget->text() );
    $self->assert_str_equals( 'string1', $value_widget->{'value'}->get() );

    $values_list->{'-selected'} = undef;
    $self->assert_dies( qr/\'-selected\' can never be undefined/,
                        sub { Lire::UI::ListWidget::_value_change_cb( $values_list ) } );
    $self->assert_str_equals( 'string1', $value_widget->text() );
    $self->assert_str_equals( 'string1', $value_widget->{'value'}->get() );
}

1;
