package tests::functional::LrXml2ReportTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase tests::functional::TestSchemaFixture/;

use Lire::Config::Build qw/ac_info/;
use Lire::ReportParser::ReportBuilder;
use File::Basename qw/dirname/;
use Cwd 'realpath';
use Carp;

sub new {
    my $self = shift->SUPER::new(@_);

    $self->tests::functional::TestSchemaFixture::init();

    $self->{'testsdir'} = realpath(dirname( __FILE__ ) . "/..");
    $self->{'report_cfg'} = "$self->{'testsdir'}/data/test.cfg";
    $self->{'report_xml'} = "$self->{'testsdir'}/data/test-sqlite.xml";
    $self->{'report13_xml'} = "$self->{'testsdir'}/data/test-1.3.xml";
    $self->{'empty_report_xml'} = "$self->{'testsdir'}/data/test-empty-sqlite.xml";

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schemas();
    $self->set_up_test_specs();
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();
    $self->tests::functional::TestSchemaFixture::tear_down();
}

sub test_version {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_xml2report -v" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    my $VERSION = ac_info( "VERSION" );
    $self->assert_matches( qr/shipped with Lire $VERSION/, $result->stderr );
}

sub test_merge_reports {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_xml2report -o xml -c $self->{'report_cfg'} -U test $self->{'report_xml'} $self->{'report_xml'} > report.xml" );

    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert( -f $self->rundir() . "/report.xml",
                   "XML report wasn't created" );
    $self->assert( -s $self->rundir() . "/report.xml",
                   "XML report is empty" );

    $self->check_xml_report( "$self->{'testsdir'}/data/test-doublemerge.xml",
                             $self->rundir() . "/report.xml" );
}

# The following two tests fail because the results of the records
# operator aren't merged.

# sub test_merge_13_reports {
#     my $self = $_[0];

#     my $result = $self->lire_run( "lr_xml2report -o xml -c $self->{'report_cfg'} -U test $self->{'report13_xml'} $self->{'report13_xml'} > report.xml" );

#     $self->annotate( $result->stderr() );
#     $self->assert_equals( 0, $result->status() );
#     $self->assert( !$result->stdout(), "stdout should be empty" );
#     $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
#                                   "There were warnings or error messages." );

#     $self->assert( -f $self->rundir() . "/report.xml",
#                    "XML report wasn't created" );
#     $self->assert( -s $self->rundir() . "/report.xml",
#                    "XML report is empty" );

#     $self->check_xml_report( "$self->{'testsdir'}/data/test-doublemerge.xml",
#                              $self->rundir . "/report.xml" );
# }

# sub test_merge_13_14_reports {
#     my $self = $_[0];

#     my $result = $self->lire_run( "lr_xml2report -o xml -c $self->{'report_cfg'} -U test $self->{'report13_xml'} $self->{'report_xml'} > report.xml" );

#     $self->annotate( $result->stderr() );
#     $self->assert_equals( 0, $result->status() );
#     $self->assert( !$result->stdout, "stdout should be empty" );
#     $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
#                                   "There were warnings or error messages." );

#     $self->assert( -f $self->rundir() . "/report.xml",
#                    "XML report wasn't created" );
#     $self->assert( -s $self->rundir() . "/report.xml",
#                    "XML report is empty" );

#     $self->check_xml_report( "$self->{'testsdir'}/data/test-doublemerge.xml",
#                              $self->rundir . "/report.xml" );
# }

sub test_double_empty_merge {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_xml2report -o xml -c $self->{'report_cfg'} -U test $self->{'empty_report_xml'} $self->{'empty_report_xml'} > report.xml" );

    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert( -f $self->rundir() . "/report.xml",
                   "XML report wasn't created" );
    $self->assert( -s $self->rundir() . "/report.xml",
                   "XML report is empty" );

    # Merging with an empty report shouldn't change anything.
    $self->check_xml_report( "$self->{'testsdir'}/data/test-empty-merge.xml",
                             $self->rundir() . "/report.xml" );
}

sub test_merge_reports_with_empty {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_xml2report -o xml -c $self->{'report_cfg'} -U test $self->{'report_xml'} $self->{'empty_report_xml'} > report.xml" );

    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert( -f $self->rundir() . "/report.xml",
                   "XML report wasn't created" );
    $self->assert( -s $self->rundir() . "/report.xml",
                   "XML report is empty" );

    # Merging with an empty report shouldn't change anything.
    $self->check_xml_report( "$self->{'testsdir'}/data/test-merge.xml",
                             $self->rundir() . "/report.xml" );
}

sub test_ascii_report {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_xml2report -o txt $self->{'report_xml'}" );
    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert( length $result->stdout(), "report is empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert_matches( qr/^Report generated:/, $result->stdout() );
    $self->assert_matches( qr/All Files/, $result->stdout() );
    $self->assert_matches( qr/User's Downloads Report/, $result->stdout() );
}

sub check_xml_report {
    my ($self, $expected_report, $gen_report ) = @_;

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report_e = $parser->parsefile( $expected_report );
    my $report_t = $parser->parsefile( $gen_report  );

    # The difference in these fields are normal
    $report_t->{'date'} = $report_e->{'date'};
    $report_t->generator( $report_e->generator );

    $self->assert_deep_equals( $report_e, $report_t );
}

1;
