package Lire::Firewall::SnortDlfConverter;

use strict;

use Lire::DlfConverter;
use Lire::Firewall qw/firewall_number2names/;
use Lire::Syslog;
use Carp;

use base qw/Lire::DlfConverter/;

sub new {
    my $proto = shift;
    bless {}, (ref $proto || $proto);
}

sub name { 'snort' }
sub title { 'SNORT network sniffer log' }
sub decription { '<para>SNORT network sniffer log</para>' }
sub schemas { qw/firewall/ }

sub handle_log_lines { 1 }

sub init_dlf_converter {
    my ($self, $process) = @_;

    $self->{'parser'} = new Lire::Syslog;
}

sub process_log_line {
    my ($self, $process, $line) = @_;

    #
    # skip invalid log entries
    #
    return $process->ignored_log_line($line)
        unless $line =~ /^...\s+\d+\s/;

    eval {
        my $log = $self->{'parser'}->parse($line);
        local $_ = $log->{'content'};
        #
        # skip invalid log entries
        #
        return $process->ignored_log_line($line)
            unless /^\[\d+:\d+:\d+\]/;

        #
        # split the log entries into the necessary fields
        #
        my @chain_infos =
                /^\[(\d+:\d+):(\d+)\]\s(.+)\s\{(.+)\}\s(.+)\s->\s(.+)\s*$/
            or croak "snort lexer failed\n";

        #
        # assign log values to hash
        #
        my %dlf = (
            'time' => $log->{'timestamp'},
            'count' => 1,
            'rule' => $chain_infos[1],
            'msg' => $chain_infos[2],
            'protocol' => $chain_infos[3]
        );
        @dlf{qw/from_ip from_port/} = split(/:/, $chain_infos[4]);
        $dlf{'from_host'} = $dlf{'from_ip'};
        @dlf{qw/to_ip to_port/} = split(/:/, $chain_infos[5]);
        $dlf{'to_host'} = $dlf{'to_ip'};

        #
        # convert numbers to names and create dlf-record
        #
        firewall_number2names(\%dlf);
        $process->write_dlf('firewall', \%dlf);
    };

    if($@) {
        $process->error($line, $@);
    }
}

sub finish_conversion {
    delete $_[0]->{'parser'};
}

1; # nag nag.

__END__

=pod

=head1 NAME

Lire::Firewall::SnortDlfConverter - convert Snort logs to firewall DLF

=head1 DESCRIPTION

B<Lire::Firewall::SnortDlfConverter> converts Snort logs into
firewall DLF format.
Input for this converter is the standard snort log file.

=head1 SEE ALSO

The Snort project website at http://www.snort.org/ .

=head1 AUTHOR

Code based upon snort2dlf code by
Torsten Fellhauer <torsten@fellhauer-web.de>.

=head1 VERSION

$Id: SnortDlfConverter.pm,v 1.12 2003/10/29 06:19:20 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2003 Stichting LogReport Foundation
Copyright (C) 2003 Torsten Fellhauer

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

# Local Variables:
# mode: cperl
# End:
