package Lire::Extensions::WWW::DomainSchema;

# vim:syntax=perl

use strict;

use Lire::AsciiDlf::ExtendedFieldsCreator;

use Lire::WWW::Domain;
use Lire::Config;
use Lire::Utils qw/tempfile/;
use Lire::Logger qw/lr_info/;

use Carp;

use base qw/ Lire::AsciiDlf::ExtendedFieldsCreator /;

sub init_computation {
    my ( $self, $dlf_info ) = @_;

    $self->{'domain_dflt'}  = $self->schema->field( "client_domain" )->default;
    $self->{'country_dflt'} = $self->schema->field( "country" )->default;

    $self->{'analyzer'} =
      new Lire::WWW::Domain( 'country_default' => $self->{'country_dflt'} );

    # Cache field's index
    $self->{'host_idx'}	  = $self->schema->field( "client_host" )->pos;
}

sub create_extended_fields {
    my ( $self, $dlf ) = @_;

    my $host = $dlf->[$self->{'host_idx'} ];
    $self->{'analyzer'}->setDomain($host);

    if ( ! defined $host ) {
	return [ undef, undef ];
    } elsif ( $host =~ /^(\d+\.\d+\.\d+)\.\d+$/ ) {
	return [ $1, $self->{'country_dflt'} ];
    } elsif ( $host =~ /([-a-zA-Z0-9]+\.[a-z]{3})$/ ||	# Non country TLD
	      $host =~ /([-a-zA-Z0-9]+\.[-a-zA-Z0-9]+\.[a-z]{2})$/ || # Country TLD
	      $host =~ /^([-a-zA-Z0-9]+)$/ # Hostname
	    )
    {
	return [ $1, $self->{'analyzer'}->getCountry() ];
    } else {
	# Uses default
	return [ $self->{'domain_dflt'}, $self->{'country_dflt'} ];
    }
}

sub end_computation {
    my ( $self ) = shift;

    $self->SUPER::end_computation( @_ );

    # Save hostnames that resulted in unknown values.
    if ( Lire::Config->get( 'lr_debug' ) ) {
	my ($fh, $filename) = tempfile( "unknown_country-XXXXXX",
					'SUFFIX' => '.txt' );
	lr_info( "saving unknown countries in ", $filename );

	foreach my $c ( keys %{$self->{'analyzer'}{'notInCountryHash'}} ) {
	    print $fh $c, "\n";
	}
	close $fh;
	unlink $filename unless -s $filename;
    }
    # Free memory
    delete $self->{'analyzer'};
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Extensions::WWW::CountrySchema -

=head1 SYNOPSIS

 use Lire::Extensions::WWW::CountrySchema;

=head1 DESCRIPTION

Interface to the Lire::WWW::Domain package.

=head1 VERSION

$Id: DomainSchema.pm,v 1.11 2004/03/20 00:26:29 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

E.L. Willighagen <egonw@logreport.org>

=cut
