/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/wizard.h"
#include "include/environment.h"

/*! \brief Konstruktor czarodzieja

	Tworzy czarodzieja, umiejscawiając go na klocku
	\param a_piece Klocek na którym ma być umieszczony czarodziej
*/	
LWWizard::LWWizard (LWPiece *a_piece):
visibility (TRUE), sleep_time(1.4), inverted_grounds(TRUE)
{
	piece = a_piece;
	direction = LW_DIRECTION_EAST;
}

/*! \brief Destruktor

	Niszczy czarodzieja
*/	
LWWizard::~LWWizard ()
{
	hide();
}

/*!	\brief Rekonstrukcja czarodzieja na podstawie opisu XML

	Rekonstrukcja czarodzieja na podstawie opisu XML zawartego w węźle 
	drzewa XML.
	\param node Węzeł drzewa XML na postawie którego dokonana zostanie
				rekonstrukcja
*/
void LWWizard::restoreFromXML (xmlNode *node)
{
LWPixmapSet *pixmapset = LWEnvironment::getPixmapSet();
	
	g_return_if_fail (pixmapset != NULL);
	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "Wizard"));	
	
	g_return_if_fail (NULL != xmlGetProp (node, (xmlChar *) "eastpixmap"));
	g_return_if_fail (NULL != xmlGetProp (node, (xmlChar *) "southpixmap"));
	g_return_if_fail (NULL != xmlGetProp (node, (xmlChar *) "westpixmap"));
	g_return_if_fail (NULL != xmlGetProp (node, (xmlChar *) "northpixmap"));

	direction_pixmaps[LW_DIRECTION_EAST] = 
		pixmapset->getPixmap ((gchar *) xmlGetProp (node, (xmlChar *) "eastpixmap"));
	
	direction_pixmaps[LW_DIRECTION_SOUTH] = 
		pixmapset->getPixmap ((gchar *) xmlGetProp (node, (xmlChar *) "southpixmap"));

	direction_pixmaps[LW_DIRECTION_WEST] = 
		pixmapset->getPixmap ((gchar *) xmlGetProp (node, (xmlChar *) "westpixmap"));

	direction_pixmaps[LW_DIRECTION_NORTH] = 
		pixmapset->getPixmap ((gchar *) xmlGetProp (node, (xmlChar *) "northpixmap"));

	if (visibility == TRUE) show ();
}

/*! \brief Ustawienie kierunku poruszania się czarodzieja

	Ustawia kierunek poruszania się czarodzieja
	\param a_direction Kierunek w którym będzie się poruszał czarodziej
*/
void LWWizard::setDirection (LWDirection a_direction)
{
	g_return_if_fail (direction_pixmaps[a_direction] != NULL);

	if (visibility == TRUE)
		piece->setForegroundPixmap (direction_pixmaps[a_direction]);

	direction = a_direction;
}

/*! \brief Pobranie kierunku, w którym porusza się czarodziej

	Pobiera kierunek, w którym porusza się czarodziej
	\return Kierunek
*/
LWDirection LWWizard::getDirection ()
{
	return direction;
}

/*! \brief Ustawianie prędkości poruszania się czarodzieja

	Ustawia prędkość poruszania się czarodzieja
	\param speed Prędkość z zakresu od 0 do 9
*/
void LWWizard::setSpeed (gint speed)
{
	sleep_time = (9.0 - (double) CLAMP(speed,0,9) ) * 0.2;
	
}

/*! \brief Obracanie czarodzieja

	Obraca czarodzieja w lewo lub w prawo
	\param turn Rodzaj zakrętu, lewy lub prawy
*/
void LWWizard::turn (LWTurn turn)
{
	switch (turn) {
		case LW_TURN_LEFT:
			switch (direction) {
				case LW_DIRECTION_EAST: 
					setDirection (LW_DIRECTION_NORTH); 
					return;
				case LW_DIRECTION_SOUTH:
					setDirection (LW_DIRECTION_EAST);
					return;
				case LW_DIRECTION_WEST:
					setDirection (LW_DIRECTION_SOUTH);
					return;
				case LW_DIRECTION_NORTH:
					setDirection (LW_DIRECTION_WEST);
					return;
			}
		case LW_TURN_RIGHT:
			switch (direction) {
				case LW_DIRECTION_EAST: 
					setDirection (LW_DIRECTION_SOUTH); 
					return;
				case LW_DIRECTION_SOUTH:
					setDirection (LW_DIRECTION_WEST);
					return;
				case LW_DIRECTION_WEST:
					setDirection (LW_DIRECTION_NORTH);
					return;
				case LW_DIRECTION_NORTH:
					setDirection (LW_DIRECTION_EAST);
					return;
			}
		}
}

/*! \brief Ukryj się

	Ukrywa czarodzieja (czyni go niewidocznym)
*/
void LWWizard::hide ()
{
	piece->setForegroundPixmap (NULL);
	visibility = FALSE;
}

/*! \brief Pojaw się

	Ukazuje czarodzieja (czyni go widocznym)
*/
void LWWizard::show ()
{
	g_return_if_fail (direction_pixmaps[direction] != NULL);

	piece->setForegroundPixmap (direction_pixmaps[direction]);
	visibility = TRUE;
}

static gboolean timeoutCb (gboolean *finished)
{
	*finished = TRUE;
	return FALSE;
}

/*! \brief Pauzuje czrodzieja

	Odstęp czasowy między ruchami
*/
void LWWizard::pause ()
{
	gboolean finished=FALSE;
	
	if (sleep_time >  0) {
		/* It's not very precise way of realising waiting,
		   but it doesn't consume whole processor time,
		   and serve events while waiting */
		
		g_timeout_add ((guint) (sleep_time * 1000.0), 
			(GSourceFunc) timeoutCb, (gpointer) &finished);
	
		do {
  			gtk_main_iteration ();
 		} while (finished == FALSE);

	} else {
		/* Serve pending events and exit */
		while (TRUE == gtk_events_pending ())
			gtk_main_iteration ();

	}
}

/*! \brief Pobranie N-tego klocka stojącego przed czarodziejem

	Pobiera N-ty klocek stojący przed czarodziejem. 
	Zerowy klocek to ten na którym znajduje się czarodziej.
	\param n Który klocek pobrać
	\return Pobrany klocek
*/
LWPiece *LWWizard::getPieceNth (gint n)
{
gint x,y;
LWPiece *gotpiece=NULL;
LWRow *gotrow;
	
	x = piece->getRow()->getPieceIndex(piece);
	
	switch (direction) {
		case LW_DIRECTION_EAST:
			gotpiece = piece->getRow()->getPieceNth(x+n);
			break;
		
		case LW_DIRECTION_SOUTH:
			y = piece->getRow()->getBoard()->getRowIndex (piece->getRow());
			gotrow = piece->getRow()->getBoard()->getRowNth (y+n);
			
			if (gotrow == NULL)
				return NULL;
			
			gotpiece = gotrow->getPieceNth(x);
			break;	

		case LW_DIRECTION_WEST:
			if (x-n < 0) 
				return NULL;

			gotpiece = piece->getRow()->getPieceNth(x-n);
			break;
		
		case LW_DIRECTION_NORTH:
			y = piece->getRow()->getBoard()->getRowIndex (piece->row);
		
			if (y-n < 0)
				return NULL;
			
			gotrow = piece->getRow()->getBoard()->getRowNth (y-n);
			
			if (gotrow == NULL)
				return NULL;
			
			gotpiece = gotrow->getPieceNth(x);
			break;
		}

	if (gotpiece == NULL)
		return NULL;
		
	gotpiece->enableInvertedGrounds (inverted_grounds);
		
	return gotpiece;
}

/*! \brief Rusz się do przodu

	Porusza się o krok do przodu.
	\return Jeśli nie ma miejsca przed czarodziejem na którym mógłby on
	stanąć, zwraca FAŁSZ. W przeciwnym razie PRAWDĘ
*/
gboolean LWWizard::stepForward ()
{
LWPiece *gotpiece;

	if (visibility == TRUE)
		piece->setForegroundPixmap (NULL);
	
	gotpiece = getPieceNth (1);
			
	if (gotpiece != NULL) piece = gotpiece;
		
	if (visibility == TRUE)
		show ();
	
	return (gotpiece != NULL) ? TRUE : FALSE;
}

/*! \brief Rusz się do tyłu

	Porusza się o krok do tyłu
	\return Jeśli nie ma miejsca za czarodziejem na którym mógłby on
	stanąć, zwraca FAŁSZ. W przeciwnym razie PRAWDĘ
*/
gboolean LWWizard::stepBack ()
{
LWPiece *gotpiece;

	if (visibility == TRUE)
		piece->setForegroundPixmap (NULL);
	
	gotpiece = getPieceNth (-1);
			
	if (gotpiece != NULL) piece = gotpiece;
		
	if (visibility == TRUE)
		show ();
	
	return (gotpiece != NULL) ? TRUE : FALSE;
}

/*! \brief Rusz się

	Rusza czarodzieja o zadaną ilość kroków.
	\param n Zadana ilość kroków, jeśli wartość ujemna porusza czarodzieja do tyłu
*/
void LWWizard::go (gint n, gboolean wait)
{
	if (n > 0)
		for (gint i = 0; i < n; i++) {
			
			if (wait == TRUE)
				pause();

			if (FALSE == stepForward())
				return;
				
		}
		
	if (n < 0)
		for (gint i = 0; i < - n; i++) {

			if (wait == TRUE)
				pause();
			
			if (FALSE == stepBack())
				return;
			
		}
}

/*! \brief Tworzenie obiektu przed czarodziejem

	Tworzy obiekt przed czarodziejem
	\param pixmap Ikona do stworzenia
	\param n Odległość od czarodzieja, w której ma być stworzony obiekt
	\return FAŁSZ jeśli nie ma takiej pozycji na planszy
*/
gboolean LWWizard::createOne (LWPixmap *pixmap, gint n)
{
	LWPiece *gotpiece = getPieceNth(n);
	
	if (gotpiece == NULL)
		return FALSE;

	gotpiece->setBackgroundPixmap (pixmap);	
	return TRUE;
}

/*! \brief Tworzenie warości przed czarodziejem

	Tworzy wartość przed czarodziejem
	\param pixmap Wartość do stworzenia
*/
void LWWizard::create (LWValue *value, gboolean wait)
{
	if (wait == TRUE)
		pause();

	GSList *list = value->getListPixmap();
	gint n = 1;
	
	for (GSList *l = list; l != NULL; l = l->next)
		if (TRUE == createOne((LWPixmap *) l->data, n))
			n++;
		else
			break;			
	
	if (visibility == TRUE)
		piece->setForegroundPixmap (NULL);

	piece = getPieceNth (n-1);

	if (visibility == TRUE)
		show();
		
	g_slist_free (list);
	
}

void LWWizard::setPiece (LWPiece *a_piece)
{
	if (piece == a_piece) return;
		
	if (visibility == TRUE)
		piece->setForegroundPixmap (NULL);
		
	piece = a_piece;

	if (visibility == TRUE)
		show();		
}

LWPiece *LWWizard::getPiece ()
{
	return piece;
}

/*! \brief Włącza odwrócone interpretowanie warstw

	Włącza lub wyłącza odwrócone interpretowanie warstw
	\param inverted Jeśli PRAWDA to zostanie włączone odwrócone
	inrepretowanie warstw, w przeciwnym wypadku wyłączone
*/
void LWWizard::enableInvertedGrounds (gboolean inverted)
{
	inverted_grounds = inverted;

	piece->enableInvertedGrounds (inverted);	
}
