//===- llvm/Support/TypeInfo.h - Support for type_info objects -*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file was developed by the LLVM research group and is distributed under
// the University of Illinois Open Source License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This class makes std::type_info objects behave like first class objects that
// can be put in maps and hashtables.  This code is based off of code in the
// Loki C++ library from the Modern C++ Design book.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_SUPPORT_TYPEINFO_H
#define LLVM_SUPPORT_TYPEINFO_H

#include <typeinfo>

namespace llvm {

struct TypeInfo {
  TypeInfo() {                     // needed for containers
    struct Nil {};  // Anonymous class distinct from all others...
    Info = &typeid(Nil);
  }

  TypeInfo(const std::type_info &ti) : Info(&ti) { // non-explicit
  }

  // Access for the wrapped std::type_info
  const std::type_info &get() const {
    return *Info;
  }

  // Compatibility functions
  bool before(const TypeInfo &rhs) const {
    return Info->before(*rhs.Info) != 0;
  }
  const char *getClassName() const {
    return Info->name();
  }

private:
  const std::type_info *Info;
};

// Comparison operators
inline bool operator==(const TypeInfo &lhs, const TypeInfo &rhs) {
  return lhs.get() == rhs.get();
}

inline bool operator<(const TypeInfo &lhs, const TypeInfo &rhs) {
  return lhs.before(rhs);
}

inline bool operator!=(const TypeInfo &lhs, const TypeInfo &rhs) {
  return !(lhs == rhs);
}

inline bool operator>(const TypeInfo &lhs, const TypeInfo &rhs) {
  return rhs < lhs;
}

inline bool operator<=(const TypeInfo &lhs, const TypeInfo &rhs) {
  return !(lhs > rhs);
}

inline bool operator>=(const TypeInfo &lhs, const TypeInfo &rhs) {
  return !(lhs < rhs);
}

} // End llvm namespace

#endif
