/*
 * channel_track.h - declaration of class channelTrack, a track-class which
 *                   contains all information about a channel and provides
 *                   a settings-window and an API for dealing with
 *                   sound-generator-plugins
 *
 * Linux MultiMedia Studio
 * Copyright (c) 2004-2005 Tobias Doerffel <tobydox@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifndef _CHANNEL_TRACK_H
#define _CHANNEL_TRACK_H

#include "qt3support.h"

#ifdef QT4

#include <QApplication>
#include <QPushButton>
#include <QPainter>

#else

#include <qapplication.h>
#include <qpushbutton.h>
#include <qpainter.h>

#endif

#include "track.h"
#include "mixer.h"
#include "midi_device.h"
#include "templates.h"


class QLineEdit;
class arpAndChordsTabWidget;
class channelButton;
class envelopeTabWidget;
class knob;
class lcdSpinBox;
class notePlayHandle;
class pianoWidget;
class soundGenerator;
class surroundArea;
class tabBar;


class channelTrack : public QWidget, public track, public midiDevice
{
	Q_OBJECT
public:
	channelTrack( trackContainer * _tc );
	virtual ~channelTrack();

	// used by sound-generator
	void FASTCALL processAudioBuffer( sampleFrame * _ab, Uint32 _frames,
						notePlayHandle * _n );

	// returns the frequency of a given tone & octave.
	// This function also includes base_tone & base_octave in
	// its calculations
	float FASTCALL frequency( notePlayHandle * _n ) const;
	Uint32 FASTCALL beatLen( notePlayHandle * _n ) const;


	// for capturing note-play-events -> need that for arpeggio,
	// filter and so on
	void FASTCALL playNote( notePlayHandle * _n );
	const QString & pluginName( void ) const;
	void FASTCALL deleteNotePluginData( notePlayHandle * _n );

	// channel-name-stuff
	inline const QString & name( void ) const
	{
		return( m_name );
	}
	void FASTCALL setName( const QString & _new_name );

	// volume & surround-position-stuff
	void FASTCALL setVolume( volume _new_volume );
	volume getVolume( void ) const;
	void FASTCALL setSurroundAreaPos( const QPoint & _p );
	const QPoint & surroundAreaPos( void ) const;
	
	// base-tone stuff
	void FASTCALL setBaseTone( tones _new_tone );
	void FASTCALL setBaseOctave( octaves _new_octave );
	inline tones baseTone( void ) const
	{
		return( m_baseTone );
	}
	inline octaves baseOctave( void ) const
	{
		return( m_baseOctave );
	}
	int FASTCALL masterKey( notePlayHandle * _n ) const;

	virtual bool FASTCALL play( const midiTime & _start,
					Uint32 _start_frame, Uint32 _frames,
							Uint32 _frame_base,
							Sint16 _tco_num = -1 );
	inline virtual trackTypes trackType( void ) const
	{
		return( m_trackType );
	}

	virtual trackContentObject * FASTCALL createTCO( const midiTime &
									_pos );


	virtual void FASTCALL saveTrackSpecificSettings( QDomDocument & _doc,
							QDomElement & _parent );
	virtual void FASTCALL loadTrackSpecificSettings( const QDomElement &
									_this );
	void FASTCALL loadPluginSettings( const QDomElement &
							_plugin_settings );

	QWidget * widgetStack( void )
	{
		return( m_widgetStack );
	}


public slots:
	void volValueChanged( float _new_value );
	void surroundAreaPosChanged( const QPoint & _new_p );
	void textChanged( const QString & _new_name );
	void toggledChannelButton( bool _on );
	void noteDone( const note & _n );


protected:
	// capture close-events for toggling channel-button in track
	void closeEvent( QCloseEvent * _ce );
	void focusInEvent( QFocusEvent * _fe );
	inline virtual QString nodeName( void ) const
	{
		return( "channeltrack" );
	}
	// invalidates all note-play-handles linked to this channel
	void invalidateAllMyNPH( void );


protected slots:
	void saveSettingsBtnClicked( void );


private:
	trackTypes m_trackType;

	tones m_baseTone;
	octaves m_baseOctave;

	QString m_name;

	QWidget * m_generalSettingsWidget;
	QLineEdit * m_channelNameLE;
	tabBar * m_tabBar;
	QWidget * m_widgetStack;
	
	// widgets in widget-stack
	soundGenerator * m_soundGenPluginWidget;
	envelopeTabWidget * m_envWidget;
	arpAndChordsTabWidget * m_arpWidget;

	// buttons for selecting widgets in widget-stack
	QPushButton * m_soundGenPluginBtn;
	QPushButton * m_envBtn;
	QPushButton * m_arpBtn;

	// knobs etc. in general-settings-widget
	knob * m_volumeKnob;
	surroundArea * m_surroundArea;
	lcdSpinBox * m_effectChannelNumber;
	QPushButton * m_saveSettingsBtn;

	// widgets in track-settings-widget

	knob * m_tswVolumeKnob;
	surroundArea * m_tswSurroundArea;
	channelButton * m_tswChannelButton;

	// test-piano at the bottom of every channel-settings-window
	pianoWidget * m_pianoWidget;


	friend class notePlayHandle;


signals:
	void noteFromMidiDeviceDone( const note & _n );

} ;




class channelButton : public QPushButton
{
public:
	channelButton( channelTrack * _channel_track ) :
		QPushButton( _channel_track->getTrackSettingsWidget() ),
		m_channelTrack( _channel_track )
	{
	}

	virtual ~channelButton()
	{
	}


protected:
	virtual void drawButtonLabel( QPainter * _p )
	{
		QString pn = m_channelTrack->pluginName() + ":";
		int extra = 
#ifdef QT4
			isChecked
#else
			isOn
#endif
				() ? -1 : -3;
		_p->setFont( pointSize<7>( _p->font() ) );
#ifdef QT4
		_p->setPen( QApplication::palette().buttonText().color() );
#else
		_p->setPen( QApplication::palette().color( QPalette::Normal,
						QColorGroup::ButtonText ) );
#endif
		_p->drawText( ( width() -
				QFontMetrics( _p->font() ).width( pn ) ) / 2 +
					extra, height() / 2 + extra, pn );
		_p->setPen( QColor( 0, 0, 0 ) );
		_p->drawText( ( width() -
				QFontMetrics( _p->font() ).width( text() ) ) /
				2 + extra, height() / 2 +
				QFontMetrics( _p->font() ).height() + extra,
								text() );
	}

private:
	channelTrack * m_channelTrack;

} ;


#endif
