/*
 * channel_track.h - declaration of class channelTrack, a track-class which
 *                   contains all information about a channel and provides
 *                   a settings-window and an API for dealing with instruments
 *
 * Copyright (c) 2004-2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifndef _CHANNEL_TRACK_H
#define _CHANNEL_TRACK_H

#include "qt3support.h"

#ifdef QT4

#include <QApplication>
#include <QPushButton>
#include <QPainter>

#else

#include <qapplication.h>
#include <qpushbutton.h>
#include <qpainter.h>

#endif

#include "track.h"
#include "mixer.h"
#include "midi_event_processor.h"
#include "gui_templates.h"
#include "tab_widget.h"


class QLineEdit;
class arpAndChordsTabWidget;
class channelButton;
class envelopeTabWidget;
class knob;
class lcdSpinBox;
class notePlayHandle;
class pianoWidget;
class presetPreviewPlayHandle;
class instrument;
class midiPort;
class midiTabWidget;
class surroundArea;
class audioPort;



class channelTrack : public QWidget, public track, public midiEventProcessor
{
	Q_OBJECT
public:
	channelTrack( trackContainer * _tc );
	virtual ~channelTrack();

	inline virtual trackTypes type( void ) const
	{
		return( m_trackType );
	}


	// used by instrument
	void FASTCALL processAudioBuffer( sampleFrame * _buf, Uint32 _frames,
						notePlayHandle * _n );

	virtual void FASTCALL processInEvent( const midiEvent & _me,
						const midiTime & _time );
	virtual void FASTCALL processOutEvent( const midiEvent & _me,
						const midiTime & _time );

	// returns the frequency of a given tone & octave.
	// This function also includes base_tone & base_octave in
	// its calculations
	float FASTCALL frequency( notePlayHandle * _n ) const;
	Uint32 FASTCALL beatLen( notePlayHandle * _n ) const;


	// for capturing note-play-events -> need that for arpeggio,
	// filter and so on
	void FASTCALL playNote( notePlayHandle * _n );
	const QString & pluginName( void ) const;
	void FASTCALL deleteNotePluginData( notePlayHandle * _n );

	// channel-name-stuff
	inline const QString & name( void ) const
	{
		return( m_name );
	}
	void FASTCALL setName( const QString & _new_name );

	// volume & surround-position-stuff
	void FASTCALL setVolume( volume _new_volume );
	volume getVolume( void ) const;
	void FASTCALL setSurroundAreaPos( const QPoint & _p );
	const QPoint & surroundAreaPos( void ) const;
	
	// base-tone stuff
	void FASTCALL setBaseTone( tones _new_tone );
	void FASTCALL setBaseOctave( octaves _new_octave );
	inline tones baseTone( void ) const
	{
		return( m_baseTone );
	}
	inline octaves baseOctave( void ) const
	{
		return( m_baseOctave );
	}
	int FASTCALL masterKey( notePlayHandle * _n ) const;

	// play everything in given frame-range - creates note-play-handles
	virtual bool FASTCALL play( const midiTime & _start,
					Uint32 _start_frame, Uint32 _frames,
							Uint32 _frame_base,
							Sint16 _tco_num = -1 );
	// create new track-content-object = pattern
	virtual trackContentObject * FASTCALL createTCO( const midiTime &
									_pos );


	// called by track
	virtual void FASTCALL saveTrackSpecificSettings( QDomDocument & _doc,
							QDomElement & _parent );
	virtual void FASTCALL loadTrackSpecificSettings( const QDomElement &
									_this );

	// load instrument whose name matches given one
	instrument * FASTCALL loadInstrument( const QString & _plugin_name );

	// parent for all internal tab-widgets
	QWidget * tabWidgetParent( void )
	{
		return( m_tabWidget );
	}


public slots:
	void volValueChanged( float _new_value );
	void surroundAreaPosChanged( const QPoint & _new_p );
	void textChanged( const QString & _new_name );
	void toggledChannelButton( bool _on );


protected:
	// capture close-events for toggling channel-button in track
	virtual void closeEvent( QCloseEvent * _ce );
	virtual void focusInEvent( QFocusEvent * _fe );
	virtual void dragEnterEvent( QDragEnterEvent * _dee );
	virtual void dropEvent( QDropEvent * _de );

	inline virtual QString nodeName( void ) const
	{
		return( "channeltrack" );
	}
	// invalidates all note-play-handles linked to this channel
	void invalidateAllMyNPH( void );

	void noteOffAll( void );


protected slots:
	void saveSettingsBtnClicked( void );


private:
	trackTypes m_trackType;

	midiPort * m_midiPort;

	audioPort * m_audioPort;


	notePlayHandle * m_notes[NOTES_PER_OCTAVE * OCTAVES];


	tones m_baseTone;
	octaves m_baseOctave;

	QString m_name;


	// widgets on the top of a channel-track-window
	tabWidget * m_generalSettingsWidget;
	QLineEdit * m_channelNameLE;
	knob * m_volumeKnob;
	surroundArea * m_surroundArea;
	lcdSpinBox * m_effectChannelNumber;
	QPushButton * m_saveSettingsBtn;

	
	// tab-widget with all children
	tabWidget * m_tabWidget;
	instrument * m_instrument;
	envelopeTabWidget * m_envWidget;
	arpAndChordsTabWidget * m_arpWidget;
	midiTabWidget * m_midiWidget;


	// test-piano at the bottom of every channel-settings-window
	pianoWidget * m_pianoWidget;


	// widgets in track-settings-widget
	knob * m_tswVolumeKnob;
	surroundArea * m_tswSurroundArea;
	channelButton * m_tswChannelButton;


	friend class channelButton;
	friend class notePlayHandle;
	friend class presetPreviewPlayHandle;


signals:
	void noteDone( const note & _n );

} ;




class channelButton : public QPushButton
{
public:
	channelButton( channelTrack * _channel_track );
	virtual ~channelButton();


protected:
	// since we want to draw a special label (instrument- and channel-name)
	// on our button, we have to re-implement this for doing so
	virtual void drawButtonLabel( QPainter * _p );

	// allow drops on this button - we just forward them to channel-track
	virtual void dragEnterEvent( QDragEnterEvent * _dee );
	virtual void dropEvent( QDropEvent * _de );


private:
	channelTrack * m_channelTrack;

} ;


#endif
