/*
 * fstserver.cpp - XFST-server
 *
 * Copyright (c) 2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

#include <vst/aeffectx.h>

#include <vector>
#include <map>

#include "fstserver.h"
#include "types.h"
#include "midi.h"
#include "communication.h"

char debugout[1024];


extern "C"
{
void wine_shared_premain( void );


void debugMsg( const char * _str )
{
	writeValue<Sint16>( DEBUG_MSG, 1 );
	writeString( _str, 1 );
}


void 
fst_error (const char *fmt, ...)
{
	va_list ap;
	char buffer[512];

	va_start (ap, fmt);
	vsnprintf (buffer, sizeof(buffer), fmt, ap);
	fst_error_callback (buffer);
	va_end (ap);
}

void 
default_fst_error_callback (const char *desc)
{
	debugMsg(desc);
}

void (*fst_error_callback)(const char *desc) = &default_fst_error_callback;


}




class VSTPlugin
{
public:
	VSTPlugin( const char * _plugin ) :
		m_handle( NULL ),
		m_fst( NULL ),
		m_plugin( NULL ),
		m_shm( NULL ),
		m_inputs( NULL ),
		m_outputs( NULL )
	{
		if( ( m_handle = fst_load( _plugin ) ) == NULL )
		{
			sprintf( debugout, "can't load plugin %s\n", _plugin );
			debugMsg( debugout );
			return;
		}

		if( ( m_fst = fst_instantiate( m_handle, hostCallback,
							this ) ) == NULL )
		{
			sprintf( debugout, "can't instantiate plugin %s\n",
								_plugin );
			debugMsg( debugout );
			return;
		}

		m_plugin = m_fst->plugin;

		/* set program to zero */
		/* i comment this out because it breaks dfx Geometer
		 * looks like we cant set programs for it
		 *
		m_plugin->dispatcher( m_plugin, effSetProgram, 0, 0, NULL, 0.0f); */
		// request rate and blocksize
		writeValue<Sint16>( GET_SAMPLE_RATE );
		writeValue<Sint16>( GET_BUFFER_SIZE );


		m_plugin->dispatcher( m_plugin, effMainsChanged, 0, 1, NULL,
									0.0f );
		if( fst_run_editor( m_fst ) )
		{
			sprintf( debugout, "cannot create editor\n" );
			debugMsg( debugout );
		}

		writeValue<Sint16>( SET_XID );
		writeValue<Sint32>( fst_get_XID( m_fst ) );
	}

	~VSTPlugin()
	{
		writeValue<Sint16>( QUIT_ACK );
		fst_close( m_fst );
		fst_unload( m_handle );
		delete[] m_inputs;
		delete[] m_outputs;
		if( m_shm != NULL )
		{
			shmdt( m_shm );
		}
	}

	void process( void )
	{
		// first we gonna post all MIDI-events we enqueued so far
		if( m_midiEvents.size() )
		{
			// since MIDI-events are not received immediately, we
			// have to have them stored somewhere even after
			// dispatcher-call, so we create static copies of the
			// data and post them
			static VstEvents events;
			static std::vector<VstMidiEvent> cur_events;
			cur_events = m_midiEvents;
			m_midiEvents.clear();
			for( unsigned int i = 0; i < cur_events.size(); ++i )
			{
				events.events[i] = (VstEvent *) &cur_events[i];
			}
			events.numEvents = cur_events.size();
			events.reserved = 0;
			m_plugin->dispatcher( m_fst->plugin, effProcessEvents,
							0, 0, &events, 0.0f );
		}

		// now we're ready to fetch sound from VST-plugin

		for( int i = 0; i < inputCount(); ++i )
		{
			m_inputs[i] = &m_shm[i * m_blockSize];
		}
		for( int i = 0; i < outputCount(); ++i )
		{
			m_outputs[i] = &m_shm[( i + inputCount() ) *
								m_blockSize];
		}

		if( m_fst->plugin->flags & effFlagsCanReplacing )
		{
			m_plugin->processReplacing( m_plugin, m_inputs,
								m_outputs,
								m_blockSize );
		}
		else
		{
			m_plugin->process( m_plugin, m_inputs, m_outputs,
								m_blockSize );
		}
		writeValue<Sint16>( PROCESS_DONE );
		// give plugin some idle-time for graphics-update and so on...
		m_plugin->dispatcher( m_plugin, effEditIdle, 0, 0, NULL, 0 );

	}

	void enqueueMidiEvent( const midiEvent & _event,
						const Uint32 _frames_ahead )
	{
		VstMidiEvent event;

		event.type = kVstMidiType;
		event.byteSize = 24;
		event.deltaFrames = _frames_ahead;
		event.flags = 0;
		event.detune = 0;
		event.noteLength = 0;
		event.noteOffset = 0;
		event.noteOffVelocity = 0;
		event.reserved1 = 0;
		event.reserved2 = 0;
		event.midiData[0] = _event.m_type + _event.m_channel;
		event.midiData[1] = _event.key();
		event.midiData[2] = _event.velocity();
		event.midiData[3] = 0;
		m_midiEvents.push_back( event );
	}

	void setSampleRate( Sint32 _rate )
	{
		m_plugin->dispatcher( m_plugin, effSetSampleRate, 0, 0, NULL, 
							    (float) _rate );
	}

	void setBlockSize( Uint32 _bsize )
	{
		if( _bsize == m_blockSize )
		{
			return;
		}
		m_blockSize = _bsize;
		resizeSharedMemory();
		m_plugin->dispatcher( m_plugin, effSetBlockSize, 0, _bsize,
								NULL, 0.0f );
	}

	Sint32 vstVersion( void ) const
	{
		return( m_plugin->dispatcher( m_plugin,
					effGetVstVersion, 0, 0, NULL, 0.0f ) );
	}

	const char * vstName( void ) const
	{
		static char buf[32];
		buf[0] = 0;
		m_plugin->dispatcher( m_plugin,
					effGetEffectName, 0, 0, buf, 0.0f );
		buf[31] = 0;
		return( buf );
	}

	const char * vstVendorString( void ) const
	{
		static char buf[64];
		buf[0] = 0;
		m_plugin->dispatcher( m_plugin,
					effGetVendorString, 0, 0, buf, 0.0f );
		buf[63] = 0;
		return( buf );
	}

	const char * vstProductString( void ) const
	{
		static char buf[64];
		buf[0] = 0;
		m_plugin->dispatcher( m_plugin,
					effGetProductString, 0, 0, buf, 0.0f );
		buf[63] = 0;
		return( buf );
	}

	Uint8 inputCount( void ) const
	{
		return( m_plugin->numInputs );
	}

	Uint8 outputCount( void ) const
	{
		return( m_plugin->numOutputs );
	}

	void resizeSharedMemory( void )
	{
		delete[] m_inputs;
		delete[] m_outputs;

		size_t s = ( inputCount() + outputCount() ) * m_blockSize *
								sizeof( float );
		if( m_shm )
		{
			shmdt( m_shm );
		}
		int shm_id;
		Uint16 shm_key = 0;
		while( ( shm_id = shmget( ++shm_key, s, IPC_CREAT | IPC_EXCL |
							0666 ) ) == -1 )
		{
		}
		m_shm = (float *) shmat( shm_id, 0, 0 );

		if( inputCount() > 0 )
		{
			m_inputs = new float * [inputCount()];
		}
		if( outputCount() > 0 )
		{
			m_outputs = new float * [outputCount()];
		}

		writeValue<Sint16>( SET_INPUT_COUNT );
		writeValue<Uint8>( inputCount() );

		writeValue<Sint16>( SET_OUTPUT_COUNT );
		writeValue<Uint8>( outputCount() );

		writeValue<Sint16>( SET_SHM_KEY_AND_SIZE );
		writeValue<Uint16>( shm_key );
		writeValue<size_t>( s );

		writeValue<Sint16>( INITIALIZATION_DONE );
	}


private:
	static long hostCallback( AEffect * _effect, long _opcode, long _index,
					long _value, void * _ptr, float _opt );

	FSTHandle * m_handle;
	FST * m_fst;
	AEffect * m_plugin;

	Uint32 m_blockSize;
	float * m_shm;

	float ** m_inputs;
	float ** m_outputs;

	std::vector<VstMidiEvent> m_midiEvents;

} ;



VSTPlugin * plugin = NULL;


#define DEBUG_CALLBACKS
#ifdef DEBUG_CALLBACKS
#define SHOW_CALLBACK debugMsg
#else
#define SHOW_CALLBACK(...)
#endif


long VSTPlugin::hostCallback( AEffect * _effect, long _opcode, long _index,
					long _value, void * _ptr, float _opt )
{
	static VstTimeInfo _timeInfo;
/*	sprintf( debugout, "host-callback, opcode = %d\n", (int) _opcode );
	SHOW_CALLBACK( debugout );*/
	
	switch( _opcode )
	{
		case audioMasterAutomate:
			/*SHOW_CALLBACK( "amc: audioMasterAutomate\n" );*/
			// index, value, returns 0
			_effect->setParameter( _effect, _index, _opt );
			return( 0 );

		case audioMasterVersion:
			SHOW_CALLBACK( "amc: audioMasterVersion\n" );
			// vst version, currently 2 (0 for older)
			return( 2 );

		case audioMasterCurrentId:	
			SHOW_CALLBACK( "amc: audioMasterCurrentId\n" );
			// returns the unique id of a plug that's currently
			// loading
			return( 0 );
		
		case audioMasterIdle:
			SHOW_CALLBACK ("amc: audioMasterIdle\n" );
			// call application idle routine (this will
			// call effEditIdle for all open editors too) 
			_effect->dispatcher( _effect, effEditIdle, 0, 0, NULL,
									0.0f );
			return( 0 );

		case audioMasterPinConnected:		
			SHOW_CALLBACK( "amc: audioMasterPinConnected\n" );
			// inquire if an input or output is beeing connected;
			// index enumerates input or output counting from zero:
			// value is 0 for input and != 0 otherwise. note: the
			// return value is 0 for <true> such that older versions
			// will always return true.
			return( 1 );

		case audioMasterWantMidi:
			SHOW_CALLBACK( "amc: audioMasterWantMidi\n" );
			// <value> is a filter which is currently ignored
			return( 0 );

		case audioMasterGetTime:
			//SHOW_CALLBACK( "amc: audioMasterGetTime\n" );
			// returns const VstTimeInfo* (or 0 if not supported)
			// <value> should contain a mask indicating which
			// fields are required (see valid masks above), as some
			// items may require extensive conversions

			memset( &_timeInfo, 0, sizeof( _timeInfo ) );

			_timeInfo.samplePos = 0;
			// TODO: _timeInfo.sampleRate = mixer::inst()->sampleRate();
			_timeInfo.flags = 0;
			// TODO: _timeInfo.tempo = songEditor::inst()->getBPM();
			_timeInfo.timeSigNumerator = 4;
			_timeInfo.timeSigDenominator = 4;
			_timeInfo.flags |= ( kVstBarsValid | kVstTempoValid );
			_timeInfo.flags |= kVstTransportPlaying;

			return( (long)&_timeInfo );

		case audioMasterProcessEvents:
			SHOW_CALLBACK( "amc: audioMasterProcessEvents\n" );
			// VstEvents* in <ptr>
			return( 0 );

		case audioMasterSetTime:
			SHOW_CALLBACK( "amc: audioMasterSetTime\n" );
			// VstTimenfo* in <ptr>, filter in <value>, not
			// supported

		case audioMasterTempoAt:
			SHOW_CALLBACK( "amc: audioMasterTempoAt\n" );
			// returns tempo (in bpm * 10000) at sample frame
			// location passed in <value>
			return( 0 );

		case audioMasterGetNumAutomatableParameters:
			SHOW_CALLBACK( "amc: audioMasterGetNumAutomatable"
							"Parameters\n" );
			return( 0 );

		case audioMasterGetParameterQuantization:	
			SHOW_CALLBACK( "amc: audioMasterGetParameter\n"
							"Quantization\n" );
			// returns the integer value for +1.0 representation,
			// or 1 if full single float precision is maintained
			// in automation. parameter index in <value> (-1: all,
			// any)
			return( 0 );

		case audioMasterIOChanged:
			plugin->resizeSharedMemory();
			SHOW_CALLBACK( "amc: audioMasterIOChanged\n" );
			// numInputs and/or numOutputs has changed
			return( 0 );

		case audioMasterNeedIdle:
			SHOW_CALLBACK( "amc: audioMasterNeedIdle\n" );
			// plug needs idle calls (outside its editor window)
			return( 0 );

		case audioMasterSizeWindow:
			// TODO using lmms-main-window-size
			SHOW_CALLBACK( "amc: audioMasterSizeWindow\n" );
			// index: width, value: height
			return( 0 );

		case audioMasterGetSampleRate:
			// TODO using mixer-call
			SHOW_CALLBACK( "amc: audioMasterGetSampleRate\n" );
			return( 0 );

		case audioMasterGetBlockSize:
			// TODO using mixer-call
			SHOW_CALLBACK( "amc: audioMasterGetBlockSize\n" );
			return( 0 );

		case audioMasterGetInputLatency:
			// TODO using mixer-call
			SHOW_CALLBACK( "amc: audioMasterGetInputLatency\n" );
			return( 0 );

		case audioMasterGetOutputLatency:
			// TODO using mixer-call
			SHOW_CALLBACK( "amc: audioMasterGetOutputLatency\n" );
			return( 0 );

		case audioMasterGetPreviousPlug:
			SHOW_CALLBACK( "amc: audioMasterGetPreviousPlug\n" );
			// input pin in <value> (-1: first to come), returns
			// cEffect*
			return( 0 );

		case audioMasterGetNextPlug:
			SHOW_CALLBACK( "amc: audioMasterGetNextPlug\n" );
			// output pin in <value> (-1: first to come), returns
			// cEffect*
			return( 0 );

		case audioMasterWillReplaceOrAccumulate:
			SHOW_CALLBACK( "amc: audioMasterWillReplaceOr"
							"Accumulate\n" );
			// returns: 0: not supported, 1: replace, 2: accumulate
			return( 0 );

		case audioMasterGetCurrentProcessLevel:
			SHOW_CALLBACK( "amc: audioMasterGetCurrentProcess"
								"Level\n" );
			// returns: 0: not supported,
			// 1: currently in user thread (gui)
			// 2: currently in audio thread (where process is
			//    called)
			// 3: currently in 'sequencer' thread (midi, timer etc)
			// 4: currently offline processing and thus in user
			//    thread
			// other: not defined, but probably pre-empting user
			//        thread.
			return( 0 );

		case audioMasterGetAutomationState:
			SHOW_CALLBACK( "amc: audioMasterGetAutomationState\n" );
			// returns 0: not supported, 1: off, 2:read, 3:write,
			// 4:read/write offline
			return( 0 );

		case audioMasterOfflineStart:
			SHOW_CALLBACK( "amc: audioMasterOfflineStart\n" );
			return( 0 );

		case audioMasterOfflineRead:
			SHOW_CALLBACK( "amc: audioMasterOfflineRead\n" );
			// ptr points to offline structure, see below.
			// return 0: error, 1 ok
			return( 0 );

		case audioMasterOfflineWrite:
			SHOW_CALLBACK( "amc: audioMasterOfflineWrite\n" );
			// same as read
			return( 0 );

		case audioMasterOfflineGetCurrentPass:
			SHOW_CALLBACK( "amc: audioMasterOfflineGetCurrent"
								"Pass\n" );
			return( 0 );

		case audioMasterOfflineGetCurrentMetaPass:
			SHOW_CALLBACK( "amc: audioMasterOfflineGetCurrentMeta"
								"Pass\n");
			return( 0 );

		case audioMasterSetOutputSampleRate:
			SHOW_CALLBACK( "amc: audioMasterSetOutputSample"
								"Rate\n" );
			// for variable i/o, sample rate in <opt>
			return( 0 );

		case audioMasterGetSpeakerArrangement:
			SHOW_CALLBACK( "amc: audioMasterGetSpeaker"
							"Arrangement\n" );
			// (long)input in <value>, output in <ptr>
			return( 0 );

		case audioMasterGetVendorString:
			SHOW_CALLBACK( "amc: audioMasterGetVendorString\n" );
			// fills <ptr> with a string identifying the vendor
			// (max 64 char)
			strcpy( (char *) _ptr, "LAD");
			return( 0 );

		case audioMasterGetProductString:
			SHOW_CALLBACK( "amc: audioMasterGetProductString\n" );
			// fills <ptr> with a string with product name
			// (max 64 char)
			strcpy( (char *) _ptr, "XFST-Server" );
			return( 0 );

		case audioMasterGetVendorVersion:
			SHOW_CALLBACK( "amc: audioMasterGetVendorVersion\n" );
			// TODO
			// returns vendor-specific version
			return( 1000 );

		case audioMasterVendorSpecific:
			SHOW_CALLBACK( "amc: audioMasterVendorSpecific\n" );
			// no definition, vendor specific handling
			return( 0 );
		
		case audioMasterSetIcon:
			SHOW_CALLBACK( "amc: audioMasterSetIcon\n" );
			// TODO
			// void* in <ptr>, format not defined yet
			return( 0 );

		case audioMasterCanDo:
			SHOW_CALLBACK( "amc: audioMasterCanDo\n" );
			// string in ptr, see below
			return( 0 );
		
		case audioMasterGetLanguage:
			SHOW_CALLBACK( "amc: audioMasterGetLanguage\n" );
			// TODO
			// see enum
			return( 0 );

		case audioMasterOpenWindow:
			SHOW_CALLBACK( "amc: audioMasterOpenWindow\n" );
			// TODO
			// returns platform specific ptr
			return( 0 );
		
		case audioMasterCloseWindow:
			SHOW_CALLBACK( "amc: audioMasterCloseWindow\n" );
			// TODO
			// close window, platform specific handle in <ptr>
			return( 0 );
		
		case audioMasterGetDirectory:
			SHOW_CALLBACK( "amc: audioMasterGetDirectory\n" );
			// TODO
			// get plug directory, FSSpec on MAC, else char*
			return( 0 );
		
		case audioMasterUpdateDisplay:
			SHOW_CALLBACK( "amc: audioMasterUpdateDisplay\n" );
			// something has changed, update 'multi-fx' display
			_effect->dispatcher( _effect, effEditIdle, 0, 0, NULL,
									0.0f );
			return( 0 );

		case audioMasterBeginEdit:
			SHOW_CALLBACK( "amc: audioMasterBeginEdit\n" );
			// begin of automation session (when mouse down),	
			// parameter index in <index>
			return( 0 );

		case audioMasterEndEdit:
			SHOW_CALLBACK( "amc: audioMasterEndEdit\n" );
			// end of automation session (when mouse up),
			// parameter index in <index>
			return( 0 );

		case audioMasterOpenFileSelector:
			SHOW_CALLBACK( "amc: audioMasterOpenFileSelector\n" );
			// open a fileselector window with VstFileSelect*
			// in <ptr>
			return( 0 );

		default:
			sprintf( debugout, "VST master dispatcher: undefed: "
				"%d, %d\n", (int) _opcode, effKeysRequired );
			SHOW_CALLBACK( debugout );
			break;
	}

	return( 0 );
}




int 
main( int argc, char * * argv ) 
{
	wine_shared_premain();

	Sint16 cmd;
	while( ( cmd = readValue<Sint16>() ) != CLOSE_VST_PLUGIN )
	{
		switch( cmd )
		{
			case LOAD_VST_PLUGIN:
				plugin = new VSTPlugin( readString().c_str() );
				break;

			case PROCESS:
				plugin->process();
				break;

			case ENQUEUE_MIDI_EVENT:
			{
				const midiEvent ev = readValue<midiEvent>();
				const Uint32 fr_ahead = readValue<Uint32>();
				plugin->enqueueMidiEvent( ev, fr_ahead );
				break;
			}

			case SET_SAMPLE_RATE:
				plugin->setSampleRate( readValue<Sint32>() );
				break;


			case SET_BUFFER_SIZE:
				plugin->setBlockSize( readValue<Uint32>() );
				break;

			case GET_VST_VERSION:
				writeValue<Sint16>( SET_VST_VERSION );
				writeValue<Sint32>( plugin->vstVersion() );
				break;

			case GET_NAME:
				writeValue<Sint16>( SET_NAME );
				writeString( plugin->vstName() );
				break;

			case GET_VENDOR_STRING:
				writeValue<Sint16>( SET_VENDOR_STRING );
				writeString( plugin->vstVendorString() );
				break;

			case GET_PRODUCT_STRING:
				writeValue<Sint16>( SET_PRODUCT_STRING );
				writeString( plugin->vstProductString() );
				break;

			default:
				debugMsg( "unhandled message!" );
				break;
		}
	}

	delete plugin;

	return( 0 );

}

