# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.

# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""tests for specific behaviour of astng nodes

Copyright (c) 2003-2005 LOGILAB S.A. (Paris, FRANCE).
http://www.logilab.fr/ -- mailto:contact@logilab.fr
"""

__revision__ = "$Id: unittest_astng.py,v 1.24 2005/05/02 20:57:29 syt Exp $"

import unittest
import sys
import parser
from os import getcwd
from os.path import join

from logilab.common.astng import builder, astng, ResolveError, NotFoundError, \
     ASTNGManager
from logilab.common.modutils import load_module_from_name

import data
from data import module as test_module

manager = ASTNGManager()
abuilder = builder.ASTNGBuilder(manager) 
MODULE = abuilder.module_build(test_module)
MODULE2 = abuilder.file_build('data/module2.py', 'data.module2')
NONREGR = abuilder.file_build('data/nonregr.py', 'data.nonregr')

def _test_dict_interface(self, node, test_attr):
    self.assert_(node[test_attr] is node.locals[test_attr])
    node.keys()
    node.values()
    node.items()
    iter(node)


class ModuleNodeTC(unittest.TestCase):

    def test_dict_interface(self):
        _test_dict_interface(self, MODULE, 'YO')
        
    def test_resolve(self):
        yo = MODULE.resolve('YO')
        self.assert_(isinstance(yo, astng.Class))
        self.assertEquals(yo.name, 'YO')        
        red = MODULE.resolve('redirect')
        self.assert_(isinstance(red, astng.Function))
        self.assertEquals(red.name, 'nested_args')
        spawn = MODULE.resolve('spawn')
        self.assert_(isinstance(spawn, astng.Class))
        self.assertEquals(spawn.name, 'Execute')
        # built-in objects
        none = MODULE.resolve('None')
        self.assertEquals(none.value, None)
        obj = MODULE.resolve('object')
        self.assert_(isinstance(obj, astng.Class))
        self.assertEquals(obj.name, 'object')
        self.assertEquals(NONREGR.resolve('enumerate').name, 'enumerate')
        # resolve packageredirection
        sys.path.insert(1, 'astng_data')
        try:
            m = abuilder.file_build('astng_data/appl/myConnection.py', 'appl.myConnection')
            cnx = m.resolve_dotted('SSL1.Connection')
            self.assertEquals(cnx.__class__, astng.Class)
            self.assertEquals(cnx.name, 'Connection')
            self.assertEquals(cnx.root().name, 'Connection1')
        finally:
            del sys.path[1]
        # raise ResolveError
        self.assertRaises(ResolveError, MODULE.resolve, 'YOAA')

        
    def test_resolve_dotted(self):
        exists = MODULE.resolve_dotted('os.path.exists')
        self.assert_(isinstance(exists, astng.Function))
        self.assertEquals(exists.name, 'exists')
        self.assertRaises(ResolveError, NONREGR.locals['toto'].resolve_dotted, 'v.get')
        # resolve sub package
        mod = MODULE2.resolve_dotted('data.module')
                          
    def test_resolve_all(self):
        exists, = MODULE.resolve_all(['os.path.exists', 'v.get'])
        self.assert_(isinstance(exists, astng.Function))
        self.assertEquals(exists.name, 'exists')
                          
    def test_wildard_import_names(self):
        m = abuilder.file_build('astng_data/all.py', 'all')
        self.assertEquals(m.wildcard_import_names(), ['Aaa', '_bla', 'name'])
        m = abuilder.file_build('astng_data/notall.py', 'notall')
        res = m.wildcard_import_names()
        res.sort()
        self.assertEquals(res, ['Aaa', 'func', 'name', 'other'])
        
    def test_as_string(self):
        """just check as_string on a whole module doesn't raise an exception
        """
        self.assert_(MODULE.as_string())
        self.assert_(MODULE2.as_string())


class ImportNodeTC(unittest.TestCase):
    
    def test_get_module_object(self):
        from_ = MODULE.locals['modutils']
        self.assertEquals(from_.get_module_object('spawn'), ('logilab.common', 'Execute'))

    def test_get_real_name(self):
        from_ = MODULE.locals['modutils']
        self.assertEquals(from_.get_real_name('spawn'), 'Execute')
        imp_ = MODULE.locals['os']
        self.assertEquals(imp_.get_real_name('os'), 'os')
        self.assertRaises(NotFoundError, imp_.get_real_name, 'os.path')
        imp_ = MODULE.locals['spawn']
        self.assertEquals(imp_.get_real_name('spawn'), 'Execute')
        self.assertRaises(NotFoundError, imp_.get_real_name, 'Execute')
        imp_ = MODULE2.locals['YO']
        self.assertEquals(imp_.get_real_name('YO'), 'YO')
        self.assertRaises(NotFoundError, imp_.get_real_name, 'data')
        
        
class FunctionNodeTC(unittest.TestCase):

    def test_dict_interface(self):
        _test_dict_interface(self, MODULE.locals['global_access'], 'local')
        
    def test_resolve(self):
        method = MODULE.locals['YOUPI'].locals['method']
        my_dict = method.resolve('MY_DICT')
        self.assert_(isinstance(my_dict, astng.Dict))
        none = method.resolve('None')
        self.assertEquals(none.value, None)
        self.assertRaises(ResolveError, method.resolve, 'YOAA')
        
    def test_resolve_argument_with_default(self):
        make_class = MODULE2.locals['make_class']
        base = make_class.resolve('base')
        self.assert_(isinstance(base, astng.Class), base.__class__)
        self.assertEquals(base.name, 'YO')
        self.assertEquals(base.root().name, 'data.module')

    def test_default_value(self):
        func = MODULE2.locals['make_class']
        self.assert_(isinstance(func.default_value('base'), astng.Getattr))
        self.assert_(isinstance(func.default_value('args'), astng.Tuple))
        self.assert_(isinstance(func.default_value('kwargs'), astng.Dict))
        self.assertRaises(astng.NoDefault, func.default_value, 'any')

    def test_navigation(self):
        function = MODULE.locals['global_access']
        self.assertEquals(function.get_statement(), function)
        l_sibling = function.previous_sibling()
        self.assert_(isinstance(l_sibling, astng.Assign))
        self.assert_(l_sibling is function.getChildNodes()[0].previous_sibling())
        r_sibling = function.next_sibling()
        self.assert_(isinstance(r_sibling, astng.Class))
        self.assertEquals(r_sibling.name, 'YO')
        self.assert_(r_sibling is function.getChildNodes()[0].next_sibling())
        last = r_sibling.next_sibling().next_sibling().next_sibling()
        self.assert_(isinstance(last, astng.Assign))
        self.assertEquals(last.next_sibling(), None)
        first = l_sibling.previous_sibling().previous_sibling().previous_sibling().previous_sibling()
        self.assertEquals(first.previous_sibling(), None)

    def test_nested_args(self):
        func = MODULE.locals['nested_args']
        self.assertEquals(func.argnames, ['a', ('b', 'c', 'd')])
        local = func.locals.keys()
        local.sort()
        self.assertEquals(local, ['a', 'b', 'c', 'd'])
        self.assertEquals(func.is_method(), 0)
        self.assertEquals(func.is_static_method(), 0)
        self.assertEquals(func.is_class_method(), 0)
       
    def test_format_args(self):
        func = MODULE2.locals['make_class']
        self.assertEquals(func.format_args(), 'any, base=data.module.YO, *args, **kwargs')
        func = MODULE.locals['nested_args']
        self.assertEquals(func.format_args(), 'a, (b,c,d)')

        
class ClassNodeTC(unittest.TestCase):

    def test_dict_interface(self):
        _test_dict_interface(self, MODULE.locals['YOUPI'], 'method')
        
    def test_resolve(self):
        klass = MODULE.locals['YOUPI']
        my_dict = klass.resolve('MY_DICT')
        self.assert_(isinstance(my_dict, astng.Dict))
        none = klass.resolve('None')
        self.assertEquals(none.value, None)
        obj = klass.resolve('object')
        self.assert_(isinstance(obj, astng.Class))
        self.assertEquals(obj.name, 'object')
        self.assertRaises(ResolveError, klass.resolve, 'YOAA')

    def test_navigation(self):
        klass = MODULE.locals['YO']
        self.assertEquals(klass.get_statement(), klass)
        l_sibling = klass.previous_sibling()
        self.assert_(isinstance(l_sibling, astng.Function), l_sibling)
        self.assertEquals(l_sibling.name, 'global_access')
        r_sibling = klass.next_sibling()
        self.assert_(isinstance(r_sibling, astng.Class))
        self.assertEquals(r_sibling.name, 'YOUPI')
        
    def test_get_ancestor_for_method(self):
        klass1 = MODULE.locals['YO']
        klass2 = MODULE.locals['YOUPI']
        anc_klass = klass2.get_ancestor_for_method('__init__')
        self.assert_(isinstance(anc_klass, astng.Class))
        self.assertEquals(anc_klass.name, 'YO')
        self.assertRaises(NotFoundError, klass2.get_ancestor_for_method, 'method')

    def test_get_ancestor_for_attribute(self):
        klass1 = MODULE.locals['YO']
        klass2 = MODULE.locals['YOUPI']
        anc_klass = klass2.get_ancestor_for_attribute('yo')
        self.assert_(isinstance(anc_klass, astng.Class))
        self.assertEquals(anc_klass.name, 'YO')
        self.assertRaises(NotFoundError, klass2.get_ancestor_for_attribute, 'member')
        
    def test_methods(self):
        klass2 = MODULE.locals['YOUPI']
        methods = [m.name for m in klass2.methods()]
        methods.sort()
        self.assertEquals(methods, ['__init__', 'class_method',
                                   'method', 'static_method'])
        methods = [m.name for m in klass2.methods(inherited=False)]
        methods.sort()
        self.assertEquals(methods, ['__init__', 'class_method',
                                   'method', 'static_method'])
        klass2 = MODULE2.locals['Specialization']
        methods = [m.name for m in klass2.methods(False)]
        methods.sort()
        self.assertEquals(methods, [])
        self.assertEquals(klass2.get_method('method').name, 'method')
        self.assertRaises(NotFoundError, klass2.get_method, 'nonexistant')
        methods = [m.name for m in klass2.methods(inherited=True)]
        methods.sort()
        self.assertEquals(methods, ['__init__', 'class_method',
                                   'method', 'static_method'])
        
    def test_rhs(self):
        my_dict = MODULE.locals['MY_DICT']
        self.assert_(isinstance(my_dict.rhs(), astng.Dict))
        a = MODULE.locals['YO'].locals['a']
        value = a.rhs()
        self.assert_(isinstance(value, astng.Const))
        self.assertEquals(value.value, 1)
        
    def test_ancestors(self):
        klass = MODULE.locals['YOUPI']
        ancs = [a.name for a in klass.ancestors()]
        self.assertEquals(ancs, ['YO'])
        klass = MODULE2.locals['Specialization']
        ancs = [a.name for a in klass.ancestors()]
        self.assertEquals(ancs, ['YOUPI', 'YO', 'YO'])

    
__all__ = ('ModuleNodeTC', 'ImportNodeTC', 'FunctionNodeTC', 'ClassNodeTC')
        
if __name__ == '__main__':
    unittest.main()
