/*
    Copyright (C) 2008 Andrew Caudwell (acaudwell@gmail.com)

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version
    3 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "logstalgia.h"

//Logstalgia

//turn performance profiling
//#DEFINE LS_PERFORMANCE_PROFILE

float gSplash = -1.0f;

std::string profile_name;
Uint32 profile_start_msec;

void profile_start(std::string profile) {
#ifdef LS_PERFORMANCE_PROFILE
    profile_start_msec = SDL_GetTicks();
    profile_name = profile;
#endif
}

void profile_stop() {
#ifdef LS_PERFORMANCE_PROFILE
    debugLog("%s took %d ms\n", profile_name.c_str(), SDL_GetTicks() - profile_start_msec);
#endif
}

void logstalgia_help() {
    logstalgia_help("");
}

void logstalgia_help(char* error) {

    if(strlen(error)) {
        printf("error: %s\n", error);
    }

    printf("Usage: logstalgia [options] file\n");
    printf("options:\n");
    printf("-b                              no bouncing\n");
    printf("-p                              hide response code\n");
    printf("-r                              hide paddle\n");
    printf("-WIDTHxHEIGHT                   set window size\n");
    printf("-f                              fullscreen\n");
    printf("-s                              simulation speed (default: 1)\n");
    printf("-u                              page summary update rate (default: 5)\n");
    printf("-x                              show full request ip/hostname\n");
    printf("-g name,regex,percent[,colour]  group urls that match a regular expression together\n");
    printf("\nIf file is '-' will attempt to read log entries from stdin\n\n");
    exit(1);
}

Logstalgia::Logstalgia(std::string logfile, float simu_speed, float update_rate) : SDLApp() {
    paddle = 0;

    maxrows = 500;

    info       = false;
    paused     = false;
    recentre   = false;
    next       = false;

    this->simu_speed  = simu_speed;
    this->update_rate = update_rate;

    spawn_delay=0;

    gHighscore = 0;

    uimessage_timer=0.0f;


    fonts =0;
    ipSummarizer=0;
    mousepos = vec2f(0.0f, 0.0f);
    paddle_target=0;

    loghandle=0;
    this->logfile = logfile;

    if(logfile.size()) {
        if(logfile.compare("-")==0) {
            logfile = "STDIN";
            loghandle = stdin;
        } else {
            loghandle = fopen(logfile.c_str(), "r");
        }

        if(!loghandle) {
            printf("error: couldn't open %s\n", logfile.c_str());
            exit(1);
        }
    } else {
        logstalgia_help("no file supplied");
    }
    debugLog("Logstalgia end of constructor\n");

    total_space = display.height - 40;
    remaining_space = total_space;

    total_entries=0;


    std::string datapath;

#ifdef _WIN32
    char szAppPath[MAX_PATH] = "";

    GetModuleFileName(0, szAppPath, MAX_PATH);

    // Extract directory
    datapath = szAppPath;
    datapath = datapath.substr(0, datapath.rfind("\\")+1) + "data\\";
#else

#ifdef LS_DATADIR
    datapath = std::string(LS_DATADIR) + "/";
#else
    datapath = std::string("data/");
#endif

#endif

    fonts = new FontPack(datapath + "font.tga", datapath + "font.finfo");

    balltex = (TextureResource*) texturemanager.grab(datapath + "ball.tga");


    infowindow = TextArea(fonts->getInstance(16,true));

    mousehide_timeout = 0.0f;
}

Logstalgia::~Logstalgia() {
    if(paddle!=0) delete paddle;
    if(fonts!=0) delete fonts;

    for(size_t i=0;i<summGroups.size();i++) {
        delete summGroups[i];
        summGroups[i]=0;
    }

    if(loghandle!=0) fclose(loghandle);
}

void Logstalgia::togglePause() {
    paused = !paused;

    if(!paused) {
        ipSummarizer->mouseOut();

        int nogrps = summGroups.size();
        for(int i=0;i<nogrps;i++) {
            summGroups[i]->mouseOut();
        }
    }
}

void Logstalgia::keyPress(SDL_KeyboardEvent *e) {
	if (e->type == SDL_KEYDOWN) {

		if (e->keysym.sym == SDLK_ESCAPE) {
		    appFinished=true;
        }

        if(e->keysym.sym == SDLK_q) {
            info = !info;
        }

        if(e->keysym.sym == SDLK_c) {
            gSplash = 15.0f;
        }

        if(e->keysym.sym == SDLK_n) {
            next = true;
        }

        if(e->keysym.sym == SDLK_SPACE) {
            togglePause();
        }

        if(e->keysym.sym == SDLK_EQUALS || e->keysym.sym == SDLK_KP_PLUS) {
            if(simu_speed<=29.0f) {
                simu_speed += 1.0f;
                recentre=true;
                setMessage("speed +1");
            }
        }

        if(e->keysym.sym == SDLK_MINUS || e->keysym.sym == SDLK_KP_MINUS) {
            if(simu_speed>=2.0f) {
                simu_speed -= 1.0f;
                recentre=true;
                setMessage("speed -1");
            }
        }

	}
}

void Logstalgia::setMessage(std::string message) {
    uimessage       = message;
    uimessage_timer = 3.0f;
}

void Logstalgia::mouseClick(SDL_MouseButtonEvent *e) {
	debugLog("click! (x=%d,y=%d)\n", e->x, e->y);

	//press
	if(e->type == SDL_MOUSEBUTTONDOWN) {
	}

	//release
	if(e->type == SDL_MOUSEBUTTONUP) {
	}

}

void Logstalgia::mouseMove(SDL_MouseMotionEvent *e) {
    mousepos = vec2f(e->x, e->y);
    SDL_ShowCursor(true);
    mousehide_timeout = 5.0f;
}

void Logstalgia::addBall(LogEntry& le, float head_start) {
    debugLog("adding ball for log entry\n");

    gHighscore++;

    std::string hostname = le.getHostname();
    std::string pageurl  = le.requestURL();

    //find appropriate summarizer for url
    int nogroups = summGroups.size();
    Summarizer* pageSummarizer= 0;
    for(int i=0;i<nogroups;i++) {
        if(summGroups[i]->supportedString(pageurl)) {
            pageSummarizer = summGroups[i];
            break;
        }
    }

    if(pageSummarizer==0) return;

    float dest_y = pageSummarizer->addString(pageurl);
    float pos_y  = ipSummarizer->addString(hostname);

    vec2f pos  = vec2f(1, pos_y);
    vec2f dest = vec2f(paddle->getX(), dest_y);

    std::string match = ipSummarizer->getBestMatchStr(hostname);


    vec3f colour = pageSummarizer->isColoured() ? pageSummarizer->getColour() : colourHash(match);

    RequestBall* ball = new RequestBall(le, fonts->getInstance(16, false), balltex, colour, pos, dest, simu_speed);
    ball->setElapsed(head_start);

    balls.push_back(ball);
    spawn_delay=spawn_speed;
}

void Logstalgia::readLog() {
    debugLog("readLog()\n");

    //am i reading from stdin?
    FILE* source = loghandle;

    if(!source) {
        debugLog("no log to read from\n");
        exit(1);
    }

    char line[1024];
    int lineno=0;

	while( fgets(line, sizeof(line), source) != NULL ) {
	    std::string linestr = std::string(line);
            LogEntry le(linestr);
            if(le.parsedOK()) {
                entries.push_back(le);
                total_entries++;
            } else {
                debugLog("error: could not read line %s\n", linestr.c_str());
            }
        lineno++;

        if(lineno>=maxrows) break;
    }

    if(entries.size()==0 && loghandle != stdin) {

        if(total_entries==0) {
            logstalgia_help("could not parse first entry");
        }

        printf("no more entires\n");
        exit(0);
    }

    debugLog("end of readLog()\n");
}

void Logstalgia::init() {
    debugLog("init called\n");

    vec3f colour = vec3f(0.5f, 0.5f, 0.5f);
    vec2f pos = vec2f(display.width-(display.width/3), rand() % display.height);

    paddle = new Paddle(pos, colour);

    readLog();

    elapsed_time=0;
    starttime = 0;
    lasttime =0;
    if(entries.size()) starttime = entries[0].getTimestamp();

    ipSummarizer = new Summarizer(fonts->getInstance(10, false), 0, 40, 0, 2.0f);

    //add default groups
    if(summGroups.size()==0) {
        //images - file is under images or
        addGroup("CSS", "\\.css\\b", 15);
        addGroup("Script", "\\.js\\b", 15);
        addGroup("Images", "/images/|\\.(jpe?g|gif|bmp|tga|ico|png)\\b", 20);
    }

    //always fill remaining space with Misc, (if there is some)
    if(remaining_space>50) {
        addGroup(summGroups.size()>0 ? "Misc" : "", ".*");
    }

    SDL_ShowCursor(false);
}

void Logstalgia::update(float t, float dt) {
    this->logic(t, dt);
    this->draw(t, dt);
}

RequestBall* Logstalgia::findNearest() {
    std::list<RequestBall*>::iterator it = balls.begin();

    float min_dist = -1.0f;
    RequestBall* nearest = 0;
    for(std::list<RequestBall*>::iterator it = balls.begin(); it != balls.end(); it++) {
        RequestBall* ball = *it;

            //special case if failed response code
            if(!ball->le.successful()) {
                continue;
            }

        if(ball->le.successful() && !ball->bounced()) {
            float dist = (paddle->getX() - ball->getX())/ball->speed;
            if(min_dist<0.0f || dist<min_dist) {
                min_dist = dist;
                nearest = ball;
            }
        }
    }

    return nearest;
}

void Logstalgia::logic(float t, float dt) {

    float sdt = dt*simu_speed;;


    if(mousehide_timeout>0.0f) {
        mousehide_timeout -= dt;
        if(mousehide_timeout<0.0f) {
            SDL_ShowCursor(false);
        }
    }

    infowindow.hide();

    //if paused, dont move anything, only check what is under mouse
    if(paused) {

        for(std::list<RequestBall*>::iterator it = balls.begin(); it != balls.end(); it++) {
            RequestBall* ball = *it;
            if(ball->mouseOver(infowindow, mousepos)) {
                break;
            }
        }

        if(paused) {

            if(!ipSummarizer->mouseOver(infowindow,mousepos)) {
                int nogrps = summGroups.size();
                for(int i=0;i<nogrps;i++) {
                    if(summGroups[i]->mouseOver(infowindow, mousepos)) break;
                }
            }
        }

        return;
    }

    //increment clock
    elapsed_time += sdt;
    currtime = starttime + (long)(elapsed_time);

    //next will fast forward clock to the time of the next entry, if the next entry is in the future
    if(next) {
        if(entries.size() > 0) {
            LogEntry le = entries[0];

            long entrytime = le.getTimestamp();
            if(entrytime > currtime) {
                elapsed_time = entrytime - starttime;
                currtime = starttime + (long)(elapsed_time);
            }
        }
        next = false;
    }

    //recalc spawn speed each second by
    if(currtime != lasttime) {
        profile_start("calc spawn speed");
        int items_to_spawn=-1;
        for(std::deque<LogEntry>::iterator it = entries.begin(); it != entries.end(); it++) {
            if((*it).getTimestamp() <= currtime) {
                items_to_spawn++;
                continue;
            }
            break;
        }

        spawn_speed = (items_to_spawn <= 0) ? 0.1f/simu_speed : (1.0f / items_to_spawn) / simu_speed;
        spawn_delay = 0.0f;
        debugLog("spawn_speed = %.2f\n", spawn_speed);
        profile_stop();

        //display date
        char datestr[256];
        char timestr[256];
        struct tm* timeinfo = localtime ( &currtime );
        strftime(datestr, 256, "%A, %d %B, %Y", timeinfo);
        strftime(timestr, 256, "%X", timeinfo);

        displaydate =datestr;
        displaytime =timestr;
    }

    lasttime=currtime;

    if((recentre || !paddle->moving()) && balls.size()>0) {
        profile_start("findNearest");

        recentre=false;

        RequestBall* ball = findNearest();

        if(ball!=0 && !(paddle->moving() && paddle_target == ball)) {
            paddle_target = ball;
            vec2f dest = ball->finish();

            paddle->moveTo((int)dest.y, ball->arrivalTime(), ball->colour);
        }

        profile_stop();
    }

    //if still not moving, recentre
    if(!paddle->moving()) {
        recentre=true;
        paddle->moveTo(display.height/2, 4, vec3f(0.5f, 0.5f, 0.5f));
    }

    //see if entries show appear
    if(entries.size() > 0 && spawn_delay<=0) {
        profile_start("add entries");

        LogEntry le = entries[0];
        if(le.getTimestamp() < currtime) {
            addBall(le, -spawn_delay);

            entries.pop_front();
        }

        profile_stop();
    }

    //read log if we run out
    if(entries.size()==0) {
        profile_start("readLog");

        readLog();

        profile_stop();

        //set start time if currently 0
        if(starttime==0 && entries.size()) starttime = entries[0].getTimestamp();
    }

    spawn_delay -= sdt;

    paddle->logic(sdt);

    profile_start("check ball status");

    for(std::list<RequestBall*>::iterator it = balls.begin(); it != balls.end(); it++) {

        RequestBall* ball = *it;

        ball->logic(dt);

        if(ball->finished()) {
            std::string url  = ball->le.requestURL();
            std::string host = ball->le.getHostname();

            int nogroups = summGroups.size();

            for(int i=0;i<nogroups;i++) {
                if(summGroups[i]->supportedString(url)) {
                    summGroups[i]->removeString(url);
                    break;
                }
            }

            ipSummarizer->removeString(host);

            it = balls.erase(it);
            delete ball;
        }
    }

    profile_stop();

    profile_start("ipSummarizer logic");
    ipSummarizer->logic(dt);
    profile_stop();

    profile_start("updateGroups logic");
    updateGroups(dt);
    profile_stop();
}

void Logstalgia::addGroup(std::string groupstr) {

    std::vector<std::string> groupdef;
    Regex groupregex("^([^,]+),([^,]+),([^,]+)(?:,([^,]+))?$");
    groupregex.match(groupstr, &groupdef);

    vec3f colour(0.0f, 0.0f, 0.0f);

    if(groupdef.size()>=3) {
        std::string groupname = groupdef[0];
        std::string groupregex = groupdef[1];
        int percent = atoi(groupdef[2].c_str());

        //check for optional colour param
        if(groupdef.size()>=4) {
            int col;
            int r, g, b;
            if(sscanf(groupdef[3].c_str(), "%02x%02x%02x", &r, &g, &b) == 3) {
                colour = vec3f( r, g, b );
                debugLog("r = %d, g = %d, b = %d\n", r, g, b);
                colour /= 255.0f;
            }
        }

        addGroup(groupname, groupregex, percent, colour);
    }
}

void Logstalgia::addGroup(std::string grouptitle, std::string groupregex, int percent, vec3f colour) {

    if(percent<0) return;

    int remainpc = (int) ( ((float) remaining_space/total_space) * 100);

    if(percent==0) {
        percent=remainpc;
    }

    if(remainpc<percent) return;

    int space = (int) ( ((float)percent/100) * total_space );

    float pos_x = display.width-(display.width/3) + 20;

    int top_gap    = total_space - remaining_space;
    int bottom_gap = display.height - (total_space - remaining_space + space);

    debugLog("add group name = %s, regex = %s, remainpc = %d, space = %d, pos_x = %.2f, top_gap = %d, bottom_gap = %d\n",
        grouptitle.c_str(), groupregex.c_str(), remainpc, space, pos_x, top_gap, bottom_gap);

    Summarizer* summ = new Summarizer(fonts->getInstance(10, false), pos_x, top_gap, bottom_gap, update_rate, groupregex, grouptitle);
//    summ->showCount(true);

    if(colour.length2() > 0.01f) {
        summ->setColour(colour);
    }

    summGroups.push_back(summ);

    remaining_space -= space;
}

void Logstalgia::updateGroups(float dt) {

    int nogrps = summGroups.size();
    for(int i=0;i<nogrps;i++) {
        summGroups[i]->logic(dt);
    }

}

void Logstalgia::drawGroups(float dt) {

    int nogrps = summGroups.size();
    for(int i=0;i<nogrps;i++) {
        summGroups[i]->draw(dt);
    }

}

void Logstalgia::draw(float t, float dt) {

    display.clear();
    glDisable(GL_FOG);

    display.mode2D();
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
    glDisable(GL_LIGHTING);

    profile_start("draw ip summarizer");

    ipSummarizer->draw(dt);

    profile_stop();


    profile_start("draw groups");

    drawGroups(dt);

    profile_stop();


    profile_start("draw balls");

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    for(std::list<RequestBall*>::iterator it = balls.begin(); it != balls.end(); it++) {
        (*it)->draw(dt);
    }

    profile_stop();


    profile_start("draw paddle");

    paddle->draw(dt);

    profile_stop();

    infowindow.draw();

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_TEXTURE_2D);

    FPInstance font32 = fonts->getInstance(32, true);
    FPInstance font16 = fonts->getInstance(16, true);
    FPInstance font10 = fonts->getInstance(10, true);

    if(uimessage_timer>0.1f) {
        glColor4f(1,1,uimessage_timer/3.0f,uimessage_timer/3.0f);

        int mwidth = font32.width(uimessage.c_str());
        font32.draw(display.width/2 - mwidth/2, display.height/2 - 20, uimessage.c_str());
        uimessage_timer-=dt;
    }

    if(gSplash>0.0f) {
        int logowidth = font32.width("Logstalgia");
        int logoheight = 100;
        int cwidth    = font16.width("Web Access Log Viewer");
        int awidth    = font10.width("(C) 2008 Andrew Caudwell");

        vec2f corner(display.width/2 - logowidth/2 - 30.0f, display.height/2 - 40);

        glDisable(GL_TEXTURE_2D);
        glColor4f(0.0f, 0.5f, 1.0f, gSplash * 0.015f);
        glBegin(GL_QUADS);
            glVertex2f(0.0f,                 corner.y);
            glVertex2f(0.0f,                 corner.y + logoheight);
            glVertex2f(display.width, corner.y + logoheight);
            glVertex2f(display.width, corner.y);
        glEnd();

        glEnable(GL_TEXTURE_2D);

        glColor4f(1,0,1,1);
        font32.draw(display.width/2 - logowidth/2,display.height/2 - 30, "^5Log^7stalgia");
        font16.draw(display.width/2 - cwidth/2,display.height/2 + 10, "Web Access Log Viewer");
        font10.draw(display.width/2 - awidth/2,display.height/2 + 30, "(C) 2008 Andrew Caudwell");

        gSplash-=dt;
    }

    glColor4f(1,1,1,1);

    if(info) {
        font16.print(5,00,"FPS = %.d\n", (int) fps);
        font16.print(5,20,"Log Entry Queue = %d", entries.size());
        font16.print(5,40,"Balls = %ld", balls.size());
        font16.print(5,60,"Currtime = %s", displaytime.c_str());
    } else {
        font16.draw(5,0,  displaydate.c_str());
        font16.draw(5,20, displaytime.c_str());
    }
    glColor4f(1,1,1,1);
    font32.print(display.width-165,display.height-40, "%08d", gHighscore);
}
