#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/DataBaseAccess/PostgreSQLDataBase.pm,v 1.37 2002/02/05 15:53:13 lsimonneau Exp $
##*****************************************************************************
##  PostgreSQLDataBase
##  Description  : PostgreSQLDataBase is an implementation of Database for PostgreSQL
##                 
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: PostgreSQLDataBase.pm,v $
#Revision 1.37  2002/02/05 15:53:13  lsimonneau
#Minor bugfixes
#
#Revision 1.36  2002/02/01 17:35:16  lsimonneau
#Modify database structure. *_values tables inherite of data.
#
#Revision 1.35  2002/01/10 15:49:17  fdubuy
#Added GetLastConnectionDate
#
#Revision 1.34  2002/01/08 16:18:27  lsimonneau
#Minor bugfixes.
#
#Revision 1.33  2001/12/05 16:32:36  slhullier
#Error management with try/otherwise ...
#
#Revision 1.32  2001/10/24 15:16:43  lsimonneau
#Modiify AcquitAlarms to work with bugged database (more than one not acquitted alarms of the same type).
#
#Revision 1.31  2001/10/23 16:32:21  lsimonneau
#Change GetAlarmInTimeInterval to match initial prototype.
#
#Revision 1.30  2001/10/05 13:51:45  lsimonneau
#add AcquitAlarm Fonction.
#
#Revision 1.29  2001/10/02 08:40:26  lsimonneau
#*** empty log message ***
#
#Revision 1.28  2001/09/20 08:10:49  lsimonneau
#Minor bugfixes.
#
#Revision 1.27  2001/09/18 08:09:17  slhullier
#Using GetDataInRelativeTimeInterval
#
#Revision 1.26  2001/09/17 13:56:53  lsimonneau
#Major feature enhancements : Add a methods called GetDataInRelativeTimeInterval.
#
#Revision 1.25  2001/09/17 08:40:57  lsimonneau
#Minor bugfixes.
#
#Revision 1.24  2001/09/17 07:31:07  lsimonneau
#Minor bug fixes.
#
#Revision 1.23  2001/08/27 14:36:06  lsimonneau
#Major features improvement. Rewriting of db request and GetData* Fonctions.
#
#Revision 1.22  2001/08/24 13:45:37  lsimonneau
#Minor bugfixes.
#
#Revision 1.21  2001/08/24 13:43:43  lsimonneau
#Minor bugfixes, add docs.
#
#Revision 1.20  2001/08/23 12:57:24  lsimonneau
#Minor bugfixes.
#
#Revision 1.19  2001/08/21 07:56:50  lsimonneau
#Minor bugfixes.
#
#Revision 1.18  2001/08/03 13:16:58  lsimonneau
#Minor bugfixes.
#
#Revision 1.17  2001/08/02 12:38:48  lsimonneau
#Minor bugfixes.
#
#Revision 1.16  2001/08/02 10:04:48  lsimonneau
#Amlioration du traitement d'ancestor.
#
#Revision 1.15  2001/07/31 15:29:13  lsimonneau
#Minor Bug fixe.
#
#Revision 1.14  2001/07/31 09:54:51  lsimonneau
#Amlioration des performances.
#
#Revision 1.13  2001/07/30 11:43:42  lsimonneau
#Ajout de l'option disable_ancestor et modification de la doc.
#
#Revision 1.12  2001/07/27 15:07:41  lsimonneau
#Minor bugfixes.
#
#Revision 1.11  2001/07/27 11:59:37  lsimonneau
#Modification pour assurer la continuit des donnes entre deux versions d' un agent.
#
#Revision 1.10  2001/06/28 11:12:57  slhullier
#
#Die->die
#
#Revision 1.9  2001/06/26 11:58:16  lsimonneau
#Minor bugfixes.
#
#Revision 1.8  2001/06/21 15:38:36  lsimonneau
#die -> Die ;)
#
#Revision 1.7  2001/06/21 15:22:55  lsimonneau
#Changement de Die en die.
#
#Revision 1.6  2001/06/20 09:25:26  lsimonneau
#Correction du bug de la fonction GetDataInTimeInterval sur la limite.
#
#Toutes les fonction trie en fonctionde la date par ordre croissant.
#
#Revision 1.5  2001/06/11 14:29:53  lsimonneau
#Ajout de "ORDER BY number" dans la requete de GetListOfAgentsOnSource
#
#Revision 1.4  2001/06/07 13:06:13  lsimonneau
#Changement du nom des package (DataBase -> LogTrend::DataBaseAccess::DataBase)
#
#Revision 1.3  2001/06/07 12:52:16  lsimonneau
#Ajout de l'utilisation de LogDie.
#
#Revision 1.2  2001/05/28 13:09:08  lsimonneau
#*** empty log message ***
#
#Revision 1.1  2001/05/28 07:30:24  slhullier
#
#Fichiers DataBase
#

package LogTrend::DataBaseAccess::PostgreSQLDataBase;
use LogTrend::DataBaseAccess::DataBase;
use DBI;

@LogTrend::DataBaseAccess::PostgreSQLDataBase::ISA = ("LogTrend::DataBaseAccess::DataBase");

##*****************************************************************************
## Constructor  public
##  Description  : creat a new PostgreSQLDataBase
##
##  Parameters   : The database name,
##                 The database server host name,
##                 The database server port,
##                 An username and a password of a database user
##*****************************************************************************
sub new
{
    my ($classname, $databasename, $host, $port, $username, $password) = @_;
  
    my $self = $classname->SUPER::new($databasename, $host, $port,
                                      $username, $password);

    bless($self, $classname);
    return $self;
}


##*****************************************************************************
## connect protected
##  Description  : Try to establish a connection with the database
##  Parameters   : none
##*****************************************************************************
sub connect
{
    my ($self, $databasename, $host, $port, $username, $password) = @_;

    die("Already connected") if defined $self->{DBHANDLE};

    my $ConnectionParameters = "dbi:Pg:dbname=$databasename";
    
    # Build the connection parametres
    # dbi:Pg:dbname=logtrend;host=serveur.orsay.atrid.fr;port=5432
    $ConnectionParameters = $ConnectionParameters.";host=".$host;
    
    $ConnectionParameters = $ConnectionParameters.";port=".$port;
    
    # Connect to the DB
    my $db_handle = DBI->connect($ConnectionParameters, $username, $password)
	or die("$!");
    
    my $result = 0;
    if ($db_handle)
    {
        $self->{DBHANDLE} = $db_handle;
        $result = 1;
    }

    return $result;    
}


##*****************************************************************************
## disconnect protected
##  Description  : Disconnect from the database
##  Parameters   : none
##*****************************************************************************
sub disconnect
{
    my $self = shift;
    
    if(defined $self->{DBHANDLE}) {
	$self->{DBHANDLE}->disconnect() or die("$!");
    }
}


##*****************************************************************************
##  GetData public
##  Description  : Return the last stored data
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name.
##*****************************************************************************

sub GetData {
    my ($self, $source, $agent, $name, $enable_ancestor) = @_;
    
    $enable_ancestor = 'enable_ancestor' unless defined $enable_ancestor;

    ##
    ## Retrieve informations on agents and data
    ##
    my ($id_agent, $id_data_description, $data_type, $agent_activation_date) = 
        $self->getDataAndAgentInfo($source, $agent, $name);
    
    return 0 unless ($data_type and $id_data_description and $id_agent);
  
    my $data_type_table_name;
    my $st_handle;	
    my @row_data = ();

    ##
    ## If there is no id_data for the main data description, try with ancestor
    ##
    while($#row_data == -1) {
	$data_type_table_name = "${data_type}_values";

        ##
	## Prepare id_data request
	##
	$st_handle = $self->{DBHANDLE}->prepare("
SELECT value, date_part('epoch', collection_date) 

FROM   $data_type_table_name

WHERE  id_data_description = ?

ORDER BY collection_date DESC

LIMIT 1
")
	    or die("Can't retrieve data for variable $name on agent $agent on source $source (prepare)");

	$st_handle->execute($id_data_description)
	    or die("Can't retrieve data for variable $name on agent $agent on source $source (execute) : $!");

	@row_data = $st_handle->fetchrow_array();
	
	$st_handle->finish();

	last unless $enable_ancestor eq 'enable_ancestor';
	
	($id_data_description, $id_agent, $data_type, $agent_activation_date) = $self->getDataAncestorInfo($id_data_description);
	
	last unless ($id_data_description and $id_agent and $data_type and $agent_activation_date);
    }

    return 0 if $#row_data == -1;
    return [@row_data];
}



##*****************************************************************************
##  GetDataType public
##  Description  : Return the type of the data
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name,
##*****************************************************************************

sub GetDataType
{
    my ($self, $source, $agent, $name) = @_;
    my $st_handle;
    my $result = 0;

    my $id_data_description = $self->getIdDataDescription($source, $agent, $name);

    # Retrieve the type name in lowercase (integer, real, ...)
    $st_handle = $self->{DBHANDLE}->prepare("
SELECT lower(data_types.type) 

FROM   data_types, agents_data_descriptions

WHERE  (agents_data_descriptions.id_data_description = ?) AND 
       (data_types.id_data_type = agents_data_descriptions.type)")
	or die("Can't retrieve the data type (prepare)");
   
    $st_handle->execute($id_data_description)
	or die("Can't retrieve the data type (execute)");
    
    my @row = $st_handle->fetchrow_array();        
    
    $st_handle->finish();

    if($#row == -1) {
	return 0;
    }
    else {
	return $row[0];
    }    
}

##*****************************************************************************
##  GetDataInTimeInterval public
##  Description  : Return all data in the specifed time interval
##
##  Parameters   : The maximum number of data to retrieve,
##                 The source name,
##                 The agent name,
##                 The data name,
##                 The start date,
##                 The end date (optional),
##                 Enable ancestor. String with 'enable_ancestor' or 'disable_ancestor'. 
##                                  Default value is 'enable_ancestor'.
##*****************************************************************************

sub GetDataInTimeInterval {
    my ($self, $max_data, $source, $agent, $name, $start, $end, $enable_ancestor) = @_;

    my @result;

    if(!(defined $end) and !(defined $enable_ancestor)) {
	$enable_ancestor = 'enable_ancestor';
    }
    elsif(defined $end and !(defined $enable_ancestor)) {		
	if($end eq 'disable_ancestor' or $end eq 'enable_ancestor') { 
	    $enable_ancestor = $end;
	    $end = undef;
	}
	else {
	    $enable_ancestor = 'enable_ancestor';
	}
    }

    ## 
    ## retrieve info on data (id_data_description, id_agent, ...)
    ##
    my ($id_agent, $id_data_description, $data_type, $agent_activation_date) = 
        $self->getDataAndAgentInfo($source, $agent, $name);

    return 0 unless ($data_type and $id_data_description and $id_agent and $agent_activation_date);

    ##
    ## prepare timestring 
    ##
    my $timestring = "(date_part('epoch', collection_date) > $start)";
    $timestring .= " AND (date_part('epoch', collection_date) < $end)" if defined $end;
    
    my @data_row;
    my $st_handle;
    while($max_data) {
	##
	## Prepare id_data request
	##
	$data_type_table_name = "${data_type}_values";

	$st_handle = $self->{DBHANDLE}->prepare("
SELECT value,
       date_part('epoch', collection_date) 

FROM   $data_type_table_name

WHERE  (id_data_description = ?) AND
       $timestring

ORDER BY collection_date DESC

LIMIT  $max_data")
	    or die("Can't retrieve data for variable $name on agent $agent on source $source (prepare)");

	$st_handle->execute($id_data_description)	   
	    or die("Can't retrieve data for variable $name on agent $agent on source $source (execute)");


	while ( @data_row = $st_handle->fetchrow_array) {	    
	    unshift @result, [@data_row];
	}

	$max_data -= $st_handle->rows;
	
	$st_handle->finish;
	
	##
	## Stop here if ancestors are disabled
	##
	last unless $enable_ancestor eq 'enable_ancestor';	
	
	##
	## Stop here if (others) ancestors could not have data 
	## in the time interval
	##
	last if $agent_activation_date < $start;
	
	($id_data_description, $id_agent, $data_type, $agent_activation_date) = $self->getDataAncestorInfo($id_data_description);
	
	##
	## Stop here if there is no more ancestors
	##
	last unless ($data_type and $id_data_description and 
		     $id_agent and $agent_activation_date);
    }
    
    if($#result == -1) {
	return 0;
    }
    else {
	return \@result;
    }    
}


##*****************************************************************************
##  GetDataInRelativeTimeInterval public
##  Description  : Return all data in the specifed time interval relative 
##                 to agent last connection date
##
##  Parameters   : The maximum number of data to retrieve,
##                 The source name,
##                 The agent name,
##                 The data name,
##                 The start date,
##                 The end date (optional),
##                 Enable ancestor. String with 'enable_ancestor' or 'disable_ancestor'. 
##                                  Default value is 'enable_ancestor'.
##*****************************************************************************

sub GetDataInRelativeTimeInterval {
    my ($self, $max_data, $source, $agent, $name, $start, $end, $enable_ancestor) = @_;

    my @result;

    if(!(defined $end) and !(defined $enable_ancestor)) {
	$enable_ancestor = 'enable_ancestor';
    }
    elsif(defined $end and !(defined $enable_ancestor)) {		
	if($end eq 'disable_ancestor' or $end eq 'enable_ancestor') { 
	    $enable_ancestor = $end;
	    $end = undef;
	}
	else {
	    $enable_ancestor = 'enable_ancestor';
	}
    }

    ## 
    ## retrieve agent last connection date
    ##
    
    my @data_row;
    my $st_handle;
    
    $st_handle = $self->{DBHANDLE}->prepare("
SELECT date_part('epoch', lastconnectiondate)

FROM   agents

WHERE  id_source = ? AND
       number = ?")
	or die("Can't retrieve agent last connection date for agent $agent on source $source (prepare)");

    $st_handle->execute($source, $agent)
	or die("Can't retrieve agent last connection date for agent $agent on source $source (execute)");


    @data_row = $st_handle->fetchrow_array;
    
    $st_handle->finish;

    return 0 if $#data_row != 0;
    
    my $lastconnectiondate=$data_row[0];

    $start = $lastconnectiondate - $start;

    $end = 0 unless defined $end;

    $end = $lastconnectiondate - $end;

    return $self->GetDataInTimeInterval($max_data, $source, $agent, $name, $start, $end, $enable_ancestor);
}


##*****************************************************************************
##  GetCollectionDateInTimeInterval public
##  Description  : Return the list of collection date for a specified data
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name,
##                 The start date,
##                 The end date (optional).
##*****************************************************************************

sub GetCollectionDateInTimeInterval {
    my ($self, $source, $agent, $name, $start, $end, $enable_ancestor) = @_;
    my @result = ();

    $enable_ancestor = 'disable_ancestor';
    if(!defined $end  and !(defined $enable_ancestor)) {
	$enable_ancestor = 'enable_ancestor';
    }
    elsif(defined $end and !(defined $enable_ancestor)) {
	if($end eq 'enable_ancestor' or $end eq 'disable_ancestor') { 
	    $enable_ancestor = $end;
	    $end = undef;
	}
    }
    
    ##
    ## Retrieve informations on agents and data
    ##
    my ($id_agent, $id_data_description, $data_type, $agent_activation_date) = $self->getDataAndAgentInfo($source, $agent, $name);
    return 0 unless ($data_type and $id_data_description and $id_agent);
    
    ##
    ## prepare timestring 
    ##
    my $timestring = "(date_part('epoch', collection_date) > $start)";

    $timestring .= " AND (date_part('epoch', collection_date) < $end)" if defined $end;

    my $st_handle;
    my @row;
    while(1) {
	##
	## Prepare id_data request
	##
	$st_handle = $self->{DBHANDLE}->prepare("

SELECT date_part('epoch', collection_date)

FROM   data 

WHERE  (id_data_description = ?) AND 
       $timestring

ORDER BY collection_date ASC")
	    or die("Can't retrieve data collection date for variable $name on agent $agent on source $source (prepare)");

	$st_handle->execute($id_data_description)
	    or die("Can't retrieve data collection date for variable $name on agent $agent on source $source (execute)");

	while(@row = $st_handle->fetchrow_array()) {
	    push @result, $row[0];
	}

	$st_handle->finish;
	
	##
	## Stop here if ancestors are disabled
	##
	last unless $enable_ancestor eq 'enable_ancestor';	
	
	##
	## Stop here if (others) ancestors could not have data 
	## in the time interval
	##
	last if $agent_activation_date < $start;
	
	($id_data_description, $id_agent, $data_type, $agent_activation_date) = $self->getDataAncestorInfo($id_data_description);
	
	##
	## Stop here if there is no more ancestors
	##
	last unless ($data_type and $id_data_description and 
		     $id_agent and $agent_activation_date);
    }
    
    return \@result;
}

##*****************************************************************************
##  GetLastConnectionDate public
##  Description  : Return LastConnectionDate of an agent
##
##  Parameters   : The source name,
##                 The agent name,
##*****************************************************************************

sub GetLastConnectionDate {
    my ($self, $source, $agent) = @_;

    my $id_agent = $self->getIdAgent($source, $agent);

    return 0 unless $id_agent;

    my $st_handle = $self->{DBHANDLE}->prepare("

SELECT lastconnectiondate

FROM   agents

WHERE  agents.id_agent = ? ")
        or die("Can't retrieve connectiondate for agent $agent on source
$source (prepare)");

    $st_handle->execute($id_agent)
        or die("Can't retrieve connectiondate  for agent $agent on source
$source (execute)");

    my $rows = $st_handle->fetchall_arrayref();

    $st_handle->finish();

    return $rows;

}


##*****************************************************************************
##  GetAlarms public
##  Description  : Return all non acquitted alarms an complex alarms of an agent
##
##  Parameters   : The source name,
##                 The agent name,
##*****************************************************************************

sub GetAlarms {
    my ($self, $source, $agent) = @_;

    my $id_agent = $self->getIdAgent($source, $agent);

    return 0 unless $id_agent;
    
    my $st_handle = $self->{DBHANDLE}->prepare("

SELECT agents_alarms_descriptions.id_alarm_level, 
       agents_alarms_descriptions.message, 
       date_part('epoch', alarms.release_date) as release_date 

FROM   agents_alarms_descriptions, 
       alarms 

WHERE  (agents_alarms_descriptions.id_agent = ?) AND 
       (alarms.id_alarm_description = agents_alarms_descriptions.id_alarm_description) AND 
       (alarms.acquittal_date IS NULL)

ORDER BY release_date")
	or die("Can't retrieve alarms for agent $agent on source $source (prepare)");
    
    $st_handle->execute($id_agent)
	or die("Can't retrieve alarms for agent $agent on source $source (execute)");

    my $rows = $st_handle->fetchall_arrayref();
    
    $st_handle->finish();
    
    return $rows;
}



##*****************************************************************************
##  GetAlarmsInTimeInterval public
##  Description  : Return all alarm and complex alarms in a time interval
##
##  Parameters   : The maximum number of data to retrieve,
##                 The source name,
##                 The agent name,
##*****************************************************************************

sub GetAlarmsInTimeInterval {
    my ($self, $max_data, $source, $agent, $start, $end) = @_;

    my $id_agent = $self->getIdAgent($source, $agent);

    return 0 unless $id_agent;
    
    my $timestring_alarm = "date_part('epoch', alarms.release_date) > $start";

    if(defined $end) {
	$timestring_alarm .= " AND date_part('epoch', alarms.release_date) < $end";
    }

    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT agents_alarms_descriptions.id_alarm_level, 
       agents_alarms_descriptions.message,
       date_part('epoch', alarms.release_date),
       date_part('epoch', alarms.acquittal_date)

FROM   agents_alarms_descriptions, 
       alarms 

WHERE  (agents_alarms_descriptions.id_agent = ?) AND 
       (alarms.id_alarm_description = agents_alarms_descriptions.id_alarm_description) AND 
       $timestring_alarm

ORDER BY release_date DESC

LIMIT $max_data")
	or die("Can't retrieve alarms for agent $agent on source $source (prepare)");
    
    $st_handle->execute($id_agent)
	or die("Can't retrieve alarms for agent $agent on source $source (execute)");

    my $result = $st_handle->fetchall_arrayref;
    
    $st_handle->finish();
    
    if($#$result == -1) {
	return 0;
    }
    else {
	return $result;
    }    
}


##*****************************************************************************
##  AcquitAlarm public
##  Description  : Acquit the specified alarm
##
##  Parameters   : source number, agent number, alarm message and alarm level
##*****************************************************************************
sub AcquitAlarm {
    my ($self, $source, $agent, $name, $level) = @_;
    
    $self->{DBHANDLE}->do("
  UPDATE alarms 

  SET acquittal_date = 'now' 

  WHERE agents.id_source = ? AND
        agents.number = ? AND
        agents_alarms_descriptions.id_agent = agents.id_agent AND
        agents_alarms_descriptions.message = ? AND
        agents_alarms_descriptions.id_alarm_level = ? AND
        alarms.id_alarm_description = agents_alarms_descriptions.id_alarm_description", {}, 
                          $source, $agent, $name, $level)
	    or die "Can't acquit alarm";
    
    return 1;    
}


##*****************************************************************************
##  GetSourcesList public
##  Description  : Return the list of sources
##
##  Parameters   : none
##*****************************************************************************

sub GetSourcesList {
    my $self = shift;
    
    ##
    ## Get sources id
    ##
    my $st_handle_source = $self->{DBHANDLE}->prepare("
SELECT id_source

FROM   sources")
        or die("Can't retrieve the sources list (prepare)");

    $st_handle_source->execute()
        or die("Can't retrieve the sources list (execute)");

    my $st_handle_info = $self->{DBHANDLE}->prepare("
SELECT description, 
       value 

FROM   sources_informations

WHERE id_source = ?")
    # Run the request
        or die("Can't retrieve infos (prepare)");
    

    ##
    ## For each sources id, retrieve informations
    ##
    my @result = ();
    my @row;
    my @info_row;
    while(@row = $st_handle_source->fetchrow_array) {
	my @source_entry = ();
	$st_handle_info->execute($row[0])
	    or die("Can't retrieve infos of source $row[0] (execute)");
	
	push @source_entry, $row[0];

	while(@info_row = $st_handle_info->fetchrow_array) {
	    push @source_entry, @info_row;
	}
	
	push @result, \@source_entry;
    }

    $st_handle_source->finish();
    $st_handle_info->finish();
    
    return \@result;
}


##*****************************************************************************
##  GetListOfAgentsOnSOurce public
##  Description  : Return the list of agents on a source.
##
##  Parameters   : The source name.
##*****************************************************************************

sub GetListOfAgentsOnSource {
    my ($self, $source) = @_;

    my  $st_handle = $self->{DBHANDLE}->prepare("
SELECT number, 
       information 

FROM agents 

WHERE id_source =  ? 

ORDER BY number ")
        or die("Can't retrieve the agents list names of source $source_number (prepare)");
    
    # Run the request
    $st_handle->execute($source)
        or die("Can't retrieve the sources list (execute)");
    
    my $rows = $st_handle->fetchall_arrayref();
    
    $st_handle->finish();
    
    return $rows;    
}



##*****************************************************************************
##  GetListOfDataNamesOnAgent public
##  Description  : Return the list of agents on a source.
##
##  Parameters   : The source name,
##                 The agent name.
##*****************************************************************************

sub GetListOfDataNameOnAgent {
    my ($self, $source, $agent) = @_;
    
    $id_agent = $self->getIdAgent($source, $agent);
    return 0 unless $id_agent;

    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT description 

FROM   agents_data_descriptions 

WHERE  id_agent = ?")
        or die("Can't retrieve the variables names with source $source and agent $agent (prepare)");

    $st_handle->execute($id_agent)
        or die("Can't retrieve the variables names with source $source an
d agent $agent (execute)");

    my $rows = $st_handle->fetchall_arrayref();
    my @col;
    my $str;

    $st_handle->finish();
    
    foreach $row (@$rows) {
        $str = lc($row->[0]);
        $str =~ s/\s/_/g;
        $str =~ s/_*(\(.*\))_*//g;

	push @col, $str;
    }

    return \@col;
}



##*****************************************************************************
##  getIdDataDescription protected
##  Description  : Return id_data_description
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name.
##*****************************************************************************

sub getIdDataDescription {
    my ($self, $source, $agent, $name) = @_;

    # Convert the name into a PostgreSQL regular expression
    $name_reg_exp = "\\^";
    $name_reg_exp .= $name;
    $name_reg_exp =~ s/_/ /g;
    $name_reg_exp .= "\\( (.*)\\)?\\\$";


    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT id_data_description 

FROM   agents_data_descriptions,
       agents

WHERE  agents.id_source = ? AND
       agents.number = ? AND
       agents_data_descriptions.id_agent = agents.id_agent AND 
       description ~* '$name_reg_exp'")
	or die("Can't retrieve the id_data_description of data $name on agent $agent on source $source (prepare)");

    # Run the request
    $st_handle->execute($source, $agent)
	or die("Can't retrieve the id_data_description of data $name on agent $agent on source $source (execute)");

    my @row = $st_handle->fetchrow_array();

    $st_handle->finish();

    if($#row == -1) {
	return 0;
    }
    else {
	return $row[0];
    }
}

##*****************************************************************************
##  getIdAgent protected 
##  Description  : Return id_agent
##
##  Parameters   : The source name,
##                 The agent name.
##*****************************************************************************

sub getIdAgent {
    my ($self, $source, $agent) = @_;       

    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT id_agent

FROM   agents 

WHERE  agents.id_source = ? AND 
       agents.number = ?")
	or die("Can't retrieve the id_agent of agent $agent_number on source $source_number (prepare)");

    # Run the request
    $st_handle->execute($source, $agent)
	or die("Can't retrieve the id_agent of agent $agent_number on source $source_number (execute)");

    my @row = $st_handle->fetchrow_array();

    $st_handle->finish();

    if($#row == -1) {
	return 0;
    }
    else {
	return $row[0];
    }
}


##*****************************************************************************
##  getDataAndAgentInfo protected 
##  Description  : Return id_agent, id_data_description and data type
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name.
##*****************************************************************************
sub getDataAndAgentInfo {
    my ($self, $source, $agent, $name) = @_;       

    # Convert the name into a PostgreSQL regular expression
    $name_reg_exp = "\\^";
    $name_reg_exp .= $name;
    $name_reg_exp =~ s/_/ /g;
    $name_reg_exp .= "\\( (.*)\\)?\\\$";

    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT agents.id_agent, agents_data_descriptions.id_data_description, lower(data_types.type), date_part('epoch', agents.activationdate)

FROM   agents, agents_data_descriptions, data_types

WHERE  agents.id_source = ? AND 
       agents.number = ? AND
       agents_data_descriptions.id_agent = agents.id_agent AND 
       agents_data_descriptions.description ~* '$name_reg_exp' AND 
       (data_types.id_data_type = agents_data_descriptions.type)")
	or die("Can't retrieve the id_agent of agent $agent_number on source $source_number (prepare)");

    # Run the request
    $st_handle->execute($source, $agent)
	or die("Can't retrieve the id_agent of agent $agent_number on source $source_number (execute)");

    my @row = $st_handle->fetchrow_array();

    $st_handle->finish();

    return @row;
}
##*****************************************************************************
##  getDataAncestorInfo private
##  Description  : Return id_agent
##
##  Parameters   : The data description id.
##  
##  Return       : The data description id of the ancestor, 
##                 The agent id of the ancestor,
##                 The data type of the ancestor.
##*****************************************************************************

sub getDataAncestorInfo {
    my ($self, $id_data_description) = @_;

    my $st_handle = $self->{DBHANDLE}->prepare("
SELECT ancestor, agents.id_agent, lower(data_types.type), date_part('epoch', agents.activationdate)

FROM   agents_data_descriptions, data_types, agents

WHERE  id_data_description = ? AND data_types.id_data_type = agents_data_descriptions.type AND agents.id_agent = agents_data_descriptions.id_agent") 
	or die("Can't retrieve the ancestor of data description $id_data_description (prepare)");

    # Run the request
    $st_handle->execute($id_data_description)
	or die("Can't retrieve the id_agent of agent $agent_number on source $source_number (execute)");

    my @row = $st_handle->fetchrow_array();

    $st_handle->finish();

    return @row;
}

1;
