#!/usr/bin/perl -w

#$Header: /mnt/u1/cvs/logtrend/logtrend-doc/doc/LogTrend-SimpleAgent-0.82.2/lib/LogTrend/Crypto/KeyManager.pm,v 1.1 2002/04/10 01:18:47 jdive Exp $
##******************************************************************************
## KeyManager
##  Description  : Manage access to RSA public keys stored in DataBase 
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##
##******************************************************************************
#$Log: KeyManager.pm,v $
#Revision 1.1  2002/04/10 01:18:47  jdive
#fixed description, uncompressed simpleagent
#
#Revision 1.3.2.4  2002/02/20 17:43:13  lsimonneau
#*** empty log message ***
#
#Revision 1.9  2002/02/20 17:40:57  lsimonneau
#*** empty log message ***
#
#Revision 1.3.2.3  2002/02/20 17:11:32  lsimonneau
#*** empty log message ***
#
#Revision 1.3.2.2  2002/02/20 14:25:34  lsimonneau
#*** empty log message ***
#
#Revision 1.7  2002/02/20 14:23:51  lsimonneau
#*** empty log message ***
#
#Revision 1.5  2002/02/20 13:51:27  lsimonneau
#Change Reply to instead of Return-Path in agents and MailBridge.pl.
#Major bugfixes in StorageServer GPG Key rings management.
#
#Revision 1.4  2002/02/13 17:00:45  lsimonneau
#Bug fixe in key ring managment
#
#Revision 1.3  2001/12/10 17:37:28  lsimonneau
#Minor bugfixes.
#
#Revision 1.2  2001/12/10 16:58:34  lsimonneau
#Use GnuPG::Interface instead of Crypt::RSA for perl 5.005 compatibility.
#
#Revision 1.1  2001/11/16 10:28:09  lsimonneau
#First version of this modules.
#Used for RSA signature authentication.
#
package LogTrend::Crypto::KeyManager;

use strict;
use LogTrend::Crypto::PostgreSQLDataBase;
use Error qw(:try);
use POSIX qw(tmpnam);
use LogTrend::Crypto::ErrorDeclaration;
use IO::Handle;
use GnuPG::Interface;
use Fcntl ':flock';


##*****************************************************************************
## Constructor  public
##  Description  : creat a new key manager
##
##  Parameters   : The database name,
##                 The database server host name,
##                 The database server port,
##                 An username and a password of a database user
##
##  Don't catch error : Error::DB_Request
##                      Error::DB_Connection
##                      Error::IO
##*****************************************************************************
sub new
{
    my ($classname, $databasename, $host, $port, $username, $password) = @_;
    my $self = {};
    
    bless($self, $classname);    
    
    $self->{DB_NAME} = $databasename;
    $self->{DB_HOST} = $host;
    $self->{DB_PORT} = $port;
    $self->{DB_USERNAME} = $username;
    $self->{DB_PASSWORD} = $password;

    ## Create a GnuPG::Interface object
    $self->{GPGHOME} = tmpnam;
    mkdir ($self->{GPGHOME}, 0700)
        or throw Error::IO("Can't create $self->{GPGHOME} : $!");
    

    open(F_READ_LOCK, "+>$self->{GPGHOME}/read_lock");
    close(F_READ_LOCK);
    open(F_RELOAD_KEYS_NEEDED_LOCK, "+>$self->{GPGHOME}/reload_keys_needed_lock");
    close(F_RELOAD_KEYS_NEEDED_LOCK);
    open(F_RELOAD_KEYS_IN_PROGRESS_LOCK, "+>$self->{GPGHOME}/reload_keys_in_progress_lock");
    close(F_RELOAD_KEYS_IN_PROGRESS_LOCK);
    
    $self->{GPG} = new GnuPG::Interface;
    $self->{GPG}->options->hash_init( armor    => 1,
                                      homedir  => $self->{GPGHOME},
                                      always_trust => 1);
    
    $self->load_public_keys;
    
    return $self;
}



##*****************************************************************************
## Method  load_public_keys private
##  Description  : load public keys for database
##
##  Parameters   : None
##
##  Throw : Error::IO
##
##  Don't catch error : Error::DB_Request
##*****************************************************************************
sub load_public_keys
{
    my $self = shift;

    ## Lock GnuPG exclusively (disable read lock)
    ## Lock NEED_RELOAD exclusively
    open(F_RELOAD_KEYS_NEEDED_LOCK, "$self->{GPGHOME}/reload_keys_needed_lock");
    flock(F_RELOAD_KEYS_NEEDED_LOCK, LOCK_EX);
    
    ## Try to lock RELOAD_IN_PROGRESS exclusively
    ## If this lock is already locked, it's because a reload is in progress
    ## So, give up and return
    open(F_RELOAD_KEYS_IN_PROGRESS_LOCK, "$self->{GPGHOME}/reload_keys_in_progress_lock");
    if(! flock(F_RELOAD_KEYS_IN_PROGRESS_LOCK, LOCK_EX | LOCK_NB)) {
        flock(F_RELOAD_KEYS_NEEDED_LOCK, LOCK_UN);
        close(F_RELOAD_KEYS_NEEDED_LOCK);
        return;
    }

    ## Wait for exclusive lock on READ_LOCK
    open(F_READ_LOCK, "$self->{GPGHOME}/read_lock");
    flock(F_READ_LOCK, LOCK_EX);

    ## Unlock NEED_RELOAD to force other processes to give up this task
    flock(F_RELOAD_KEYS_NEEDED_LOCK, LOCK_UN);
    close(F_RELOAD_KEYS_NEEDED_LOCK);    


    ## Reload keys
    unlink("$self->{GPGHOME}/pubring.gpg");

    $self->{SOURCE_LIST} = {};

    # Connect to the database
    my $database = new LogTrend::Crypto::PostgreSQLDataBase($self->{DB_NAME}, $self->{DB_HOST}, $self->{DB_PORT},
                                                            $self->{DB_USERNAME}, $self->{DB_PASSWORD});
    
    my $list = $database->getListOfKeys;
    
    my $input   = IO::Handle->new();
    my $output  = IO::Handle->new();
    my $error   = IO::Handle->new();
    my $status_fh   = IO::Handle->new();
    my $handles = GnuPG::Handles->new( stdin  => $input,
                                       stdout => $output,
                                       stderr => $error,
                                       status => $status_fh,
                                       );
    
    # Add public keys to GPG key ring.
    my $key_nbr = @$list;
    
    my $import_pid = $self->{GPG}->import_keys( handles => $handles );
    for(my $i = 0; $i < $key_nbr; $i++) {
        my $pub_key = $list->[$i]->[0];
        
        print $input $pub_key;
    }
    
    close $input;
    waitpid($import_pid, 0);
    
    my $imported_key = 0;
    my $stat;
    my $i=0;
    while(($stat = <$status_fh>) =~ /^\[GNUPG:\] IMPORTED ([^\s]+) (.+)$/) {
        $self->{SOURCE_ID}->{$2} = $list->[$i]->[1];
        $i++;
    }
    
    if($stat =~ /^\[GNUPG:\] IMPORT_RES (\d+)/) {
        $imported_key = $1;
    }
    else {
        flock(F_RELOAD_KEYS_IN_PROGRESS_LOCK, LOCK_UN);
        close(F_RELOAD_KEYS_IN_PROGRESS_LOCK);
      
        flock(F_READ_LOCK, LOCK_UN);
        close(F_READ_LOCK);

        throw Error::IO("Unexpected response from GPG : $stat\n");
    }
    
    if($imported_key != $key_nbr) {
        my @err = <$error>;
        warn "Warning : Only $imported_key key(s) has been imported for database.\n".
             "          There are probably invalid data in database.".
             "Errors returned by GPG :\n".
             "@err";
    }    
    
    close $output;
    close $error;
    close $status_fh;    


    ## Wait a little second to ensure that other waiting processes
    ## give up the reload. 
    sleep(1);

    ## Unlock all locks 
    flock(F_RELOAD_KEYS_IN_PROGRESS_LOCK, LOCK_UN);
    close(F_RELOAD_KEYS_IN_PROGRESS_LOCK);

    flock(F_READ_LOCK, LOCK_UN);
    close(F_READ_LOCK);
    
}



##*****************************************************************************
## Method verify_signature public
##  Description  : try to authenticate the signature.
##
##  Parameters   : A message and an armoured GPG signature
##
##  Return       : A hash like ( source_id => NBR,
##                               identity => "The key owner",
##                             )
##
##  Throw error : Error::Authentication,
##                Error::Not_A_Signature,
##                Error::IO
##*****************************************************************************
sub verify_signature {
    my ($self, $message, $signature) = @_;
    
    ## Verify if the signature is a valid GPG armoured signature
    if($signature !~ /\-+BEGIN PGP SIGNATURE\-+.+\-+END PGP SIGNATURE\-+/s) {
	throw Error::Not_A_Signature("");
    }

    my $result;
    try {
        $result = $self->real_verify_signature($message, $signature);
    }
    catch Error::Authentication with {
        ## If the authentication has failed, reload keys from database and 
        ## retry to authentificate the source

	$self->load_public_keys;
	$result = $self->real_verify_signature($message, $signature);        
    };
    
    return $result;
}




##*****************************************************************************
## Method real_verify_signature private
##  Description  : try to authenticate the signature.
##
##  Parameters   : the message and an armoured GPG signature
##
##  Return       : A hash like ( source_id => NBR,
##                               identity => "The key owner",
##                             )
##                 if the signature is authenticated, or undef otherwise.
##
##*****************************************************************************
sub real_verify_signature {
    my ($self, $message, $signature) = @_;

    my $datafile = tmpnam;
    my $signfile = tmpnam;

    open (DATA, ">$datafile")
        or throw IO::Error("Can't open $datafile : $!");
    print DATA $message
        or throw IO::Error("Can't access to $datafile : $!");
 
    close DATA
        or throw IO::Error("Can't close $datafile : $!");
    
    open (SIGN, ">$signfile")
        or throw IO::Error("Can't open $signfile : $!");

    print SIGN $signature
        or throw IO::Error("Can't access to $signfile : $!");

    close SIGN
        or throw IO::Error("Can't close $signfile : $!");
    


    my $input   = IO::Handle->new();
    my $output  = IO::Handle->new();
    my $error   = IO::Handle->new();
    my $status_fh = IO::Handle->new();

    ## Lock GnuPG in shared mode
    open(F_READ_LOCK, "$self->{GPGHOME}/read_lock");
    flock(F_READ_LOCK, LOCK_SH);
            
    my $handles = GnuPG::Handles->new( stdin  => $input,
                                       stdout => $output,
                                       stderr => $error,
                                       status => $status_fh,
                                       );

    my $verify_pid = $self->{GPG}->wrap_call ( handles  => $handles,
                                               commands => "--verify",
                                               command_args => [$signfile, $datafile],
                                             );

    waitpid( $verify_pid, 0);

    ## Unlock GnuPG
    flock(F_READ_LOCK, LOCK_UN);
    close(F_READ_LOCK);
    
    close($input);
    close($output);
    close($error);


    unlink $signfile, $datafile;

    my %res;
    while( my $status = <$status_fh> ) {
        if($status =~ /\[GNUPG:\] GOODSIG ([^\s]+) (.+)/){
            if(! exists $self->{SOURCE_ID}->{$2}) {
                throw Error::Authentication("Good signature, but unknown source\n");
            }

            $res{identity} = $2;
            $res{source_id} = $self->{SOURCE_ID}->{$2};
            
        }
        elsif($status =~ /\[GNUPG:\] VALIDSIG ([^\s]+) ([^\s]+) ([^\s]+)/) {
        }
        elsif($status =~ /\[GNUPG:\] SIG_ID ([^\s]+) ([^\s]+) ([^\s]+)/) {
        }
        elsif($status =~ /\[GNUPG:\] BADSIG/){            
            throw Error::Authentication("Bad signature");
        }
        elsif($status =~ /\[GNUPG:\] ERRSIG/) {
            throw Error::Authentication("Error in signature");
        }
        elsif($status =~ /^\[GNUPG:\] NODATA/) {
            throw Error::Not_A_Signature("Not signature data");
        }
        elsif($status =~ /^\[GNUPG:\] UNEXPECTED/) {
            throw Error::Not_A_Signature("Unsepected data in signature");
        }
        elsif($status =~ /^\[GNUPG:\] TRUST_NEVER/) {
            throw Error::Authentication("User is trusted with NEVER");
        }
        elsif($status =~ /^\[GNUPG:\] TRUST_\w+/) {
        }
        elsif($status =~ /^\[GNUPG:\] SIGEXIPRED/) {
            throw Error::Authentication("Signature expired");
        }
        elsif($status =~ /^\[GNUPG:\] KEYREVOKED/) {
            throw Error::Authentication("Public key revoked");
        }
        elsif($status =~ /^\[GNUPG:\] BADARMOR/) {
            throw Error::Not_A_Signature("Bad armored data");
        }
        elsif($status =~ /^\[GNUPG:\] NO_PUBKEY/) {
            throw Error::Authentication("Can't found public key");
        }
        else {
            throw Error::IO("Unexpected response from GPG : $status\n");
        }

    }
    
    close($status_fh);

    return \%res;
}

1;
