#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Agent/AgentDescriptionToDB/XMLToDB.pm,v 1.6 2001/11/16 10:32:28 lsimonneau Exp $
#
#******************************************
#                                         *
# LogTrend 1.0.0.0 - Atrid Systemes       *
#                                         *
# Author : David Mascle d.mascle@atrid.fr *
#                                         *
#******************************************
#
#$Log: XMLToDB.pm,v $
#Revision 1.6  2001/11/16 10:32:28  lsimonneau
#Major feature enhancement : Implementation of authentification with RSA signatu
#re.
#
#IMPORTANT : Agents are incompatible with older version.
#
#Revision 1.5  2001/10/05 11:55:08  slhullier
#in the other case -> otherwise
#
#Revision 1.4  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.3  2001/02/20 17:01:27  slhullier
#
#Mise en commentaire (+tab etc)
#
#Revision 1.2  2001/02/20 15:50:20  slhullier
#
#Apres modification du nom de certaines classes
#
#Revision 1.1.1.1  2001/02/06 09:48:28  fdubuy
#
#First CVS version : all work done by D.Mascle
#
#


package LogTrend::Agent::AgentDescriptionToDB::XMLToDB;

use strict;
use XML::DOM;
use Error;
use LogTrend::Agent::AgentDescriptionToDB::ErrorDeclaration;

#**********************************************************************************
#                                                                                 *
# XMLToDB if a class which is in charge of extracting data from an xml agent's   *
# description in order to store them in a database.                               *
#                                                                                 *
#**********************************************************************************

#**************************************************************************
#
# status : public
#
# function : constructor
#
# parameters :
#
# dblayer : a reference to a connected dblayer
# filename : an agent's description file (in XML format)
#
# returns : a reference to an instance of the class
#
#**************************************************************************

sub new
  {
    my ($classname, $dblayer, $filename) = @_;
    my $self = {};
    bless($self, $classname);
    $self->{DBLAYER} = $dblayer;
    return $self;
  }

#*****************************************************************************************
#
# status : public
#
# function : extracts data from an agent's description file and saves it to the database
#
# parameters : none
#
# Throw error : Error::XML
#               Error::DB_Request
#               Error::DB_AgentDescription
#*****************************************************************************************

sub ConvertXMLToDB
  {
    my ($self, $source_id, $xmldata) = @_;
    my $result = 0;

    # creates the DOM
    my $parser = new XML::DOM::Parser(ErrorContext => 1);
    # parses the agent's description file
    my $doc;
    eval
    {
        # the eval is used to prevent from savage exits from the DOM object
        $doc = $parser->parse($xmldata);
    };
   
    if($@) {
        throw Error::XML($@);
    }
    
    # get the root node of the document
    my $rootnodelist = $doc->getElementsByTagName("AgentDescription");

    if(! @$rootnodelist ) {
        throw Error::XML("Can't find AgentDescription tag");
    }
    
    my $rootnode = $rootnodelist->item(0);
    $self->{AGENTSOURCE} = $self->GetAgentSource($rootnode);
    if($self->{AGENTSOURCE} != $source_id) {
        throw Error::Permission("You are authenticated as source $source_id.\n".
                                "You can not access to data of the source $self->{AGENTSOURCE}");
    }
    
    $self->{AGENTNUMBER} = $self->GetAgentNumber($rootnode);
    $self->{AGENTVERSION} = $self->GetAgentVersion($rootnode);
    $self->{AGENTACTIVATIONDATE} = $self->GetAgentActivationDate($rootnode);
    $self->{AGENTINFORMATION} = $self->GetAgentInformation($rootnode);
    $self->{AGENTDATADESCRIPTIONSETNODE} = $self->GetAgentDataDescriptionSetNode($rootnode);
    $self->{AGENTALARMDESCRIPTIONSETNODE} = $self->GetAgentAlarmDescriptionSetNode($rootnode);
    
    # we add the agent's description to the database
    $self->{DBLAYER}->AddAgent($self->{AGENTSOURCE}, $self->{AGENTNUMBER}, $self->{AGENTVERSION}, $self->{AGENTACTIVATIONDATE}, $self->{AGENTINFORMATION});
    
    # we add the data's and the alarm's descriptions to the database
    $self->AddDataDescriptionSetToDB();
    $self->AddAlarmDescriptionToDB();
}

#*********************************************
#
# status : private
#
# function : gets the first subnode of a node
#
# parameters : a node and a subnode name
#
# returns : a subnode or nothing
#
#*********************************************

sub GetANode
  {
    my ($self, $node, $nodename) = @_;
    my $nodelist = $node->getElementsByTagName($nodename);
    
    if(! @$nodelist ) {
        throw Error::XML("Can't find $nodename tag");
    }
    
    my $thenode = $nodelist->item(0);
    return $self->GetValue($thenode);
}

#*******************************************************
#
# status : private
#
# function : gets the first subnode reference of a node
#
# parameters : a node and a subnode name
#
# returns : a subnode reference or nothing
#
#*******************************************************

sub GetANodeRef {
    my ($self, $node, $nodename) = @_;
    my $result = undef;

    my $nodelist = $node->getElementsByTagName($nodename);

    if(! @$nodelist ) {
        throw Error::XML("Can't find $nodename tag");
    }

    return  $nodelist->item(0);
}

#***************************************************
#
# status : private
#
# function : gets the source of an agent
#
# parameters : the parent's node of the source node
#
# returns : the source of an agent
#
#***************************************************

sub GetAgentSource {
    my ($self, $node) = @_;

    my $result = $self->GetANode($node, "Source");

    return $result;
}

#***************************************************
#
# status : private
#
# function : gets the number of an agent
#
# parameters : the parent's node of the number node
#
# returns : the number of an agent
#
#***************************************************

sub GetAgentNumber  {
    my ($self, $node) = @_;

    my $result = $self->GetANode($node, "Number");

    return $result;
}

#****************************************************
#
# status : private
#
# function : gets the version of an agent
#
# parameters : the parent's node of the version node
#
# returns : the version of an agent
#
#****************************************************

sub GetAgentVersion {
    my ($self, $node) = @_;

    my $result = $self->GetANode($node, "Version");

    return $result;
}

#***********************************************************
#
# status : private
#
# function : gets the activationdate of an agent
#
# parameters : the parent's node of the activationdate node
#
# returns : the activationdate of an agent
#
#***********************************************************

sub GetAgentActivationDate {
    my ($self, $node) = @_;

    my $result = $self->GetANode($node, "ActivationDate");

    return $result;
}

#********************************************************
#
# status : private
#
# function : gets the information of an agent
#
# parameters : the parent's node of the information node
#
# returns : the information of an agent
#
#********************************************************

sub GetAgentInformation {
    my ($self, $node) = @_;
    
    my $result = $self->GetANode($node, "Information");
    
    return $result;
}

#************************************************
#
# status : private
#
# function : gets the value associated to a node
#
# parameters : a node
#
# returns : a node value
#
#************************************************

sub GetValue {
    my ($self, $node) = @_;
    
    my $nodevalueref = $node->getFirstChild();
    if (!defined $nodevalueref)
    {
        throw Error::XML("Can't extract value of a tag");
    }

    return $nodevalueref->getNodeValue();
}

#********************************************************************
#
# status : private
#
# function : gets an agent's data description node
#
# parameters : the parent's node of an agent's data description node
#
# returns : the agent's data description node
#
#********************************************************************

sub GetAgentDataDescriptionSetNode {
    my ($self, $node) =@_;

    my $result = $self->GetANodeRef($node, "DataDescriptionSet");

    return $result;
}

#*********************************************************************
#
# status : private
#
# function : gets the agent's alarm description node
#
# parameters : the parent's node of an agent's alarm description node
#
# returns : the agent's alarm description node
#
#*********************************************************************

sub GetAgentAlarmDescriptionSetNode {
    my ($self, $node) =@_;
    
    my $result = $self->GetANodeRef($node, "AlarmDescriptionSet");

    return $result;
}

#*******************************************
#
# status : private
#
# function : gets the type of a type's node
#
# parameters : a type's node
#
# returns : the type of a type's node
#
#*******************************************

sub GetDataType {
    my ($self, $node) = @_;
    my $result;

    my $typenodelist = $node->getElementsByTagName("Type");
    if (defined $typenodelist)
    {
        my $typenode = $typenodelist->item(0);
        if (defined $typenode)
        {
            if ($typenode->getElementsByTagName("Text")->getLength() > 0)
            {
                $result = "Text";
            }
            elsif ($typenode->getElementsByTagName("Integer")->getLength() > 0)
            {
                $result = "Integer";
            }
            elsif ($typenode->getElementsByTagName("Real")->getLength() > 0)
            {
                $result = "Real";
            }
            elsif ($typenode->getElementsByTagName("Date")->getLength() > 0)
            {
                $result = "Date";
            }
            elsif ($typenode->getElementsByTagName("Time")->getLength() > 0)
            {
                $result = "Time";
            }
            elsif ($typenode->getElementsByTagName("DateTime")->getLength() > 0)
            {
                $result = "DateTime";
            }
            else {
                throw Error::XML("Can't find a known data type in description");
            }
        }
    }
    
    return $result;
}

#*************************************************************************
#
# status : private
#
# function : gets the data's description of a data's description node
#
# parameters : a data's description node
#
# returns : a table made of the data's number, type, unit and description
#
#*************************************************************************

sub GetDataDescription {
    my ($self, $node) = @_;
    
    my $number = $self->GetANode($node, "Number");
    my $type = $self->GetDataType($node);
    my $unit = $self->GetANode($node, "Unit");
    my $description = $self->GetANode($node, "Description");
    
    return ($number, $type, $unit, $description);
}

#*********************************************************************
#
# status : private
#
# function : adds the data's descriptions if possible to the database
#
# parameters : none
#
#*********************************************************************

sub AddDataDescriptionSetToDB {
    my $self = shift;
    my $result = 1;

    my @datadescriptionsetlist = $self->{AGENTDATADESCRIPTIONSETNODE}->getElementsByTagName("Data");
    
    my $nbdata = @datadescriptionsetlist;
    for (my $i = 0; $i < $nbdata; $i++) {
        my $datanode = $datadescriptionsetlist[$i];
        my ($number, $type, $unit, $description) = $self->GetDataDescription($datanode);

        # we now populate the database with the data's descriptions
        $self->{DBLAYER}->AddDataDescription($self->{AGENTSOURCE}, $self->{AGENTNUMBER}, $self->{AGENTVERSION},
                                             $self->{AGENTACTIVATIONDATE}, $number, $type, $unit, $description);
    }
}

#**********************************************************************
#
# status : private
#
# function : adds the alarm's descriptions if possible to the database
#
# parameters : none
#
# returns : 1 if successfull and 0 otherwise
#
#**********************************************************************

sub AddAlarmDescriptionToDB
  {
    my $self = shift;

    my @alarmdescriptionsetlist = $self->{AGENTALARMDESCRIPTIONSETNODE}->getElementsByTagName("Alarm");

    my $nbalarm = @alarmdescriptionsetlist;
    for (my $i = 0; $i < $nbalarm; $i++) {
        my $alarmnode = $alarmdescriptionsetlist[$i];
        my ($number, $level, $message) = $self->GetAlarmDescription($alarmnode);
        
        # we now populate the database with the alarm's descriptions
        $self->{DBLAYER}->AddAlarmDescription($self->{AGENTSOURCE}, $self->{AGENTNUMBER}, $self->{AGENTVERSION}, 
                                              $self->{AGENTACTIVATIONDATE}, $number, $level, $message);
    }
}

#**************************************************************************
#
# status : private
#
# function : gets an alarm level associated to an alarm's description node
#
# parameters : an alarm's description node
#
# returns : an alarm level
#
#**************************************************************************

sub GetAlarmLevel {
    my ($self, $node) = @_;
    my $result;

    my $levelnodelist = $node->getElementsByTagName("Level");

    if(! @$levelnodelist) {
        throw Error::XML("Can't found Level tag");
    }

    my $levelnode = $levelnodelist->item(0);
    if ($levelnode->getElementsByTagName("Info")->getLength() > 0) {
        $result = "Info";
    }
    elsif ($levelnode->getElementsByTagName("Warning")->getLength() > 0) {
        $result = "Warning";
    }
    elsif ($levelnode->getElementsByTagName("Error")->getLength() > 0) {
        $result = "Error";
    }
    else {
        throw Error::XML("Can't find Alarm Level in agent description");
    }
    
    return $result;
}

#************************************************************************
#
# status : private
#
# function : gets the alarm's description of an alarm's description node
#
# parameters : an alarm's description node
#
# returns : a table made of an alarm's number, level and message
#
#************************************************************************

sub GetAlarmDescription {
    my ($self, $node) = @_;
    
    my $number = $self->GetANode($node, "Number");;
    my $level = $self->GetAlarmLevel($node);
    my $message = $self->GetANode($node, "Message");;
    
    return ($number, $level, $message);
}

1;
