#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Visu/Web/Zone/Dial.pm,v 1.7 2001/11/19 12:55:50 slhullier Exp $
##******************************************************************************
## Class Dial  isa Zone
##  Description  : Zone de type 'compteur de vitesse' (aka Dial)
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Authors      : Christophe Najar c.najar@atrid.fr
##               : Laurent Simonneau l.simonneau@atrid.fr
##******************************************************************************
#$Log: Dial.pm,v $
#Revision 1.7  2001/11/19 12:55:50  slhullier
#Accent bug fixed at all ...
#
#Revision 1.6  2001/11/19 10:56:54  slhullier
#little bit stable, before big modification
#
#Revision 1.5  2001/11/16 16:45:01  slhullier
#Text color format
#
#Revision 1.4  2001/11/15 16:08:57  slhullier
#UserNode managing first version (still bugs)
#
#Revision 1.3  2001/11/14 15:59:45  slhullier
#No more global value for SystemsDeclaration
#
#Revision 1.2  2001/11/12 10:03:00  slhullier
#no more tab
#
#Revision 1.1  2001/10/24 09:02:56  lsimonneau
#Remove Web or Report Prefix from Zone files.
#
#Revision 1.1  2001/10/23 12:19:39  lsimonneau
#Add Web Zone implementation.
#
#Revision 1.24  2001/10/05 11:11:09  slhullier
#Some french texts to english
#
#Revision 1.23  2001/09/19 08:42:40  lsimonneau
#Major bugfixes : use Expression in stead of DataBaseAccess.
#
#Revision 1.22  2001/08/31 16:03:21  lsimonneau
#Minor bugfixes.
#
#Revision 1.21  2001/08/31 07:26:08  lsimonneau
#Minor bugfixes.
#
#Revision 1.20  2001/08/24 12:18:54  lsimonneau
#Remove Debug print.
#
#Revision 1.19  2001/08/23 15:46:03  lsimonneau
#Decrease minimal size.
#
#Revision 1.18  2001/08/23 11:51:03  lsimonneau
#Debuggage dial, now, dial could not be larger than the zone.
#
#Revision 1.17  2001/08/22 12:43:49  lsimonneau
#Bug fixe : Print NO DATA where no data was found in DataBase.
#
#Revision 1.16  2001/08/21 08:45:07  slhullier
#
#Colors personnalisable + dial mis en place
#
#Revision 1.15  2001/08/17 13:32:43  slhullier
#
#Correct accents for pictures/HTML Perl 5.005/5.6
#
#Revision 1.14  2001/08/10 10:02:46  lsimonneau
#*** empty log message ***
#
#Revision 1.13  2001/08/10 09:35:11  lsimonneau
#Modification des tags de couleurs. Couleur des aiguilles rglable.
#
#Revision 1.12  2001/08/09 09:48:16  lsimonneau
#Amlioration de la gestion des valeur Max et Min ainsi que de la graduation.
#
#Revision 1.11  2001/08/09 08:32:57  lsimonneau
#Correction de bug, limitation de la taille maximale.
#
#Revision 1.10  2001/08/08 15:34:30  lsimonneau
#Amlioration des couleurs et ajout des graduations.
#
#Revision 1.9  2001/08/08 11:41:45  lsimonneau
#Minor enhancements.
#
#Revision 1.8  2001/08/08 08:42:35  lsimonneau
#Rcriture complte du code graphique.
#
#Revision 1.7  2001/08/06 16:28:08  lsimonneau
#Debuggage/Recriture partiel de Dial.
#
#Revision 1.6  2001/08/03 13:07:27  slhullier
#*** empty log message ***
#
#Revision 1.5  2001/08/02 18:50:17  cnajar
#*** empty log message ***
#
#Revision 1.4  2001/08/02 12:19:17  cnajar
#*** empty log message ***
#
#Revision 1.2  2001/07/30 16:20:53  cnajar
#*** empty log message ***
#
#Revision 1.1  2001/07/23 09:32:40  cnajar
#Premier version
#

package LogTrend::Visu::Web::Zone::Dial;

use strict;
use GD;
use Math::Trig;

use LogTrend::Common::Duration;
use LogTrend::Visu::Utils;
use LogTrend::Visu::Zone;
use LogTrend::Visu::Request;
use LogTrend::Visu::Response;
use LogTrend::Visu::Constants;

use vars qw( @ISA );
@ISA = ("LogTrend::Visu::Zone");

##******************************************************************************
## Constructor  public
##  Description  : creat a new LineGraph object
##  Parameters   : the Zone XML node
##******************************************************************************
sub new
{
    my ($classname,$rootnode,$realrootnode,$systemsDeclaration) = @_;
    my $self = $classname->SUPER::new($rootnode,$realrootnode,$systemsDeclaration);
    bless($self, $classname);

    my ($list,$node,$attributes,$attrnode);

    ##===========================================================================
    ## Dial tag
    ##===========================================================================
    $attributes = $rootnode->getAttributes() || die("Error in \"Dial\".");

    $attrnode = $attributes->getNamedItem("PixelHeight");

    $self->{PIXELHEIGHT} = 210;
    $attributes = $rootnode->getAttributes() || die("Error in \"Dial\".");
    $attrnode = $attributes->getNamedItem("PixelHeight");
    if( defined( $attrnode ) )
    {  $self->{PIXELHEIGHT} = AccentsXML2Text( $attrnode->getValue() ); }


    ##===========================================================================
    ## X tag
    ##===========================================================================
    $list = $rootnode->getElementsByTagName("X") || die("Dial: No \"X\" tag.");
    $node = $list->item(0) || die("Dial: No \"X\" tag.");
    $attributes = $node->getAttributes() || die("Error in \"X\" tag.");
    ##---------------------------------------------------------------------------
    $attrnode = $attributes->getNamedItem("Duration");

    if( defined( $attrnode ) )
    {
        $self->{START} = Duration( $attrnode->getValue() );
        $self->{STOP}  = 0;
    }
    else
    {
        $attrnode = $attributes->getNamedItem("Start") || die("No 'Start' field in \"X\" tag.");
        $self->{START} = Duration( $attrnode->getValue() );
        $attrnode = $attributes->getNamedItem("Stop") ||  die("No 'Stop' field in \"X\" tag.");
        $self->{STOP} = Duration( $attrnode->getValue() );
    }


    ##===========================================================================
    ## Y tag
    ##===========================================================================
    $list = $rootnode->getElementsByTagName("Y");
    $node = $list->item(0);
    if( defined($node) )
    {
        $attributes = $node->getAttributes() || die("Error in \"Y\".");

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("LowerBoundary");
        if( defined( $attrnode ) )
        {  $self->{Y_LOWERBOUNDARY} = $attrnode->getValue(); }
        # default = 0

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("UpperBoundary");
        if( defined( $attrnode ) )
        {  $self->{Y_UPPERBOUNDARY} = $attrnode->getValue(); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Graduation");
        if( defined( $attrnode ) )
        {  $self->{Y_GRADUATION} = $attrnode->getValue(); }
    }




    ##===========================================================================
    ## Colors tag
    ##===========================================================================
    $list = $rootnode->getElementsByTagName("Colors");
    $node = $list->item(0);
    my @pictureBGColor = RGBtoInt( $self->{SYSTEMSDECLARATION}->getColor('BackGround') );
    my @gridColor      = (80, 80, 80);
    my @dialBgColor    = (224, 224, 224);
    my @textColor      = RGBtoInt( $self->{SYSTEMSDECLARATION}->getColor('Text') );


    if( defined($node) )
    {
        $attributes = $node->getAttributes() || die("Error in \"Colors\".");

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("PictureBG");
        if( defined( $attrnode ) )
        { @pictureBGColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Grid");
        if( defined( $attrnode ) )
        { @gridColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("GraphBG");
        if( defined( $attrnode ) )
        { @dialBgColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Text");
        if( defined( $attrnode ) )
        { @textColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

    }

    ##===========================================================================
    ## ArrowsColors tag
    ##===========================================================================
    $list = $rootnode->getElementsByTagName("ArrowsColors");
    $node = $list->item(0);
    my @minArrowColor     = (0, 192, 0);
    my @currentArrowColor = (0, 0, 0);
    my @maxArrowColor     = (255, 0, 0);


    if( defined($node) )
    {
        $attributes = $node->getAttributes() || die("Error in \"ArrowsColors\".");

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Max");
        if( defined( $attrnode ) )
        { @maxArrowColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Current");
        if( defined( $attrnode ) )
        { @currentArrowColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

        ##------------------------------------------------------------------------
        $attrnode = $attributes->getNamedItem("Min");
        if( defined( $attrnode ) )
        { @minArrowColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }
    }

    $self->{PICTUREBGCOLOR} = \@pictureBGColor;
    $self->{GRIDCOLOR}      = \@gridColor;
    $self->{DIALBGCOLOR}    = \@dialBgColor;
    $self->{TEXTCOLOR}      = \@textColor;
    $self->{MINARROWCOLOR}  = \@minArrowColor;
    $self->{MAXARROWCOLOR}  = \@maxArrowColor;
    $self->{CURRENTARROWCOLOR} = \@currentArrowColor;

    return $self;
}


##******************************************************************************
## Method treat  public >Zone
##  Description  : treat a request
##  Parameters   : a Request
##  Return value : a Response
##******************************************************************************
sub treat
{
    my ($self,$request) = @_;

    if( $request->firstNodeExists() )
    {
        return $self->generatePicture( $self->{COLSIZE}*$self->{WIDTH},
                                       $self->{PIXELHEIGHT} );
    }
    else
    {
        my $response = new LogTrend::Visu::Response();
        $response->contentType("text/html");
        $response->print( "      <IMG SRC=\"".$self->ident()."/img/\">\n" );
        return $response;
    }
}


##******************************************************************************
## Method generatePicture
##******************************************************************************
sub generatePicture
{
    my ($self,$sizeX,$sizeY) = @_;

    # Calculate Dial Params
    $self->initDialParams($sizeX, $sizeY, 3/4*$sizeY, 80);

    my $image = new GD::Image($self->{SIZE_X}, $self->{SIZE_Y});
    if( !defined($image) )
    {
        my $response = new LogTrend::Visu::Response();
        $response->contentType("text/html");
        $response->print( "<HTML><BODY><CENTER>\n" );
        $response->print( "Error in <CODE>new GD::Image</CODE>&nbsp;: $!<BR>\n" );
        $response->print( "Furthermore, the image size is too small<BR>\n" );
        $response->print( "</CENTER></BODY></HTML>\n" );
        return $response;
    }

    if($self->{SIZE_Y} < 120) {
        my $response = new LogTrend::Visu::Response();
        $image->interlaced('true');
        $image->colorAllocate(255,255,255);
        my $textColor = $image->colorAllocate(0,0,0);
        $image->string( gdTinyFont, 0, 0, "TOO SMALL", $textColor );
        $response->contentType("image/png");
        $response->print( $image->png() );
        return $response;
    }

    $image->interlaced('true');


##===========================================================================
## Allocate colors
##===========================================================================

    my $colors;

    $colors->{pictureBG} = $image->colorAllocate( @{$self->{PICTUREBGCOLOR}} );
    $colors->{grid}      = $image->colorAllocate( @{$self->{GRIDCOLOR}} );
    $colors->{dial_bg}   = $image->colorAllocate( @{$self->{DIALBGCOLOR}} );
    $colors->{text}      = $image->colorAllocate( @{$self->{TEXTCOLOR}} );
    $colors->{min}       = $image->colorAllocate( @{$self->{MINARROWCOLOR}});
    $colors->{max}       = $image->colorAllocate( @{$self->{MAXARROWCOLOR}});
    $colors->{current}   = $image->colorAllocate( @{$self->{CURRENTARROWCOLOR}});

##===========================================================================
## Get Data from DataBase
##===========================================================================
    my $expression = $self->{EXPRESSIONS}->[0];
    my $time = time();
    my $query = $expression->evalInTimeInterval (100000, $self->{START}, $self->{STOP});

    # If there is no data, just print NO DATA
    if(! $query or $#$query == -1) {
        $image->string(gdLargeFont, $sizeX/2-10, $sizeY/2-20, "NO DATA" ,$colors->{text});
        my $response = new LogTrend::Visu::Response();
        $response->contentType("image/png");
        $response->print( $image->png() );
        return $response;
    }

##===========================================================================
## Analyze Data (look for min and Max)
##===========================================================================

    my $higher = $query->[0][0];
    my $lower = $higher;
    my $current_value = $query->[$#$query][0];
    my $data;

    foreach $data (@$query)
    {
        if ($data->[0] < $lower) {
            $lower = $data->[0];
        }
        elsif ($data->[0] > $higher) {
            $higher = $data->[0];
        }
    }

##===========================================================================
## Define Upper and Lower Bounds
##===========================================================================

    my $valMin = $lower;
    $valMin = $self->{Y_LOWERBOUNDARY} if( defined( $self->{Y_LOWERBOUNDARY} ) and $self->{Y_LOWERBOUNDARY} < $lower);

    my $valMax = $higher;
    $valMax = $self->{Y_UPPERBOUNDARY} if( defined( $self->{Y_UPPERBOUNDARY} ) and $self->{Y_UPPERBOUNDARY} > $higher);

   my $valPeriod;
   if( defined( $self->{Y_GRADUATION} ) )
   {
      $valPeriod = $self->{Y_GRADUATION};
   }
   else
   {
      my $valDuration = $valMax-$valMin;
      my $log = log($valDuration*5) / log(10);
      if( $log < 0 ) { $log--; }
      $valPeriod   = ( 10 ** int( $log ) )/10;

      if($self->{SIZE_Y} < 150 or $self->{DIAL_ANGLE} < 80) {
          $valPeriod *= 2;
      }
   }

   if( !defined( $self->{Y_UPPERBOUNDARY} ) || $self->{Y_UPPERBOUNDARY} < $higher)
   {
       my $start = (int( ($valMax+$valPeriod) /$valPeriod))*$valPeriod;
       $valMax = $start;
   }
   if( !defined( $self->{Y_LOWERBOUNDARY} ) || $self->{Y_LOWERBOUNDARY} > $lower)
   {
      my $start = (int( ($valMin-$valPeriod) /$valPeriod))*$valPeriod;
      $valMin = $start;
   }

    $self->{Y_UPPERBOUNDARY} = $valMax;
    $self->{Y_LOWERBOUNDARY} = $valMin;
    $self->{Y_GRADUATION} = $valPeriod;

##===========================================================================
## Some widgets
##===========================================================================
    # Paint background
    $image->filledRectangle( 0, 0, $sizeX, $sizeY, $colors->{pictureBG} );

    # Create a brush for bold lines
    my $bold_brush = new GD::Image(2,2);
    my $brush_color = $bold_brush->colorAllocate(@{$self->{GRIDCOLOR}});
    $bold_brush->fill(0, 0, $brush_color);
    $image->setBrush($bold_brush);

    $image->string( gdMediumBoldFont, 7, $self->{SIZE_Y} - gdMediumBoldFont->height - 5, $self->{TITLE}, $colors->{text} );

    # Draw dial base and arc
    $image->line($self->{BASE_X_LEFT},
                 $self->{BASE_Y},
                 $self->{BASE_X_RIGHT},
                 $self->{BASE_Y},
                 gdBrushed);

    $image->arc($self->{BASE_X_CENTER}, $self->{SIZE_Y}-2,
                2*$self->{DIAL_RAY}, 2*$self->{DIAL_RAY},
                180 + rad2deg($self->{LOWER_BOUNDARY_ANGLE}),
                180 + rad2deg($self->{LOWER_BOUNDARY_ANGLE} + $self->{DIAL_ANGLE}),
                gdBrushed);

    # draw dial's borders
    $self->drawBorders($image, gdBrushed);

    $image->fill($self->{SIZE_X}/2, $self->{SIZE_Y}/2, $colors->{dial_bg});

    # Increase the LOWER_BOUNDARY_ANGLE
    $self->{LOWER_BOUNDARY_ANGLE} += deg2rad(5);
    $self->{DIAL_ANGLE} -= deg2rad(10);

    $self->drawGrid($image, $colors->{grid});

    # draw max, min and current value lines
    $self->drawArrow($image, $lower, $colors->{min});
    $self->drawArrow($image, $higher, $colors->{max});
    $self->drawArrow($image, $current_value, $colors->{current});

    if($self->{SIZE_Y} >= 150 and rad2deg($self->{DIAL_ANGLE}) >= 70) {
        $self->drawValue($image, $lower, $colors->{min});
        $self->drawValue($image, $higher, $colors->{max});
        $self->drawValue($image, $current_value, $colors->{current});
    }

    # Hide lines source with a filled arc
    $image->arc($self->{BASE_X_CENTER}, $self->{SIZE_Y},
                2*$self->{DIAL_BASE_RAY}, 2*$self->{DIAL_BASE_RAY},
                180 + rad2deg($self->{LOWER_BOUNDARY_ANGLE}) - 5,
                180 + rad2deg($self->{LOWER_BOUNDARY_ANGLE} + $self->{DIAL_ANGLE}) + 5,
                $colors->{grid});

    $image->fill($self->{BASE_X_CENTER}, $self->{BASE_Y}-5, $colors->{grid});


  ##=========================================================================
  ## Responding
  ##=========================================================================
      my $response = new LogTrend::Visu::Response();
      $response->contentType("image/png");
      $response->print( $image->png() );
      return $response;
}


##******************************************************************************
## Method drawGrid  private
##  Description  : draw the grid
##  Parameters   : the image, the color
##  Return value :
##******************************************************************************

sub drawGrid {
    my ($self, $image, $color) = @_;
    my $nb_grad = ($self->{Y_UPPERBOUNDARY} - $self->{Y_LOWERBOUNDARY}) / $self->{Y_GRADUATION};

    my $j=0;
    for(my $i = $self->{Y_LOWERBOUNDARY};
        $i <= $self->{Y_UPPERBOUNDARY};
        $i += $self->{Y_GRADUATION}) {

        my @coords_top = $self->getValueLineCoord($i, $self->{DIAL_RAY});
        my @coords_bottom = $self->getValueLineCoord($i, $self->{DIAL_RAY}*29/30);

        $image->line(@coords_bottom[2, 3], @coords_top[2, 3], $color);

        if($nb_grad < 8 || ($j++ % 2)==0 ) {
            my @coords = $self->getValueLineCoord($i, $self->{DIAL_RAY}*29/30);

            my $string_width = gdTinyFont->width * length("$i");
            my $string_height = gdTinyFont->height;

            $image->string(gdTinyFont,
                           $coords[2] - $string_width/2,
                           $coords[3] + $string_height/2,
                           $i, $color);
        }
    }
}

##******************************************************************************
## Method drawValue  private
##  Description  : print the value
##  Parameters   : the image, the value, the color
##  Return value :
##******************************************************************************

sub drawValue {
    my ($self, $image, $value, $color) = @_;

    my $valueAngle = $value * $self->{DIAL_ANGLE} / ($self->{Y_UPPERBOUNDARY} - $self->{Y_LOWERBOUNDARY});
    $valueAngle += $self->{LOWER_BOUNDARY_ANGLE};

    my $string_width = gdSmallFont->width * length("$value");
    my $string_height = gdSmallFont->height;

    my @coords = $self->getValueLineCoord($value, $self->{DIAL_RAY});

    $image->string(gdSmallFont,
                   $coords[2] - $string_width/2,
                   $coords[3] - $string_height - 25*abs(cos($valueAngle)) - $string_width/10,
                   $value, $color);
}

##******************************************************************************
## Method drawBorders  private
##  Description  : draw a border line
##  Parameters   : the image, the value, the color
##  Return value :
##******************************************************************************

sub drawBorders {
    my ($self, $image, $color) = @_;

    my @coords = $self->getValueLineCoord($self->{Y_LOWERBOUNDARY}, $self->{DIAL_RAY});
    @coords[0, 1] = ($self->{BASE_X_LEFT}, $self->{BASE_Y});
    $image->line(@coords, $color);

    @coords = $self->getValueLineCoord($self->{Y_UPPERBOUNDARY}, $self->{DIAL_RAY});
    @coords[0, 1] = ($self->{BASE_X_RIGHT}, $self->{BASE_Y});

    $image->line(@coords, $color);

}


##******************************************************************************
## Method drawLine  private
##  Description  : draw a line corresponding to a value
##  Parameters   : the image, the value, the color
##  Return value :
##******************************************************************************

sub drawLine {
    my ($self, $image, $value, $color) = @_;
    my @coords = $self->getValueLineCoord($value, $self->{DIAL_RAY});

    $image->line(@coords, $color);
}


##******************************************************************************
## Method drawArrow  private
##  Description  : draw an arrow corresponding to a value
##  Parameters   : the image, the value, the color
##  Return value :
##******************************************************************************

sub drawArrow {
    my ($self, $image, $value, $color) = @_;
    my @coords = $self->getValueLineCoord($value, $self->{DIAL_RAY}*6/7);

    my $arrow_value = ($self->{Y_UPPERBOUNDARY} - $self->{Y_LOWERBOUNDARY})*1/40;

    my @arrow_left_point_coords = $self->getValueLineCoord($value - $arrow_value,
                                                           $self->{DIAL_RAY}*39/48);
    my @arrow_right_point_coords = $self->getValueLineCoord($value + $arrow_value,
                                                           $self->{DIAL_RAY}*39/48);

    $image->line(@arrow_left_point_coords[2, 3], @coords[2, 3], $color);
    $image->line(@arrow_right_point_coords[2, 3], @coords[2, 3], $color);

    $image->line(@coords, $color);
}


##******************************************************************************
## Method getValueLineCoord  private
##  Description  : return start and end lines' point coordinate
##  Parameters   : the value and the ray length.
##  Return value :
##******************************************************************************
sub getValueLineCoord {
    my ($self, $value, $ray) = @_;

    my $lineAngle = ($value-$self->{Y_LOWERBOUNDARY}) * $self->{DIAL_ANGLE} / ($self->{Y_UPPERBOUNDARY} - $self->{Y_LOWERBOUNDARY});
    $lineAngle += $self->{LOWER_BOUNDARY_ANGLE};

    my $value_top_X = $ray * cos($lineAngle);
    $value_top_X = $self->{BASE_X_CENTER} - $value_top_X;

    my $value_top_Y =  $ray * sin($lineAngle);
    $value_top_Y = $self->{SIZE_Y} - $value_top_Y;


    my $value_bottom_X = $self->{DIAL_BASE_RAY} * cos($lineAngle);
    $value_bottom_X = $self->{BASE_X_CENTER} - $value_bottom_X;

    my $value_bottom_Y =  $self->{DIAL_BASE_RAY} * sin($lineAngle);
    $value_bottom_Y = $self->{SIZE_Y} - $value_bottom_Y + 1;


    return ($value_bottom_X, $value_bottom_Y, $value_top_X, $value_top_Y);
}


##******************************************************************************
## Method initDialParams  private
##  Description  : Calculate usefull values on Dial like :
##  Parameters   : sizeX and sizeY (size of the component),
##                 the height of the Dial,
##                 the angle of the dial (in degree)
##  Return value : none (initialize attributes in $self)
##******************************************************************************
sub initDialParams {
    my ($self, $sizeX, $sizeY, $dialHeight, $dialAngle) = @_;

    $dialAngle = deg2rad($dialAngle);

    die "Dial Ray must be inferior to the component height" if $dialHeight >= $sizeY;

    # The center of arc a the top of the dial is suppose to be in X = sizeX/2,  Y = sizeY
    my $dialRay = $dialHeight +  ($sizeY - $dialHeight)/2;
    my $dialBaseRay = $dialRay/3;

    # Decrease dialHeight if the dial is larger than sizeX
    my $dialWidth = $dialRay * cos((pi - $dialAngle)/2);
    my $y_offset = 0;

    if($dialWidth*2 > $sizeX) {
          $dialWidth = 9*$sizeX/20;
          $dialRay = $dialWidth/cos((pi - $dialAngle)/2);

        my $old_size = $sizeY;

        $sizeY = 8*$dialRay/7;
        $dialHeight =  2*$dialRay - $sizeY;
          $dialBaseRay = $dialRay/3;

          ##
          ## If the size is < 150,
          ## reduce the angle.
          ##
          if($sizeY < 150 and $old_size >= 150) {
              $sizeY = 150;
              $dialRay = 7*$sizeY/8;
              $dialBaseRay = $dialRay/3;
              $dialHeight =  2*$dialRay - $sizeY;

              $dialAngle = pi-2*acos($dialWidth/$dialRay);
          }
    }

    # coordinates of dial base
    my ($base_X_left, $base_X_center, $base_X_right, $base_Y);

    $base_Y = $sizeY - ($sizeY - $dialHeight)/2;

    # Dial base is centered in component
    $base_X_center = $sizeX/2;

    # Caculate the size of the base
    my $base_size = 2*($sizeY-$base_Y)*tan($dialAngle/2);
    ($base_X_left, $base_X_right) = ($base_X_center-$base_size/2, $base_X_center+$base_size/2);

    # Caculate angle for Y_LOWERBOUNDARY
    my $lowerBoundaryAngle = pi/2 - $dialAngle/2;

    # store results in $self
    $self->{DIAL_ANGLE}           = $dialAngle;
    $self->{DIAL_HEIGHT}          = $dialHeight;
    $self->{DIAL_RAY}             = $dialRay;
    $self->{DIAL_BASE_RAY}        = $dialBaseRay;
    $self->{LOWER_BOUNDARY_ANGLE} = $lowerBoundaryAngle;
    $self->{BASE_X_LEFT}          = $base_X_left;
    $self->{BASE_X_CENTER}        = $base_X_center;
    $self->{BASE_X_RIGHT}         = $base_X_right;
    $self->{BASE_Y}               = $base_Y;
    $self->{SIZE_X}               = $sizeX;
    $self->{SIZE_Y}               = $sizeY;
}

1;


