#!/bin/bash
#
#  build-initrd.sh
#
#  Written by Jari Ruusu, March 9 2004
#
#  Copyright 2001,2002,2003,2004 by Jari Ruusu.
#  Redistribution of this file is permitted under the GNU Public License.
#
#  Changes by Hauke Johannknecht <ash@ash.de> 11/2001
#     - added Pivot mode
#     - added conffile-loading
#     - added initrdonly-mode
#     - added device 0 nodes
#
#
#  Initrd can use two different methods to switch to encrypted root device:
#  change_root (USEPIVOT=0) and pivot_root (USEPIVOT=1). change_root method
#  is present in at least 2.2 and 2.4 kernels, and it works ok. pivot_root
#  is present in 2.4 and later kernels, and offers much nicer wrong password
#  case handling because initrd code can properly shutdown the kernel.
#  Proper shutdown is important for software RAID devices and such.
#  change_root and pivot_root require slightly different kernel and
#  bootloader configuration.
#
#  kernel .config :  CONFIG_BLK_DEV_RAM=y
#  (USEPIVOT=0)      CONFIG_BLK_DEV_RAM_SIZE=4096
#                    CONFIG_BLK_DEV_INITRD=y
#                    CONFIG_MINIX_FS=y
#                    CONFIG_PROC_FS=y
#                    CONFIG_CRAMFS=n  (or CONFIG_CRAMFS=m)
#
#  kernel .config :  CONFIG_BLK_DEV_RAM=y
#  (USEPIVOT=1)      CONFIG_BLK_DEV_RAM_SIZE=4096
#                    CONFIG_BLK_DEV_INITRD=y
#                    CONFIG_MINIX_FS=y
#
#  /etc/lilo.conf :  initrd=/boot/initrd.gz
#  (USEPIVOT=0)      root=/dev/ram1
#                                 ^
#  /etc/lilo.conf :  append="init=/linuxrc rootfstype=minix"
#  (USEPIVOT=1)      initrd=/boot/initrd.gz
#                    root=/dev/ram0
#                                 ^
#  /boot/grub/menu.lst :  root (hd0,0)
#  (USEPIVOT=0)           kernel /vmlinuz ro root=101
#                         initrd /initrd.gz         ^
#
#  /boot/grub/menu.lst :  root (hd0,0)
#  (USEPIVOT=1)           kernel /vmlinuz root=100 init=/linuxrc rootfstype=minix
#                         initrd /initrd.gz      ^
#
#  usage :  ./build-initrd.sh [configfile]
#           lilo
#           mkdir /boot/modules-`uname -r`
#           cp -p /lib/modules/`uname -r`/block/loop.*o /boot/modules-`uname -r`/
#

### All default-values can be altered via the configfile

# Unencrypted /boot partition. If devfs is enabled (USEDEVFS=1), this must
# be specified as genuine devfs name.
BOOTDEV=/dev/hda1

# /boot partition file system type
BOOTTYPE=ext2

# Encrypted root partition. If devfs is enabled (USEDEVFS=1), this must
# be specified as genuine devfs name.
CRYPTROOT=/dev/hda2

# root partition file system type
ROOTTYPE=ext2

# Encryption type (AES128 / AES192 / AES256) of root partition
CIPHERTYPE=AES128

# Optional password seed for root partition
#PSEED="-S XXXXXX"

# Optional password iteration count for root partition
#ITERCOUNTK="-C 100"

# This code is passed to cipher transfer function.
LOINIT="-I 0"

# 1 = use gpg key file to mount root partition, 0 = use normal key.
# If this is enabled (USEGPGKEY=1), file named rootkey.gpg or whatever
# GPGKEYFILE is set to must be manually copied to /boot (or to
# EXTERNALGPGDEV device if EXTERNALGPGFILES=1). If rootkey.gpg is not
# encrypted with symmetric cipher, pubring.gpg and secring.gpg must be
# manually copied to /boot (or to EXTERNALGPGDEV device if
# EXTERNALGPGFILES=1). Try to keep pubring.gpg as small as possible, because
# trustdb.gpg must fit into ramdisk which has very limited space available.
USEGPGKEY=1

# gpg key filename. Only used if USEGPGKEY=1
GPGKEYFILE=rootkey.gpg

# 1 = mount removable device EXTERNALGPGDEV that contains gpg key files
# 0 = don't mount
EXTERNALGPGFILES=0

# Device name that contains gpg key files. If devfs is
# enabled (USEDEVFS=1), this must be specified as genuine devfs name.
# Only used if EXTERNALGPGFILES=1
EXTERNALGPGDEV=/dev/fd0

# Removable device EXTERNALGPGDEV file system type
# Only used if EXTERNALGPGFILES=1
EXTERNALGPGTYPE=ext2

# 1 = use loop module, 0 = loop driver linked to kernel
USEMODULE=1

# 1 = use pivot_root, 0 = use old change_root
# See above header for root= and append= lilo.conf definitions.
# pivot_root is not available on 2.2 and older kernels.
# Always enable pivot_root for 2.4 and later kernels.
USEPIVOT=1

# 1 = stop after creating and copying initrd, 0 = also copy tools/libs
INITRDONLY=0

# Source root directory where files are copied from
SOURCEROOT=

# Destination root directory where files are written to
DESTINATIONROOT=

# dest-dir below dest-root
DESTINATIONPREFIX=/boot

# Name of created init ram-disk
INITRDGZNAME=initrd.gz

# Encrypted root loop device index (0 ... 7), 5 == /dev/loop5
# Device index must be one character even if max_loop is greater than 8
# _must_ match /etc/fstab entry:   /dev/loop5  /  ext2  defaults,xxxx  0  1
ROOTLOOPINDEX=5

# Temporary loop device index used in this script, 7 == /dev/loop7
TEMPLOOPINDEX=7

# Additional loop module parameters.
# Example: LOOPMODPARAMS="max_loop=8 lo_prealloc=125,5,200"
LOOPMODPARAMS=""

# 1 = use devfs, 0 = use classic disk-based device names. If this is
# enabled (USEDEVFS=1) then setting USEPIVOT=1 is also required and kernel
# must be configured with CONFIG_DEVFS_FS=y CONFIG_DEVFS_MOUNT=y
USEDEVFS=0

# 1 = load national keyboard layout, 0 = don't load
# You _must_ manually copy correct keyboard layout to /boot/default.kmap
# which must be in uncompressed form. (can not be .gz file)
LOADNATIONALKEYB=0

# Initial delay in seconds before /linuxrc attempts to mount /boot
# partition. Slow devices (USB dongles) may need some delay.
INITIALDELAY=0

# 1 = prompt for BOOT-TOOLS media and ENTER press before mounting /boot
# 0 = normal case, don't prompt
TOOLSPROMPT=0

# 1 = use "rootsetup" program that executes losetup to initialize loop
# 0 = use normal "losetup" program directly to initialize loop
# If enabled, rootsetup program (+libs) _must_ be manually copied to /boot.
USEROOTSETUP=0

# 1 = use dietlibc to build linuxrc. This permits passing parameters to init.
# 0 = use glibc to build linuxrc. This prevents passing parameters to init
# and includes hacks that may be incompatible with some versions of glibc.
# The dietlibc can be found at http://www.fefe.de/dietlibc/
USEDIETLIBC=1

# 1 = load extra module, 0 = don't load
# If this is enabled, module must be manually copied to
# /boot/modules-KERNELRELEASE/ directory under name like foomatic.o
EXTRAMODULELOAD1=0
EXTRAMODULENAME1="foomatic"
EXTRAMODULEPARAMS1="frobnicator=123 fubar=abc"
# 1 = load extra module, 0 = don't load
EXTRAMODULELOAD2=0
EXTRAMODULENAME2=""
EXTRAMODULEPARAMS2=""
# 1 = load extra module, 0 = don't load
EXTRAMODULELOAD3=0
EXTRAMODULENAME3=""
EXTRAMODULEPARAMS3=""
# 1 = load extra module, 0 = don't load
EXTRAMODULELOAD4=0
EXTRAMODULENAME4=""
EXTRAMODULEPARAMS4=""
# 1 = load extra module, 0 = don't load
EXTRAMODULELOAD5=0
EXTRAMODULENAME5=""
EXTRAMODULEPARAMS5=""

### End of options


if [ $# = 1 ] ; then
    if [ ! -f $1 ] ; then
        echo "ERROR: Can't find configfile '$1'"
        echo "Usage: $0 [configfile]"
        exit 1
    fi
    echo "Loading config from '$1'"
    . $1
fi

DEVFSSLASH1=
DEVFSRAMDSK=/dev/ram
if [ ${USEDEVFS} == 1 ] ; then
    DEVFSSLASH1=/
    DEVFSRAMDSK=/dev/rd/
fi
DEVFSSLASH2=
if [ -c /dev/.devfsd ] ; then
    DEVFSSLASH2=/
fi
LOSETUPPROG=losetup
if [ ${USEROOTSETUP} == 1 ] ; then
    LOSETUPPROG=rootsetup
fi
if [ ${USEGPGKEY} == 0 ] ; then
    EXTERNALGPGFILES=0
fi
GPGMOUNTDEV=
GPGMNTPATH=lib
if [ ${EXTERNALGPGFILES} == 1 ] ; then
    GPGMOUNTDEV=${EXTERNALGPGDEV}
    GPGMNTPATH=mnt
fi

if [ ${USEDIETLIBC} == 1 ] ; then
    x=`which diet`
    if [ x${x} == x ] ; then
        echo "*****************************************************************"
        echo "***  This script was configured to build linuxrc using        ***"
        echo "***  dietlibc, but it appears that dietlibc is unavailable.   ***"
        echo "***  Script aborted.                                          ***"
        echo "*****************************************************************"
        exit 1
    fi
fi

set -e
umask 077
cat - <<EOF >tmp-c-$$.c

/* Note: this program does not initialize C library, so all level 3    */
/* library calls are forbidden. Only level 2 system calls are allowed. */

#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <fcntl.h>
#include <time.h>

#if ${USEPIVOT}
# if ${USEDIETLIBC}
   extern int pivot_root(const char *, const char *);
# else
#  include <sys/syscall.h>
#  include <errno.h>
#  include <linux/unistd.h>
#  if !defined(__NR_pivot_root) && !defined(SYS_pivot_root) && defined(__i386__)
#   define __NR_pivot_root 217
    static _syscall2(int,pivot_root,const char *,new_root,const char *,put_old)
#  else
#   define pivot_root(new_root,put_old) syscall(SYS_pivot_root,new_root,put_old)
#  endif
# endif
#endif

#if (defined(__GLIBC__) && __GLIBC__ >= 2) || defined(__dietlibc__)
# include <sys/reboot.h>
# define HaltKernel() reboot(0xCDEF0123)  /* LINUX_REBOOT_CMD_HALT */
#else
extern int reboot(int, int, int);
# define HaltKernel() reboot(0xfee1dead, 672274793, 0xCDEF0123)
#endif

#if ${USEDIETLIBC}
static char ** argv_init;
static char ** envp_init;
extern char ** environ;
#else
static char * argv_init[] = { "init", 0, };
static char * envp_init[] = { "HOME=/", "TERM=linux", 0, };
#endif

void strCat(char *d, char *s)
{
    while(*d) d++;
    while(*s) *d++ = *s++;
    *d = 0;
}

void wrStr(char *s)
{
    char *p = s;
    int x = 0;

    while(*p) p++, x++;
    write(1, s, x);
}

int exeWait(char *p)
{
    int x, y;
    char *a[50], *e[1];

    if(!(x = fork())) {
        while(*p && (x < ((sizeof(a) / sizeof(char *)) - 1))) {
            a[x++] = p;
            while(*p && (*p != ' ') && (*p != '\t')) p++;
            while((*p == ' ') || (*p == '\t')) *p++ = 0;
        }
        e[0] = a[x] = 0;
        if(x) execve(a[0], &a[0], &e[0]);
        _exit(1);
    }
    if(x == -1) {
        wrStr("linuxrc: fork failed\n");
        return(1);
    }
    waitpid(x, &y, 0);
    if(!WIFEXITED(y) || (WEXITSTATUS(y) != 0)) {
        wrStr("Command \""); wrStr(p); wrStr("\" returned error\n");
        return(1);
    }
    return(0);
}

void doHalt()
{
    int x, y;
    struct timespec req;

    sync();
    if(!(x = fork())) HaltKernel();
    waitpid(x, &y, 0);
    for(;;) {
        req.tv_sec = 5;
        req.tv_nsec = 0;
        nanosleep(&req, 0);
    }
}

void runInit()
{
#if ${USEDIETLIBC}
    if(argv_init[0]) argv_init[0] = "init";
    envp_init = environ;
#endif
    execve("/sbin/init", argv_init, envp_init);
    execve("/etc/init", argv_init, envp_init);
    execve("/bin/init", argv_init, envp_init);
    wrStr("Bummer! Launching init failed\n");
    doHalt();
}

#if ${USEDIETLIBC}
int main(int argc, char **argv)
#else
void _start()
#endif
{
    int x = 0;
    struct utsname un;
    char buf[1000], *modext;

#if ${USEDIETLIBC}
    argv_init = argv;
#endif

#if ${USEPIVOT}
    if(getpid() != 1) {
        /* pivot_root was configured, but kernel has */
        /* wandered off to change_root code path!    */
        wrStr("ERROR: initrd config says USEPIVOT=1, but bootloader acts like USEPIVOT=0\n");
        _exit(0);
    }
#else
    if(getpid() == 1) {
        /* change_root was configured, but kernel    */
        /* has wandered off to pivot_root code path! */
        wrStr("ERROR: initrd config says USEPIVOT=0, but bootloader acts like USEPIVOT=1\n");
        runInit();
    }
#endif

#if ${INITIALDELAY}
    {
        struct timespec req;
        req.tv_sec = ${INITIALDELAY};
        req.tv_nsec = 0;
        nanosleep(&req, 0);
    }
#endif

#if ${TOOLSPROMPT}
    wrStr("Please insert BOOT-TOOLS media, and press ENTER  ");
    read(0, buf, sizeof(buf));
#endif

    /* this intentionally mounts /boot partition as /lib */
    if(mount("${BOOTDEV}", "/lib", "${BOOTTYPE}", MS_MGC_VAL | MS_RDONLY, 0)) {
        wrStr("Mounting ${BOOTDEV} as /lib failed\n");
        goto fail4;
    }

#if ${LOADNATIONALKEYB}
    buf[0] = 0;
    strCat(buf, "/lib/loadkeys /lib/default.kmap");
    exeWait(buf);
#endif

    uname(&un);
    if((un.release[0] > '2') || (un.release[1] != '.') || (un.release[2] >= '6') || (un.release[3] != '.')) {
        modext = ".ko";
    } else {
        modext = ".o";
    }

#if ${USEMODULE}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/loop");
    strCat(buf, modext);
    strCat(buf, " ${LOOPMODPARAMS}");
    if(exeWait(buf)) goto fail5;
#endif

#if ${EXTRAMODULELOAD1}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/${EXTRAMODULENAME1}");
    strCat(buf, modext);
    strCat(buf, " ${EXTRAMODULEPARAMS1}");
    if(exeWait(buf)) goto fail5;
#endif
#if ${EXTRAMODULELOAD2}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/${EXTRAMODULENAME2}");
    strCat(buf, modext);
    strCat(buf, " ${EXTRAMODULEPARAMS2}");
    if(exeWait(buf)) goto fail5;
#endif
#if ${EXTRAMODULELOAD3}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/${EXTRAMODULENAME3}");
    strCat(buf, modext);
    strCat(buf, " ${EXTRAMODULEPARAMS3}");
    if(exeWait(buf)) goto fail5;
#endif
#if ${EXTRAMODULELOAD4}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/${EXTRAMODULENAME4}");
    strCat(buf, modext);
    strCat(buf, " ${EXTRAMODULEPARAMS4}");
    if(exeWait(buf)) goto fail5;
#endif
#if ${EXTRAMODULELOAD5}
    buf[0] = 0;
    strCat(buf, "/lib/insmod /lib/modules-");
    strCat(buf, &un.release[0]);
    strCat(buf, "/${EXTRAMODULENAME5}");
    strCat(buf, modext);
    strCat(buf, " ${EXTRAMODULEPARAMS5}");
    if(exeWait(buf)) goto fail5;
#endif

#if ${EXTERNALGPGFILES}
    if(mount("${EXTERNALGPGDEV}", "/mnt", "${EXTERNALGPGTYPE}", MS_MGC_VAL | MS_RDONLY, 0)) {
        wrStr("Mounting ${EXTERNALGPGDEV} containing gpg key files failed.\n");
        goto fail5;
    }
#endif

#if ${USEGPGKEY} && ${USEPIVOT}
    /* gpg needs a writable ramdisk, so remount rw */
    mount("${DEVFSRAMDSK}0", "/", "minix", MS_MGC_VAL | MS_REMOUNT, 0);
#endif

    tryAgain:
#if !(${USEROOTSETUP})
    wrStr("\nEncrypted file system, please supply correct password to continue\n\n");
#endif

    buf[0] = 0;
    strCat(buf, "/lib/${LOSETUPPROG} -e ${CIPHERTYPE} ${PSEED} ${ITERCOUNTK} ${LOINIT}");
#if ${USEGPGKEY}
    strCat(buf, " -K /${GPGMNTPATH}/${GPGKEYFILE} -G /");
    unlink("/trustdb.gpg");
#endif
    strCat(buf, " /dev/loop${DEVFSSLASH1}${ROOTLOOPINDEX} ${CRYPTROOT}");

    if(exeWait(buf)) {
        if(++x >= 5) goto fail3;
        goto tryAgain;
    }
#if ${USEGPGKEY}
    unlink("/trustdb.gpg");
#endif

#if !(${USEROOTSETUP})
    wrStr("\n");
#endif

#if ${USEPIVOT}
    if(mount("/dev/loop${DEVFSSLASH1}${ROOTLOOPINDEX}", "/new-root", "${ROOTTYPE}", MS_MGC_VAL | MS_RDONLY, 0)) {
        wrStr("Looks like you didn't say the magic word. Mounting /dev/loop${DEVFSSLASH1}${ROOTLOOPINDEX} failed\n");
        buf[0] = 0;
        strCat(buf, "/lib/${LOSETUPPROG} -d /dev/loop${DEVFSSLASH1}${ROOTLOOPINDEX}");
        if(exeWait(buf)) goto fail3;
        if(++x >= 5) goto fail3;
        goto tryAgain;
    }
#if ${EXTERNALGPGFILES}
    umount("/mnt");
#endif
    umount("/lib");
    if(chdir("/new-root")) {
        wrStr("chdir() to /new-root failed\n");
        goto fail1;
    }
    if(pivot_root(".", "initrd")) {
        wrStr("pivot_root() to new root failed. Older kernels don't have pivot_root()\n");
        fail1:
        chdir("/");
        umount("/new-root");
        goto fail4;
    }
    chdir("/");

#if ${USEDEVFS}
    if(mount("none", "dev", "devfs", MS_MGC_VAL, 0)) {
        wrStr("Mounting /dev failed\n");
        goto fail1;
    }
#endif

    close(0);
    close(1);
    close(2);
    open("dev/console", O_RDWR, 0);
    dup(0);
    dup(0);

    if(chroot(".")) {
        wrStr("chroot() to new root failed\n");
        goto fail1;
    }
    wrStr("Pivoting to encrypted root completed successfully\n");
    runInit();
#else
    if(mount("none", "/proc", "proc", MS_MGC_VAL, 0)) {
        wrStr("Mounting /proc failed\n");
        goto fail3;
    }
    if((x = open("/proc/sys/kernel/real-root-dev", O_WRONLY, 0)) == -1) {
        wrStr("Unable to open real-root-dev\n");
        goto fail2;
    }
    write(x, "0x70${ROOTLOOPINDEX}\n", 6);
    close(x);
    fail2:
    umount("/proc");
#endif

    fail3:
#if ${EXTERNALGPGFILES}
    umount("/mnt");
#endif
    fail5:
    umount("/lib");
    fail4:
#if ${USEPIVOT}
    doHalt();
#endif
    _exit(0);
}
EOF

if [ ${USEDIETLIBC} == 1 ] ; then
    diet gcc -Wall -O2 -s -static -pipe tmp-c-$$.c -o tmp-c-$$
else
    gcc -Wall -O2 -s -static -nostartfiles -pipe tmp-c-$$.c -o tmp-c-$$
fi
rm -f tmp-c-$$.[co]

x=`cat tmp-c-$$ | wc -c`
y=`expr ${x} + 1023`
x=`expr ${y} / 1024`
y=`expr ${x} + 12`
if [ ${USEGPGKEY} == 1 ] ; then
    y=`expr ${y} + 40`
fi

dd if=/dev/zero of=tmp-i-$$ bs=1024 count=${y}
/sbin/losetup /dev/loop${DEVFSSLASH2}${TEMPLOOPINDEX} tmp-i-$$
/sbin/mkfs -t minix -i 32 /dev/loop${DEVFSSLASH2}${TEMPLOOPINDEX}
/sbin/losetup -d /dev/loop${DEVFSSLASH2}${TEMPLOOPINDEX}
mkdir tmp-d-$$
mount -t minix tmp-i-$$ tmp-d-$$ -o loop=/dev/loop${DEVFSSLASH2}${TEMPLOOPINDEX}
cd tmp-d-$$

mkdir dev lib
mv ../tmp-c-$$ linuxrc

if [ ${EXTERNALGPGFILES} == 1 ] ; then
    mkdir mnt
fi

if [ ${USEGPGKEY} == 1 ] ; then
    ln -s lib bin
    ln -s ${GPGMNTPATH}/pubring.gpg pubring.gpg
    ln -s ${GPGMNTPATH}/secring.gpg secring.gpg
fi

if [ ${USEPIVOT} == 1 ] ; then
    mkdir new-root
else
    mkdir proc
fi

# <device name prefix> <major dev-id> <minor dev-id start> <device 0 suffix>
function maybeMakeDiskNode
{
    x=$3
    for i in "$4" 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 ; do
        for y in ${BOOTDEV} ${CRYPTROOT} ${GPGMOUNTDEV} ; do
            if [ ${y} == /dev/$1${i} ] ; then
                mknod dev/$1${i} b $2 ${x}
                mknodCount=`expr ${mknodCount} + 1`
            fi
        done
        x=`expr ${x} + 1`
    done
    return
}
if [ ${USEDEVFS} == 0 ] ; then
    mknodCount=0
    mknodRefCount=2
    if [ ${EXTERNALGPGFILES} == 1 ] ; then
        mknodRefCount=`expr ${mknodRefCount} + 1`
    fi

    maybeMakeDiskNode hda  3   0 ""
    maybeMakeDiskNode hdb  3  64 ""
    maybeMakeDiskNode hdc 22   0 ""
    maybeMakeDiskNode hdd 22  64 ""
    maybeMakeDiskNode hde 33   0 ""
    maybeMakeDiskNode hdf 33  64 ""
    maybeMakeDiskNode hdg 34   0 ""
    maybeMakeDiskNode hdh 34  64 ""
    maybeMakeDiskNode sda  8   0 ""
    maybeMakeDiskNode sdb  8  16 ""
    maybeMakeDiskNode sdc  8  32 ""
    maybeMakeDiskNode sdd  8  48 ""
    maybeMakeDiskNode sde  8  64 ""
    maybeMakeDiskNode sdf  8  80 ""
    maybeMakeDiskNode sdg  8  96 ""
    maybeMakeDiskNode sdh  8 112 ""
    maybeMakeDiskNode scd 11   0  0
    maybeMakeDiskNode md   9   0  0
    maybeMakeDiskNode fd   2   0  0

    if [ ${mknodCount} != ${mknodRefCount} ] ; then
        echo "*****************************************************************"
        echo "***  Internal build-initrd.sh error condition detected. This  ***"
        echo "***  script was supposed to create block device nodes for     ***"
        echo "***  BOOTDEV=, CRYPTROOT= and possibly EXTERNALGPGDEV= but    ***"
        echo "***  lacked knowledge of how to create at least one of them.  ***"
        echo "***  Script aborted.                                          ***"
        echo "*****************************************************************"
        cd ..
        umount tmp-d-$$
        rmdir tmp-d-$$
        rm tmp-i-$$
        exit 1
    fi

    mknod dev/console c 5 1
    mknod dev/tty c 5 0
    mknod dev/tty1 c 4 1
    mknod dev/null c 1 3
    mknod dev/zero c 1 5
    mknod dev/ram0 b 1 0
    mknod dev/ram1 b 1 1
    mknod dev/loop${ROOTLOOPINDEX} b 7 ${ROOTLOOPINDEX}
fi

cd ..
df tmp-d-$$
umount tmp-d-$$
rmdir tmp-d-$$
sync ; sync ; sync
gzip -9 tmp-i-$$
mv tmp-i-$$.gz ${DESTINATIONROOT}${DESTINATIONPREFIX}/${INITRDGZNAME}
ls -l ${DESTINATIONROOT}${DESTINATIONPREFIX}/${INITRDGZNAME}

if [ ${INITRDONLY} == 1 ] ; then
    echo Done.
    sync
    exit 0
fi

z="/sbin/losetup"
if [ ${USEMODULE}${EXTRAMODULELOAD1}${EXTRAMODULELOAD2}${EXTRAMODULELOAD3}${EXTRAMODULELOAD4}${EXTRAMODULELOAD5} != 000000 ] ; then
    z="${z} /sbin/insmod"
    if [ -r ${SOURCEROOT}/sbin/insmod.modutils ] ; then
        z="${z} /sbin/insmod.modutils"
    fi
    if [ -r ${SOURCEROOT}/sbin/insmod.old ] ; then
        z="${z} /sbin/insmod.old"
    fi
fi
if [ ${LOADNATIONALKEYB} == 1 ] ; then
    z="${z} "`which loadkeys`
fi
if [ ${USEGPGKEY} == 1 ] ; then
    z="${z} "`which gpg`
fi
for x in ${z} ; do
    echo Copying ${SOURCEROOT}${x} to ${DESTINATIONROOT}${DESTINATIONPREFIX}
    cp -p ${SOURCEROOT}${x} ${DESTINATIONROOT}${DESTINATIONPREFIX}
    y=`ldd ${SOURCEROOT}${x} | perl -ne 'if(/ => ([^ ]*) /){print "$1\n"}'`
    for a in ${y} ; do
        echo Copying ${SOURCEROOT}${a} to ${DESTINATIONROOT}${DESTINATIONPREFIX}
        cp -p ${SOURCEROOT}${a} ${DESTINATIONROOT}${DESTINATIONPREFIX}
    done
done

if [ ! -d ${DESTINATIONROOT}/initrd ] ; then
    mkdir ${DESTINATIONROOT}/initrd
fi

echo Done.
sync
exit 0
